import logging
import os
import shutil

from sandbox import sdk2
from sandbox.projects.common import build
from sandbox.projects.common import constants as consts
from sandbox.projects.common import teamcity
from sandbox.projects.common.arcadia import sdk as arcadiasdk
from sandbox.projects.common.build.YaMake2 import YaMake2
from sandbox.sdk2.helpers import subprocess
from sandbox.sdk2.vcs.git import Git

USERVER_URL = 'ssh://git@bb.yandex-team.ru/taxi/userver.git'
USERVER_DIR = 'userver'

USERVICES_DIR = 'taxi/uservices'
YA_MAKE_CODEGEN_DIR = USERVICES_DIR + '/arc-gen/ya-make-codegen'
ACTUAL_GIT_COMMIT = USERVICES_DIR + '/userver/actual-git-commit'

LXC_CONTAINER = 3140560785


class TplatformBuildUservicesTier0(YaMake2):
    class Requirements(YaMake2.Requirements):
        cores = 8
        disk_space = 250 * 1024
        ram = 32 * 1024

    class Parameters(YaMake2.Parameters):

        _container = sdk2.parameters.Container(
            'lxc container',
            default_value=LXC_CONTAINER,
            platform='linux_ubuntu_16.04_xenial',
        )
        with sdk2.parameters.Group('Build parameters'):
            build_system = build.parameters.BuildSystem(
                default_value=consts.DISTBUILD_BUILD_SYSTEM,
            )
            build_type = build.parameters.BuildSystem(
                default_value=consts.RELEASE_WITH_DEBUG_INFO_BUILD_TYPE,
            )
            kill_timeout = 3600 * 9  # 9h
            targets = sdk2.parameters.String(
                'Targets (semicolon separated)',
                default_value='; '.join(
                    [
                        USERVICES_DIR + '/libraries/',
                        USERVICES_DIR + '/services/',
                    ],
                ),
            )
            keep_on = build.parameters.KeepOn(default_value=True)
            fail_on_any_error = True

        with sdk2.parameters.Group('Teamcity parameters'):
            branch = sdk2.parameters.String(
                'Branch', required=True, default='refs/heads/develop',
            )
            commit = sdk2.parameters.String('Commit', required=True)
            test_targets = sdk2.parameters.String(
                'Targets for tests (semicolon separated)', required=True,
            )
            check_libraries = sdk2.parameters.Bool(
                'Check uservices/libraries', default=True,
            )

        with sdk2.parameters.Group('Tokens'):
            tokens = sdk2.parameters.YavSecret(
                'YAV secret identifier',
                default='sec-01fvera5z43mzf7m3yeb7ftec6',
            )
            arc_secret = build.parameters.ArcSecret(
                default='sec-01fvera5z43mzf7m3yeb7ftec6#arc.token',
            )
            ya_yt_token_yav_secret = build.parameters.YtTokenYavSecret(
                'Yav secret with YT token',
                default_value='sec-01fvera5z43mzf7m3yeb7ftec6#yt-token',
            )

    def pre_build(self, source_dir):
        with self.memoize_stage.create_arc_patch:
            logging.info(
                'Check uservices tier0 on branch: {branch} commit: {commit}'.format(
                    branch=self.Parameters.branch,
                    commit=self.Parameters.commit,
                ),
            )
            self.path_to_arcadia = source_dir

            self.checkout_userver()
            self.apply_git_patch()
            self.gen_ya_make_files()

            self.check_arc_status()

    def workdir_path(self, *args):
        return str(self.path(*args))

    def arcadia_path(self, *args):
        return str(os.path.join(self.path_to_arcadia, *args))

    def artifacts_path(self, *args):
        teamcity_artifacts_dir = self.workdir_path('teamcity_artifacts')
        if not os.path.isdir(teamcity_artifacts_dir):
            logging.info(
                'Make directory {dir}'.format(dir=teamcity_artifacts_dir),
            )
            os.mkdir(teamcity_artifacts_dir)

        return str(os.path.join(teamcity_artifacts_dir, *args))

    def checkout_userver(self):
        with sdk2.ssh.Key(
                private_part=self.Parameters.tokens.data()['ssh-private.key'],
        ):
            vcs_root = Git(url=USERVER_URL, filter_branches=False)
            vcs_root.clone(
                self.workdir_path(USERVER_DIR),
                branch=self.Parameters.branch,
                commit=self.Parameters.commit,
            )
        logging.info(
            'Checkout userver in directory: {path}'.format(
                path=self.workdir_path(USERVER_DIR),
            ),
        )

    def apply_git_patch(self):
        actual_git_commit = self.get_actual_git_commit()
        if actual_git_commit == self.Parameters.commit:
            logging.info(
                'Actual git commit is equal to target commit, applying patch was skipped',
            )
            return

        diff = self.get_diff(actual_git_commit)

        with sdk2.helpers.ProcessLog(
                self, logger=logging.getLogger('apply_patch'),
        ) as pl:
            proc = subprocess.Popen(
                [
                    'git',
                    'apply',
                    '--whitespace=nowarn',
                    '--directory',
                    USERVICES_DIR + '/userver',
                ],
                cwd=self.arcadia_path(),
                stdin=subprocess.PIPE,
                stdout=pl.stdout,
                stderr=pl.stderr,
            )
            err = proc.communicate(diff)[1]
            if proc.returncode:
                raise RuntimeError(
                    'The patch cannot be applied: {}'.format(err),
                )

    def get_actual_git_commit(self):
        with open(self.arcadia_path(ACTUAL_GIT_COMMIT)) as file:
            actual_git_commit = file.readline().strip()
            logging.info('Actual git commit is {}'.format(actual_git_commit))
        return actual_git_commit

    def get_diff(self, actual_git_commit):

        with sdk2.helpers.ProcessLog(
                self, logger=logging.getLogger('get_diff'),
        ) as pl:
            proc = subprocess.Popen(
                [
                    'git',
                    'diff',
                    '--patch',
                    '--ignore-submodules',
                    '{base}..{target}'.format(
                        base=actual_git_commit, target=self.Parameters.commit,
                    ),
                ],
                cwd=self.workdir_path(USERVER_DIR),
                stdout=subprocess.PIPE,
                stderr=pl.stderr,
            )
            diff = proc.communicate()[0]
            if proc.returncode:
                raise RuntimeError('Git diff failed')
            return diff

    def gen_ya_make_files(self):
        arcadiasdk.do_build(
            build_system=consts.YMAKE_BUILD_SYSTEM,
            source_root=self.arcadia_path(),
            targets=[self.arcadia_path(YA_MAKE_CODEGEN_DIR)],
            results_dir=self.workdir_path(),
            clear_build=False,
        )

        with sdk2.helpers.ProcessLog(
                self, logger=logging.getLogger('gen_ya_make_files'),
        ) as pl:
            proc = subprocess.Popen(
                [
                    self.workdir_path(YA_MAKE_CODEGEN_DIR, 'ya-make-codegen'),
                    '.',
                ],
                cwd=self.arcadia_path(USERVICES_DIR),
                stdout=pl.stdout,
                stderr=pl.stderr,
            )
            proc.communicate()
            if proc.returncode:
                raise RuntimeError('Generating ya makes failed!')

    def check_arc_status(self):
        with sdk2.helpers.ProcessLog(
                self, logger=logging.getLogger('check_arc_status'),
        ) as pl:
            proc = subprocess.Popen(
                ['arc', 'status'],
                cwd=self.arcadia_path(),
                stdout=subprocess.PIPE,
                stderr=pl.stderr,
            )
            out, err = proc.communicate()

            if proc.returncode:
                raise RuntimeError('Arc status failed: {err}'.format(err=err))

            if not out:
                logging.info('No changes found')
                return

            logging.info('Changed files:\n {diff}'.format(diff=out))

        self.save_arc_diff_in_file()

    def save_arc_diff_in_file(self):
        with sdk2.helpers.ProcessLog(
                self, logger=logging.getLogger('save_patch_file'),
        ) as pl:
            proc = subprocess.Popen(
                ['arc', 'diff', '.'],
                cwd=self.arcadia_path(USERVICES_DIR),
                stdout=subprocess.PIPE,
                stderr=pl.stderr,
            )
            diff = proc.communicate()[0]
            if proc.returncode:
                raise RuntimeError('Arc diff failed!')

            diff_file = self.artifacts_path('diff.patch')
            with open(diff_file, 'w') as file:
                file.write(diff)

            logging.info(
                'Save diff in file {diff_file}'.format(diff_file=diff_file),
            )

    def post_build(self, source_dir, output_dir, pack_dir):
        arcadiasdk.do_build(
            build_system=consts.YMAKE_BUILD_SYSTEM,
            build_type=consts.RELEASE_WITH_DEBUG_INFO_BUILD_TYPE,
            check_return_code=True,
            clear_build=False,
            junit_report_path=self.artifacts_path('junit_report.xml'),
            keep_on=True,
            results_dir=self.workdir_path(),
            source_root=source_dir,
            targets=self.get_test_targets(),
            test=True,
            test_size_filter=['SMALL', 'MEDIUM', 'LARGE'],
        )
        YaMake2.post_build(self, source_dir, output_dir, pack_dir)

    def get_test_targets(self):
        targets = [
            USERVICES_DIR + '/services/' + target.strip()
            for target in self.Parameters.test_targets.split(';')
        ]

        if self.Parameters.check_libraries:
            targets.append(USERVICES_DIR + '/libraries/')

        logging.info('Run tests on {}'.format(targets))
        return targets

    def on_finish(self, prev_status, status):
        self.publish_artifacts()

    def publish_artifacts(self):
        junit_report = str(self.log_path('junit_report.xml'))
        if os.path.exists(junit_report):
            shutil.copy(junit_report, self.artifacts_path())
        if self.is_artifacts_exist():
            artifacts_resource = teamcity.TeamcityArtifacts(
                self, 'Teamcity artifacts', self.artifacts_path(),
            )
            sdk2.ResourceData(artifacts_resource).ready()

    def is_artifacts_exist(self):
        for root, dirs, files in os.walk(self.artifacts_path()):
            if files:
                return True
        logging.info('No files in artifacts dir, nothing will be saved')
        return False
