import os
import logging
import yaml
import pytz

import datetime
from typing import NamedTuple

from sandbox import sdk2
import sandbox.common.types.client as ctc
from sandbox.projects.common.arcadia import sdk as arcadiasdk
from concurrent.futures import TimeoutError


class CountTier0Services(sdk2.Task):
    class Parameters(sdk2.Parameters):

        with sdk2.parameters.Group('YDB parameters'):
            secret_ydb_token = sdk2.parameters.YavSecret(
                "YAV secret identifier",
                default="sec-01ftrnb4qhbxp889tj0myt4tj3@ver-01ftrnb4qswwy6crfwdgp925zj"
            )
            ydb_endpoint = sdk2.parameters.String(
                'Cluster (endpoint)',
                default="grpc://ydb-ru-prestable.yandex.net:2135"
            )
            ydb_database = sdk2.parameters.String(
                'Database',
                default='/ru-prestable/ydb_home/gareeva-alice/count-tier-0-services-ydb'
            )
            ydb_table = sdk2.parameters.String(
                'Table',
                default="count_tier_0_services_table"
            )

    ServicesInfo = NamedTuple('ServicesInfo',
                              [('checking_date', str),
                               ('count_tier_0_services', int),
                               ('count_all_services', int)])

    def on_execute(self):
        count_tier_0_services, count_all_services = self._count_tier_0_services()
        checking_date = datetime.datetime.now(pytz.timezone('Europe/Moscow')).strftime('%Y-%m-%dT00:00:00Z')
        logging.info('\n date: {}, ya_make_services: {}, all_services: {}'
                     .format(checking_date, count_tier_0_services, count_all_services))
        self._save_to_ydb(self.ServicesInfo(checking_date, count_tier_0_services, count_all_services))

    def _count_tier_0_services(self):
        count_tier_0_services = count_all_services = 0
        with arcadiasdk.mount_arc_path(sdk2.svn.Arcadia.ARCADIA_TRUNK_URL) as arcadia_path:
            services_path = os.path.join(arcadia_path, 'taxi', 'uservices', 'services')
            if not os.path.exists(services_path):
                raise Exception('doesnt exist: {}'.format(services_path))
            all_services = os.listdir(services_path)
            for service in all_services:
                yaml_service_path = os.path.join(services_path, service, 'service.yaml')
                if os.path.exists(yaml_service_path):
                    is_ya_make_enabled = self._is_ya_make_enabled(yaml_service_path)
                    logging.info('enabled_ya_make: {}, \t'
                                 'service: {}'.format(is_ya_make_enabled,
                                                      service))
                    if is_ya_make_enabled:
                        count_tier_0_services += 1
                    count_all_services += 1
                else:
                    logging.info('not a service: {}'.format(service))
        return count_tier_0_services, count_all_services

    @staticmethod
    def _is_ya_make_enabled(yaml_service_path):
        with open(yaml_service_path, "r") as yaml_file:
            try:
                yaml_dict = yaml.safe_load(yaml_file)
                if 'ya-make' in yaml_dict and 'enabled' in yaml_dict['ya-make']:
                    return yaml_dict['ya-make']['enabled']
            except yaml.YAMLError as exc:
                raise Exception(exc)
        return False

    def _save_to_ydb(self, services_info):
        import ydb
        driver_config = ydb.DriverConfig(endpoint=self.Parameters.ydb_endpoint,
                                         database=self.Parameters.ydb_database,
                                         auth_token=self.Parameters.secret_ydb_token.data()['ydb_token'])
        with ydb.Driver(driver_config) as driver:
            try:
                driver.wait(timeout=5)
                session = driver.table_client.session().create()
                self._create_table(session)
                self._add_row_to_table(session, services_info)
            except TimeoutError:
                raise Exception('Connect failed to YDB')

    def _create_table(self, session):
        import ydb
        session.create_table(
            os.path.join(self.Parameters.ydb_database, self.Parameters.ydb_table),
            ydb.TableDescription()
                .with_column(ydb.Column('checking_date', ydb.OptionalType(ydb.PrimitiveType.Datetime)))
                .with_column(ydb.Column('tier_0_services', ydb.OptionalType(ydb.PrimitiveType.Uint32)))
                .with_column(ydb.Column('all_services', ydb.OptionalType(ydb.PrimitiveType.Uint32)))
                .with_primary_key('checking_date')
        )

    def _add_row_to_table(self, session, services_info):
        import ydb
        session.transaction(ydb.SerializableReadWrite()).execute(
            """
            UPSERT INTO {} (checking_date, tier_0_services, all_services)
            VALUES (Datetime("{}"), {}, {});
            """.format(self.Parameters.ydb_table,
                       services_info.checking_date,
                       services_info.count_tier_0_services,
                       services_info.count_all_services),
            commit_tx=True,
        )

    class Requirements(sdk2.Requirements):
        client_tags = ctc.Tag.Group.LINUX
        cores = 1
        disk_space = 128
        ram = 128

        class Caches(sdk2.Requirements.Caches):
            pass
