import logging
import json
import os
from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess
from sandbox import common

wait_until_items = [
    "domcontentloaded",
    "load",
    "networkidle",
    "commit"
]

class TrackerScreenshooterTestData(sdk2.Resource):
    """ Screenshooter test code """

class TrackerScreenshooterPwTestResultsData(sdk2.Resource):
    """ Screenshooter playwright raw test results """

class TrackerScreenshooterReportData(sdk2.Resource):
    """ Screenshooter test report """

class TrackerScreenshooter(sdk2.Task):
    """
    Screenshooter Task
    """

    class Requirements(sdk2.Requirements):
        cores = 8
        ram = 8072
        disk_space = 1024
        dns = common.types.misc.DnsType.DNS64

        class Caches(sdk2.Requirements.Caches):
            pass  # Do not use any shared caches (required for running on multislot agent)

    class Parameters(sdk2.Task.Parameters):
        description = "Screenshooter Task"
        title = sdk2.parameters.String("Title", default="Screenshooter Tests")
        testHost = sdk2.parameters.String("Test host", required=True)
        goldenHost = sdk2.parameters.String("Golden host", required=True)
        dataGridPath = sdk2.parameters.String("Path to Wiki Grid", required=True)
        usersSecret = sdk2.parameters.YavSecret("Users data", required=True)
        tokens = sdk2.parameters.YavSecret("Tokens", required=True)
        user = sdk2.parameters.String("Default user")
        goldenUser = sdk2.parameters.String("Default golden user")
        timeout = sdk2.parameters.Integer("Test timeout")
        delay = sdk2.parameters.Integer("Delay before screenshot", default=1000)
        increaseDelayOnRetry = sdk2.parameters.Bool("Increase delay on retry")
        retries = sdk2.parameters.Integer("Test retries", default=1)
        waitUntil = sdk2.parameters.String(
            "Page load event",
            choices=[(item, item) for item in wait_until_items]
        )
        mask = sdk2.parameters.JSON("Mask elements")
        screenshotOptions = sdk2.parameters.JSON("Screenshot options")
        testsOptions = sdk2.parameters.JSON("Tests options")
        actions = sdk2.parameters.JSON("Actions")
        headers = sdk2.parameters.JSON("Custom headers")
        cookies = sdk2.parameters.JSON("Custom cookies")
        skips = sdk2.parameters.JSON("Tests to skip")
        useProdDataForTesting = sdk2.parameters.Bool("Use prod data for testing")
        useTestingDataForGolden = sdk2.parameters.Bool("Use testing data for golden")
        trackerNotification = sdk2.parameters.JSON("Notification to Tracker Issue")

    class Context(sdk2.Task.Context):
        report_resource_id = None

    @sdk2.header(title="Screenshooter Tests Result")
    def header(self):
        if self.Context.report_resource_id:
            proxy = sdk2.Resource[self.Context.report_resource_id].http_proxy
            return "<a href=\"{}/index.html\">Report</a>".format(proxy)
        else:
            return None

    def run_subprocess(self, name, command, env={}):
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger(name)) as process_log:
            return_code = subprocess.Popen(
                command,
                cwd=os.getcwd(),
                stdout=process_log.stdout,
                stderr=process_log.stderr,
                env=env
            ).wait()
            if return_code != 0:
                raise Exception("Command returned non-zero exit status {}".format(return_code))

    def on_execute(self):
        logging.info("Screenshooter task executed")

        config = {}
        config['cores'] = self.Requirements.cores
        config['title'] = self.Parameters.title
        config['testHost'] = self.Parameters.testHost
        config['goldenHost'] = self.Parameters.goldenHost
        config['dataGridPath'] = self.Parameters.dataGridPath

        if self.Parameters.user:
            config['user'] = self.Parameters.user
        if self.Parameters.goldenUser:
            config['goldenUser'] = self.Parameters.goldenUser
        if self.Parameters.timeout:
            config['timeout'] = self.Parameters.timeout
        if self.Parameters.delay:
            config['delay'] = self.Parameters.delay
        if self.Parameters.increaseDelayOnRetry:
            config['increaseDelayOnRetry'] = self.Parameters.increaseDelayOnRetry
        if self.Parameters.retries:
            config['retries'] = self.Parameters.retries
        if self.Parameters.waitUntil:
            config['waitUntil'] = self.Parameters.waitUntil
        if self.Parameters.mask:
            config['mask'] = self.Parameters.mask
        if self.Parameters.screenshotOptions:
            config['screenshotOptions'] = self.Parameters.screenshotOptions
        if self.Parameters.testsOptions:
            config['testsOptions'] = self.Parameters.testsOptions
        if self.Parameters.actions:
            config['actions'] = self.Parameters.actions
        if self.Parameters.headers:
            config['headers'] = self.Parameters.headers
        if self.Parameters.cookies:
            config['cookies'] = self.Parameters.cookies
        if self.Parameters.skips:
            config['skips'] = self.Parameters.skips
        if self.Parameters.useProdDataForTesting:
            config['useProdDataForTesting'] = self.Parameters.useProdDataForTesting
        if self.Parameters.useTestingDataForGolden:
            config['useTestingDataForGolden'] = self.Parameters.useTestingDataForGolden

        config_json = json.dumps(config)
        logging.info("Config %s", config_json)

        with open('config.json', 'wb') as config_file:
            config_file.write(config_json)

        with open('users.json', 'wb') as users_file:
            users_file.write(self.Parameters.usersSecret.data()[self.Parameters.usersSecret.default_key])

        cwd = os.getcwd()

        self.run_subprocess(
            "node",
            [
                "/usr/bin/env",
                "node",
                "--version"
            ]
        )

        self.run_subprocess(
            "ls",
            [
                "ls",
                "-lah",
                "/screenshooter"
            ]
        )

        self.run_subprocess(
            "files",
            [
                "ln",
                "-s",
                "/screenshooter/node_modules",
                cwd + "/node_modules"
            ]
        )
        self.run_subprocess(
            "files",
            [
                "cp",
                "/screenshooter/node_modules/@yandex-data-ui/screenshooter/playwright.config.ts",
                cwd + "/playwright.config.ts"
            ]
        )
        self.run_subprocess(
            "files",
            [
                "cp",
                "/screenshooter/package.json",
                cwd + "/package.json"
            ]
        )

        self.run_subprocess(
            "ls",
            [
                "ls",
                "-lah",
                "."
            ]
        )

        tokens = self.Parameters.tokens.data()

        subprocess_env = os.environ.copy()
        subprocess_env["CI"] = "1"
        subprocess_env["REQUIRED_CORES"] = str(self.Requirements.cores)
        subprocess_env["PLAYWRIGHT_BROWSERS_PATH"] = "/screenshooter/browsers"
        subprocess_env["USERS_DATA_PATH"] = cwd + "/users.json"
        subprocess_env["SCREENSHOOTER_CONFIG_PATH"] = cwd + "/config.json"
        subprocess_env["WIKI_OAUTH_TOKEN"] = tokens["wiki"]
        subprocess_env["TRACKER_OAUTH_TOKEN"] = tokens["tracker"]
        if self.Parameters.trackerNotification:
            subprocess_env['TRACKER_NOTIFICATION'] = json.dumps(self.Parameters.trackerNotification)

        self.run_subprocess(
            "screenshooter",
            [
                "/usr/bin/env",
                "node",
                "/screenshooter/node_modules/.bin/screenshooter",
                cwd + "/config.json"
            ],
            subprocess_env
        )

        test_data_resource_data = sdk2.ResourceData(TrackerScreenshooterTestData(
            self,
            "Generated test code",
            cwd + "/screenshooter.test.ts"
        ))
        test_data_resource_data.ready()

        report_data_resource = TrackerScreenshooterReportData(
            self,
            "Report",
            cwd + "/playwright-report"
        )

        self.Context.report_resource_id = report_data_resource.id
        subprocess_env["REPORT_HEADER"] = "{}\nhttps://sandbox.yandex-team.ru/task/{}\n[Report]({}/index.html)".format(
            self.Parameters.title,
            self.id,
            report_data_resource.http_proxy
        )

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("playwright")) as process_log:
            tests_return_code = subprocess.Popen(
                [
                    "/usr/bin/env",
                    "node",
                    "/screenshooter/node_modules/.bin/playwright",
                    "test",
                    "--config=" + cwd + "/playwright.config.ts",
                    cwd + "/screenshooter.test.ts"
                ],
                cwd=cwd,
                stdout=process_log.stdout,
                stderr=process_log.stderr,
                env=subprocess_env
            ).wait()

        pw_test_results_path = cwd + "/pw-test-results"

        try:
            if os.path.exists(pw_test_results_path) and not len(os.listdir(pw_test_results_path)) == 0:
                with open(pw_test_results_path + '/placeholder', 'wb') as placeholder_file:
                    placeholder_file.write('')
                pw_test_results_data_resource_data = sdk2.ResourceData(TrackerScreenshooterPwTestResultsData(
                    self,
                    "Playwright raw results",
                    pw_test_results_path
                ))
                pw_test_results_data_resource_data.ready()
        except Exception:
            sys.exc_clear()

        report_data_resource_data = sdk2.ResourceData(report_data_resource)
        report_data_resource_data.ready()

        if tests_return_code != 0:
            raise common.errors.TaskFailure("Tests failed")
