# coding=utf-8

import logging
import os
import re
import time

import requests

from sandbox import sdk2
from sandbox.sdk2.vcs.svn import Arcadia, SvnError

ROOT_ZONE_URL = "https://www.internic.net/domain/root.zone"
ROOT_ZONE_FILE_NAME = "root.zone"
MAX_COUNT_SAVE_FILE = 31


def get_root_zone():
    try:
        result = requests.get(ROOT_ZONE_URL)
        if result.status_code != 200:
            logging.error("Error get Root zone from %s and returned %d", result.content, result.status_code)
            return "", False
    except requests.RequestException as error:
        logging.error("Error occured while get Root Zone: '%s'.", error)
        return "", False
    logging.info("Get Root Zone successfuly")
    return result.text, True


class DnsSaveRootZone(sdk2.Task):
    """Коммит файла в Аркадию."""

    class Parameters(sdk2.Task.Parameters):
        file_path = sdk2.parameters.String("Аркадийный путь к каталогу", default="noc/traffic/dns/root_hints/",
                                           required=True)
        commit_message = sdk2.parameters.String("Сообщение коммита", default="SKIP_CHECK() Update file", required=True)
        arcadia_user = sdk2.parameters.String("Автор коммита", default="zomb-sandbox-rw", required=True)
        with sdk2.parameters.Output:
            arcadia_commit_url = sdk2.parameters.String("Commit URL", default="", required=True)

    def on_execute(self):
        # Checkout Arcadia
        arcadia_url = os.path.join(
            Arcadia.ARCADIA_TRUNK_URL,
            os.path.dirname(self.Parameters.file_path))
        arcadia_dir = Arcadia.checkout(arcadia_url, str(self.path("arcadia")))
        logging.info("Arcadia is checkouted to {}".format(arcadia_dir))

        # Write to file
        project_dir = os.path.join(arcadia_dir, os.path.basename(self.Parameters.file_path))
        file_name = os.path.join(project_dir, os.path.basename(ROOT_ZONE_FILE_NAME + "." + str(int(time.time()))))
        if os.path.isfile(file_name):
            logging.info("File {} exists".format(file_name))
            return

        rz, ok = get_root_zone()
        if not ok:
            return
        with open(file_name, "w") as fd:
            fd.write(rz)
        logging.info("Write content to file {}".format(file_name))
        Arcadia.add(file_name)
        logging.info("Add file {} to arcadia".format(file_name))

        # Rotate old file
        rh_files = os.listdir(project_dir)
        for f in rh_files:
            if not f.startswith(ROOT_ZONE_FILE_NAME):
                rh_files.remove(f)
        if len(rh_files) > MAX_COUNT_SAVE_FILE:
            for f in sorted(rh_files, reverse=True)[MAX_COUNT_SAVE_FILE:]:
                Arcadia.delete(os.path.join(project_dir, f))
                logging.info("Delete file {} from arcadia".format(f))

        # Commit file
        try:
            result = Arcadia.commit(arcadia_dir, self.Parameters.commit_message, self.Parameters.arcadia_user)
            logging.info("Commit result: {}".format(result))
            revision = re.findall(r'Committed revision (\d+)\.', result)
            if revision:
                self.Parameters.arcadia_commit_url = "https://a.yandex-team.ru/arc/commit/{}".format(revision[0])
        except SvnError as e:
            logging.error(e.message)
            # For Arcadia reviews
            review = re.findall(r'https://a\.yandex-team\.ru/review/(\d+)', e.message)
            if review:
                self.Parameters.arcadia_commit_url = "https://a.yandex-team.ru/review/{}".format(review[0])

        logging.info("Commit/Review URL: {}\nAll done".format(self.Parameters.arcadia_commit_url))
