import json
import logging
import os

import requests

import sandbox.common.types.task as ctt
from sandbox import sdk2
from sandbox.projects.common.ya_deploy import release_integration
from sandbox.sandboxsdk import environments

INTERNIC_URL = "https://www.internic.net/domain/named.root"
JUGGLER_HOST = "juggler-push.search.yandex.net"
MAX_LABEL_LENGTH = 63
MAX_FQDN_LENGTH = 255
PORN_RPZ_MIN_SIZE = 838860800  # 800 MiB
PORN_TABLE = "//home/antispam/export/dns/porno"
THREATS_RPZ_MIN_SIZE = 5242880  # 5 MiB
THREATS_TABLE = "//home/antivir/prod/export/dns/threats"
YT_CLUSTER = "arnold"
YA_SEARCH_DOMAINS = [
    "yandex.by", "yandex.ru", "yandex.ua", "yandex.kz", "images.yandex.by",
    "images.yandex.ru", "images.yandex.ua", "images.yandex.kz"
]


class CustomSandboxTaskException(Exception):
    pass


def send_signal_to_juggler(message, status="OK"):
    post_payload = {
        "events": [{
            "description": message,
            "host": "safedns-sandbox",
            "service": "safedns_rpz_creation",
            "instance": "",
            "status": status,
            "tags": ["sandbox", "release_integration", "rpz"]
        }]
    }
    headers = {"Content-Type": "application/json"}
    try:
        result = requests.post("http://{}/events".format(JUGGLER_HOST),
                               headers=headers,
                               data=json.dumps(post_payload))
        if result.status_code != 200:
            logging.error(
                "Juggler returned %d with %s to our push about RPZ creation",
                result.status_code, result.content)
            return False
    except requests.RequestException as error:
        logging.error("Error occured while sending to Juggler: '%s'.", error)
        return False
    logging.info("Pushed event to Juggler successfuly")
    return True


def get_name_roots(retries=3):
    if retries < 0:
        return None
    logging.info("Connecting to %s, will retry %d times", INTERNIC_URL,
                 retries)
    try:
        response = requests.get(INTERNIC_URL)
    except requests.RequestException as request_error:
        logging.error("Downloading of name roots failed: %s", request_error)
        return get_name_roots(retries=retries - 1)
    if response.status_code != 200:
        logging.error("INTERNIC response status is %d", response.status)
        return get_name_roots(retries=retries - 1)
    named_root_data = response.text
    logging.info("Download successful!")
    return named_root_data


def gen_google_domains():
    country_domains_list = [
        "com", "ad", "ae", "com.af", "com.ag", "com.ai", "al", "am", "co.ao",
        "com.ar", "as", "at", "com.au", "az", "ba", "com.bd", "be", "bf", "bg",
        "com.bh", "bi", "bj", "com.bn", "com.bo", "com.br", "bs", "bt",
        "co.bw", "by", "com.bz", "ca", "cd", "cf", "cg", "ch", "ci", "co.ck",
        "cl", "cm", "cn", "com.co", "co.cr", "com.cu", "cv", "com.cy", "cz",
        "de", "dj", "dk", "dm", "com.do", "dz", "com.ec", "ee", "com.eg", "es",
        "com.et", "fi", "com.fj", "fm", "fr", "ga", "ge", "gg", "com.gh",
        "com.gi", "gl", "gm", "gp", "gr", "com.gt", "gy", "com.hk", "hn", "hr",
        "ht", "hu", "co.id", "ie", "co.il", "im", "co.in", "iq", "is", "it",
        "je", "com.jm", "jo", "co.jp", "co.ke", "com.kh", "ki", "kg", "co.kr",
        "com.kw", "kz", "la", "com.lb", "li", "lk", "co.ls", "lt", "lu", "lv",
        "com.ly", "co.ma", "md", "me", "mg", "mk", "ml", "com.mm", "mn", "ms",
        "com.mt", "mu", "mv", "mw", "com.mx", "com.my", "co.mz", "com.na",
        "com.nf", "com.ng", "com.ni", "ne", "nl", "no", "com.np", "nr", "nu",
        "co.nz", "com.om", "com.pa", "com.pe", "com.pg", "com.ph", "com.pk",
        "pl", "pn", "com.pr", "ps", "pt", "com.py", "com.qa", "ro", "ru", "rw",
        "com.sa", "com.sb", "sc", "se", "com.sg", "sh", "si", "sk", "com.sl",
        "sn", "so", "sm", "sr", "st", "com.sv", "td", "tg", "co.th", "com.tj",
        "tk", "tl", "tm", "tn", "to", "com.tr", "tt", "com.tw", "co.tz",
        "com.ua", "co.ug", "co.uk", "com.uy", "co.uz", "com.vc", "co.ve", "vg",
        "co.vi", "com.vn", "vu", "ws", "rs", "co.za", "co.zm", "co.zw", "cat"
    ]
    plain_domains = ["google.{}".format(x) for x in country_domains_list]
    www_domains = ["www.google.{}".format(x) for x in country_domains_list]
    plain_domains.extend(www_domains)
    return plain_domains


def validate_fqdn_len(fqdn):
    if len(fqdn) > MAX_FQDN_LENGTH:
        return False
    labels = fqdn.split(".")
    for label in labels:
        if len(label) > MAX_LABEL_LENGTH:
            return False
    return True


class SafeDnsThreatsRpzLayer(sdk2.Resource):
    any_arch = True
    auto_backup = False
    executable = False
    releasable = True
    ttl = 1


class SafeDnsPornRpzLayer(sdk2.Resource):
    any_arch = True
    auto_backup = False
    executable = False
    releasable = True
    ttl = 1


class SafeDnsYaSearchRpzLayer(sdk2.Resource):
    any_arch = True
    auto_backup = False
    executable = False
    releasable = True
    ttl = 1


class SafeDnsGoogleSearchRpzLayer(sdk2.Resource):
    any_arch = True
    auto_backup = False
    executable = False
    releasable = True
    ttl = 1


class SafeDnsRootHintsLayer(sdk2.Resource):
    any_arch = True
    auto_backup = False
    executable = False
    releasable = True
    ttl = 1


class SafeDnsConfig(release_integration.ReleaseToYaDeployTask2, sdk2.Task):

    YP_TOKEN_YAV_SECRET_ID = "sec-01f9v3hezm7armjdcv58rec6mn"

    class Requirements(sdk2.Task.Requirements):
        environments = [environments.PipEnvironment("yandex-yt")]

    class Parameters(sdk2.Task.Parameters):
        yp_api_url = sdk2.parameters.String(
            "YP address",
            default=release_integration.DEFAULT_YP_API_URL,
            required=True)
        tokens = sdk2.parameters.YavSecret(
            "YAV secret identifier", default="sec-01f9v3hezm7armjdcv58rec6mn")

    def get_yp_api_url(self):
        return self.Parameters.yp_api_url

    def on_execute(self):
        from yt.wrapper import YtClient

        tokens = self.Parameters.tokens.data()
        yt_token = tokens["yt-token"]
        named_root_data = get_name_roots()
        if not named_root_data:
            msg = "Name roots can't be downloaded"
            send_signal_to_juggler(msg, status="CRIT")
            raise CustomSandboxTaskException(msg)
        g_search_domains = gen_google_domains()
        yt_client = YtClient(proxy=YT_CLUSTER,
                             token=yt_token,
                             config={"tabular_data_format": "json"})
        res_root_data = sdk2.ResourceData(
            SafeDnsRootHintsLayer(self,
                                  "DNS root servers hints",
                                  "named.root",
                                  pack_tar=2,
                                  ttl=1))
        res_threats_zone = sdk2.ResourceData(
            SafeDnsThreatsRpzLayer(self,
                                   "Threats RPZ file",
                                   "rpz.threats",
                                   pack_tar=2,
                                   ttl=1))
        res_porno_zone = sdk2.ResourceData(
            SafeDnsPornRpzLayer(self,
                                "Porno RPZ file",
                                "rpz.porno",
                                pack_tar=2,
                                ttl=1))
        res_g_search_zone = sdk2.ResourceData(
            SafeDnsGoogleSearchRpzLayer(self,
                                        "Google search zones",
                                        "rpz.gsearch",
                                        pack_tar=2,
                                        ttl=1))
        res_ya_search_zone = sdk2.ResourceData(
            SafeDnsYaSearchRpzLayer(self,
                                    "Yandex search zones",
                                    "rpz.yasearch",
                                    pack_tar=2,
                                    ttl=1))
        res_root_data.path.write_text(named_root_data)
        logging.info("Pulling FQDNs from YT")
        logging.info("Pulling threats to %s", res_threats_zone.path)
        with open(str(res_threats_zone.path), "w") as threats_file:
            threats_file.write("$ORIGIN rpz.threats.\n")
            for row in yt_client.read_table(THREATS_TABLE):
                fqdn = list(row.values())[0]
                if not validate_fqdn_len(fqdn):
                    continue
                threats_file.write("{} CNAME .\n".format(fqdn))
        fstats = os.stat(str(res_threats_zone.path))
        if fstats.st_size < THREATS_RPZ_MIN_SIZE:
            send_signal_to_juggler("Threats RPZ is too small: {}MiB".format(
                fstats.st_size // 1024 // 1024), status="CRIT")
            raise CustomSandboxTaskException(
                "Resulted threats RPZ size is too small: {}".format(
                    fstats.st_size // 1024 // 1024))
        logging.info("Pulling porno to %s", res_porno_zone.path)
        with open(str(res_porno_zone.path), "w") as porno_file:
            porno_file.write("$ORIGIN rpz.porno.\n")
            for row in yt_client.read_table(PORN_TABLE):
                fqdn = list(row.values())[0]
                if not validate_fqdn_len(fqdn):
                    continue
                porno_file.write("{} CNAME .\n".format(fqdn))
        fstats = os.stat(str(res_porno_zone.path))
        if fstats.st_size < PORN_RPZ_MIN_SIZE:
            send_signal_to_juggler("Porn RPZ is too small: {}MiB".format(
                fstats.st_size // 1024 // 1024), status="CRIT")
            raise CustomSandboxTaskException(
                "Resulted porno RPZ size is too small: {}".format(
                    fstats.st_size // 1024 // 1024))
        with open(str(res_g_search_zone.path), "w") as gsearch_file:
            gsearch_file.write("$ORIGIN rpz.gsearch\n")
            for domain in g_search_domains:
                gsearch_file.write("{} CNAME .\n".format(domain))
        with open(str(res_ya_search_zone.path), "w") as yasearch_file:
            yasearch_file.write("$ORIGIN rpz.yasearch\n")
            for domain in YA_SEARCH_DOMAINS:
                yasearch_file.write("{} CNAME .\n".format(domain))
        logging.info("All done!")

    def on_success(self, prev_status):
        sdk2.Task.on_success(self, prev_status)
        release_integration.ReleaseToYaDeployTask2.on_release(
            self,
            dict(releaser=self.author,
                 release_status=ctt.ReleaseStatus.STABLE,
                 release_subject="SafeDNS unbound RPZ files",
                 email_notifications=dict(to=[], cc=[]),
                 release_comments="SafeDNS unbound RPZ files"))
        send_signal_to_juggler("RPZ creation successful")
