# coding=utf-8
import os
import uuid
from datetime import datetime, timedelta


def last_logs_tables(yt, path, days):
    today = datetime.now().date()
    return tables_for_daterange(yt, path, today - timedelta(days), today)


TIMESTAMP_FORMAT = '%Y-%m-%dT%H:%M:%S'
DATE_FORMAT = '%Y-%m-%d'


def _parse_timestamp(timestamp):
    import pytz
    moscow_tz = pytz.timezone('Europe/Moscow')
    if timestamp is None or timestamp == '' or timestamp == 'invalid-date':
        return None
    for date_format in (TIMESTAMP_FORMAT, DATE_FORMAT):
        try:
            return moscow_tz.localize(datetime.strptime(timestamp, date_format))
        except ValueError:
            continue
    raise ValueError('Timestamp has unsupported format: {}'.format(timestamp))


def _get_timestamp_from_table_path(table_path):
    if table_path is None:
        return None
    return _parse_timestamp(str(table_path).split('/')[-1])


def _table_date(table):
    table_date = table.split('/')[-1]
    try:
        d = datetime.strptime(table_date, '%Y-%m-%d')
    except ValueError:
        d = datetime.strptime(table_date, '%Y-%m-%dT%H:%M:%S')
    return d.date()


def tables_for_daterange(yt, path, begin, end):
    tables = []
    for t in yt.search(path, node_type='table'):
        try:
            if begin <= _table_date(t) <= end:
                tables.append(t)
        except:
            continue

    return tables


def _get_tables(yt, root_dir, start, end=None, columns=None, last_offset=None, table_time_delta=timedelta(days=1),
                logger=None):
    logger.info('_get_tables(start=%s, end=%s)', start, end)
    for table in yt.search(root_dir, node_type='table', attributes=['row_count']):
        if logger:
            logger.info('table=%s', table)

        table_time = _get_timestamp_from_table_path(table)
        if table_time is None:
            continue

        if start-table_time_delta <= table_time and (end is None or table_time < end):
            if logger:
                logger.info('table_time=%s, start=%s, end=%s, last_offset=%s', table_time, start, end, last_offset)
            start_index = last_offset if table_time == start else 0
            end_index = table.attributes['row_count']
            yield yt.TablePath(table, columns=columns, start_index=start_index, end_index=end_index)


def _get_table_time_delta(yt, root_dir):
    table_times = sorted([_get_timestamp_from_table_path(table)
                          for table in yt.search(root_dir, node_type='table', attributes=['row_count'])
                          if _get_timestamp_from_table_path(table) is not None])
    if not table_times:
        return timedelta(days=1)
    return min(map(lambda a, b: b - a, table_times[:-1], table_times[1:]))


def yt_read_tables(yt, tables, columns=None, yt_format=None, add_index=False):
    from yt.wrapper import JsonFormat
    yt_format = yt_format or JsonFormat()
    table_generator = (yt.TablePath(table, columns=columns) for table in tables)
    for table in table_generator:
        for index, record in enumerate(yt.read_table(table, format=yt_format)):
            if add_index:
                record['row_index'] = index
                record['table_name'] = table
            yield record


def create_safe_temp_table(ytc, prefix, path='//home/rasp/tmp/', attributes=None):
    """
    Создает временную таблицу в `path`
    (по мотивам https://st.yandex-team.ru/YTADMINREQ-6326)
    Удалением занимается вызывающая сторона
    """

    if prefix[0] == '/':
        # Делаем из абослютного пути относительный
        prefix = prefix[1:]

    table_name = os.path.join(path, prefix + '-' + str(uuid.uuid4()))
    ytc.create('table', table_name, recursive=True, attributes=attributes)

    return table_name


class YtClientFactory(object):
    @staticmethod
    def create(proxy, token):
        from yt.wrapper import YtClient
        config = {
            'operation_tracker': {
                'stderr_logging_level': 'WARNING',
                'progress_logging_level': 'WARNING',
            },
            'read_retries': {
                'enable': True,
            },
            'clear_local_temp_files': True,
            'pickling': {
                'python_binary': '/skynet/python/bin/python',
            },
            'remote_temp_tables_directory': '//home/avia/tmp',
        }

        # https://ml.yandex-team.ru/thread/yt/167477611142886638/
        try:
            import gevent
            config['pickling']['additional_files_to_archive'] = [
                (os.path.join(os.path.dirname(gevent.__file__), 'signal.py'), 'gevent/signal.py')
            ]

        except ImportError:
            pass

        return YtClient(
            proxy=proxy,
            token=token,
            config=config,
        )


def copy_table_to_weak(ytc, from_table, to_table, logger=None):
    """
    Команде abt надо чтоб мы генерировали таблицы с weak схемой. YQL всегда генерирует strong-таблицы.
    Поменять схему готовой таблице нельзя.
    Решение от команды yt - сделать пустую таблицу с weak-схемой и смержить в нее таблицу с нашими данными
    с подходящим schema_inference_mode.
    :param ytc: YtClient
    :param from_table: таблица с данными со strong-схемой
    :param to_table: путь к итоговой таблице, куда будут скопированы данные
    :return:
    """
    if logger:
        logger.info('Copy %s to %s with weak shema', from_table, to_table)

    ytc.create('table', to_table, ignore_existing=True)
    if ytc.get_attribute(to_table, 'schema_mode') != 'weak':
        raise ValueError('%s schema is not weak', to_table)
    ytc.run_merge(from_table, to_table, spec={'schema_inference_mode': 'from_output'})

    if logger:
        logger.info('Table copied')
