# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
import os
from datetime import datetime, timedelta

from sandbox import sdk2
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

from sandbox.projects.travel_analytics.lib import yql_helpers as yqlh
from sandbox.projects.travel_analytics.lib.yt_helpers import copy_table_to_weak, YtClientFactory


YESTERDAY = datetime.today() - timedelta(days=1)


class TravelMakeExternalAbtTable(sdk2.Task):
    """
    Send Travel wizards metrics to ABT. Train, avia
    """

    _yt_client = None
    _yql_client = None

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 1024

        class Caches(sdk2.Requirements.Caches):
            pass # We do not need caches

        environments = (
            PipEnvironment('yandex-yt'),
            PipEnvironment('requests'),
            PipEnvironment('yql')
        )

    class Parameters(sdk2.Task.Parameters):

        with sdk2.parameters.Group('Map reduce settings') as mr_block:
            mr_cluster = sdk2.parameters.String('MapReduce cluster', default='hahn', required=True)
            vaults_owner = sdk2.parameters.String('MapReduce user', required=True)
            mr_dir = sdk2.parameters.String('Directory', required=True, default='//home/avia/abt_wizard')
            tag_name = sdk2.parameters.String('Tag name', required=True, default='calc_avia_wizard_show_metrics')
            parameter_prefix = sdk2.parameters.String('Parameter prefix', required=True, default='wiz_avia')
            yql_title = sdk2.parameters.String('YQL title',
                                               required=True,
                                               default='[YQL] Generating Avia wizard show metrics')
            yt_token_vault_name = sdk2.parameters.String('YT Token vault name', required=True, default='YT_TOKEN')

        with sdk2.parameters.Group('YQL settings') as yql_block:
            yql_token_vault_name = sdk2.parameters.String('YQL token vault name', required=True, default='YQL_TOKEN')
            query = sdk2.parameters.String("YQL Query", multiline=True, required=False)

        with sdk2.parameters.Group('Date settings') as date_block:
            left_date = sdk2.parameters.String('Start date (default yesterday)', required=False)
            right_date = sdk2.parameters.String('End date (default yesterday)', required=False)

        with sdk2.parameters.Group('Debug settings') as debug_settings:
            debug_run = sdk2.parameters.Bool('Debug run', default=False, required=True)

    @property
    def yt_client(self):
        if self._yt_client is None:
            import yt.wrapper as yt
            self._yt_client = YtClientFactory.create(
                proxy=self.Parameters.mr_cluster,
                token=sdk2.Vault.data(self.Parameters.vaults_owner, self.Parameters.yt_token_vault_name),
            )

        return self._yt_client

    @property
    def yql_client(self):
        if self._yql_client is None:
            from yql.api.v1.client import YqlClient
            self._yql_client = YqlClient(
                token=sdk2.Vault.data(self.Parameters.vaults_owner, self.Parameters.yql_token_vault_name),
                db=self.Parameters.mr_cluster,
            )

        return self._yql_client

    def create_tag(self, date, cluster='hahn'):
        return 'cluster={cluster}_{tag_name}_{date}'.format(
            cluster=cluster,
            tag_name=self.Parameters.tag_name,
            date=date.strftime('%Y%m%d'),
        )

    def set_tag(self, tag):
        import requests
        return requests.get('http://rem-bernstein.n.yandex-team.ru/set_tag?tag=' + tag)

    def create_and_set_tag(self, date):
        tag = self.create_tag(date, self.Parameters.mr_cluster)
        logging.info('Tag: %s', tag)
        if not self.Parameters.debug_run:
            self.set_tag(tag)

    def process_date(self, date, output_path):
        logging.info('Process date: %s', date.strftime('%Y-%m-%d'))

        self.create_temp_table_with_abt_metrics(date, output_path)
        self.create_and_set_tag(date)

    def create_temp_table_with_abt_metrics(self, date, output_path):
        from yql.client.parameter_value_builder import YqlParameterValueBuilder as ValueBuilder

        r = self.yql_client.query(
            self.Parameters.query,
            syntax_version=1,
            title=self.Parameters.yql_title,
        ).run(parameters=ValueBuilder.build_json_map({
            '$OutputPath': ValueBuilder.make_string(output_path),
            '$Date': ValueBuilder.make_date(date),
            '$parameter_prefix': ValueBuilder.make_string(self.Parameters.parameter_prefix)
        }))

        logging.info('YQL Operation: %s', r.share_url)
        r.wait_progress()

        if not r.is_success:
            yqlh.log_errors(r, logging)
            raise SandboxTaskFailureError('YQL query failed')

        logging.info('YQL query done in %s table', output_path)

    def _parse_date(self, s):
        if not s:
            return YESTERDAY
        try:
            return datetime.today() + timedelta(days=int(s))
        except ValueError:
            return datetime.strptime(s, '%Y-%m-%d')

    def on_execute(self):
        logging.info('Start')

        left_date = self._parse_date(self.Parameters.left_date).date()
        logging.info('Left date: %s', left_date.strftime('%Y-%m-%d'))
        right_date = self._parse_date(self.Parameters.right_date).date()
        logging.info('Right date: %s', right_date.strftime('%Y-%m-%d'))

        if not self.yt_client.exists(self.Parameters.mr_dir):
            self.yt_client.create('map_node', self.Parameters.mr_dir, recursive=True)

        current_date = left_date
        while current_date <= right_date:
            self.process_date(current_date, self.Parameters.mr_dir)
            current_date += timedelta(days=1)

        logging.info('End')
