# -*- coding: utf-8 -*-

import logging

import sandbox.common.types.task as ctt
import sandbox.common.types.misc as ctm

from sandbox.sandboxsdk import sandboxapi
from sandbox.projects.trendbox_ci.beta.modules.meta import get_subtasks, create_subtask


class MetaTaskManager(object):
    def __init__(self, task):
        self.task = task

    @property
    def subtasks(self):
        return get_subtasks(self.task)

    @property
    def failed_subtasks(self):
        return self.filter_failed(self.subtasks)

    @property
    def not_finished_subtasks(self):
        return self.filter_not_finished(self.subtasks)

    @staticmethod
    def filter_failed(tasks):
        failure_statuses = (ctt.Status.FAILURE,) + tuple(ctt.Status.Group.BREAK)
        return filter(lambda task: task.status in failure_statuses, tasks)

    @staticmethod
    def filter_not_finished(tasks):
        finished_statuses = ctt.Status.Group.FINISH | ctt.Status.Group.BREAK
        return filter(lambda task: task.status not in finished_statuses, tasks)

    def store_failed_tasks(self, tasks):
        self.task.Context.failed_tasks.extend(t.id for t in tasks)

    def store_not_finished_tasks(self, tasks):
        self.task.Context.not_finished_tasks.extend(t.id for t in tasks)

    def create_subtask(self, task_type, wait_tasks=None, reusable=False, **parameters):
        if isinstance(wait_tasks, list):
            wait_tasks = filter(lambda task: task is not None, wait_tasks)

        reuse_task = reusable and self.task.project_conf.get('reuse_subtasks_cache', False)
        reuse_task = reuse_task and self.task.Parameters.reuse_subtasks_cache

        subtask_parameters = self.get_subtask_default_parameters()
        subtask_parameters.update(parameters)
        subtask_parameters.update(
            wait_tasks=self.get_tasks_ids(wait_tasks),
            reuse_task_cache=reuse_task,
        )

        return create_subtask(self.task, task_type, **subtask_parameters)

    def register_subtasks(self, subtasks):
        for subtask in subtasks:
            self.task.Context.subtasks_ids.append(subtask.id)

    def start_subtasks(self, subtasks):
        # subtasks может содержать None для задач, которые нужно пропустить.
        ids = [task.id for task in subtasks if task]

        result = self.task.server.batch.tasks.start.update(ids)
        logging.info('Subtasks start result {}'.format(result))
        for res in result:
            if res['status'] != ctm.BatchResultStatus.SUCCESS:
                msg = 'Task #{} is not enqueued: {}'.format(res['id'], res['message'])
                self.task.Context.noncritical_errors.append(msg)
                logging.error(msg)

        return ids

    def stop_tasks(self, tasks):
        stop_tasks_ids = [task.id for task in tasks]

        if len(stop_tasks_ids):
            logging.debug('Trying to stop tasks: {}'.format(stop_tasks_ids))
            self.task.server.batch.tasks.stop = map(int, stop_tasks_ids)  # PUT request

    def get_subtask_default_parameters(self):
        param_names = (
            # system params
            'owner',
            'priority',
            'notifications',
        )
        return {param_name: getattr(self.task.Parameters, param_name, None) for param_name in param_names}

    @staticmethod
    def get_tasks_ids(tasks):
        if not tasks:
            return []
        return [
            sandboxapi.Sandbox._to_id(task)
            for task in (tasks if hasattr(tasks, '__iter__') else [tasks])
        ]
