# -*- coding: utf-8 -*-

from sandbox.common.types import task as ctt
from sandbox.common.types import resource as ctr


class ReportsManager(object):
    """
    Reports collection.
    """

    def __init__(self, task):
        """
        :param task: Sandbox task instance.
        """
        self.task = task

    @staticmethod
    def status_badge(status):
        """
        Return HTML to display specified status.
        :param status: Status to display.
        :return: HTML string.
        """
        status_badge = {
            ctt.Status.SUCCESS: '<span class="status status_success">success</span>',
            ctt.Status.FAILURE: '<span class="status status_failure">failure</span>',
            ctr.State.BROKEN: '<span class="status status_broken">broken</span>',
        }
        return status_badge[status]

    @staticmethod
    def description(url, report_title, main_path):
        url = '{}/{}'.format(url, main_path) if main_path else url

        return'<a href="{url}" target="_blank">{text}</a>'.format(url=url, text=report_title)

    def lifelines_link(self):
        url = 'https://lifelines.si.yandex-team.ru/task/{}'.format(self.task.id)
        link = '<a href="{}" target="_blank">Lifelines</a>'.format(url)
        return {'<h3>Task flow</h3>': link}

    def trendbox_ui_logs_link(self, url_template):
        """
        :param url_template: шаблон ссылки на логи в Trendbox UI
        :type url_template: basestring
        :rtype: dict
        """
        if not url_template:
            return {}

        url = url_template.format(sandbox_task_id=self.task.id)

        return {'<h3 id="logs">Check steps logs</h3>': u'<a href="{}">Steps Logs</a>'.format(url)}

    def reports(self, reports):
        """
        Table with links to the reports.
        :return: Report object.
        """
        items = []

        if not reports:
            return {}

        for report in reports:
            report_id = report['id']
            report_info = self.task.server.resource[report_id].read()
            report_url = report_info['http']['proxy']
            report_status = report_info['attributes']['status']
            # see https://st.yandex-team.ru/FEI-4998
            if report_info['state'] == ctr.State.BROKEN:
                report_status = ctr.State.BROKEN
            main_path = report.get('main', None)
            report_title = report.get('title', None)

            items.append({
                'Status': self.status_badge(report_status),
                'Report': self.description(report_url, report_title, main_path),
            })

        return {'<h3 id="checks-reports">Checks reports</h3>': items}
