# -*- coding: utf-8 -*-

import logging

from sandbox import sdk2
import sandbox.common.types.task as ctt

from sandbox.projects.resource_types import TASK_CUSTOM_LOGS
from sandbox.projects.trendbox_ci.beta.resources import TRENDBOX_CI_REPORT_RESOURCE_BETA, TRENDBOX_CI_RESOURCE_BETA


class ResourcesManager(object):
    """
    Manager that helps to work with resources.
    """

    def __init__(self, task):
        """
        :param task: Task instance.
        """
        self.task = task

    def create_report(self, resource_path=None, add_to_context=False, **kwargs):
        """
        :param resource_path: path to resource relative to project or absolute path to resource
        :param add_to_context: appends generated resource id to `self.task.Context.report_resources`
        :param kwargs: additional attributes of created resource
        :return: Resource descriptor instance
        """
        # if resource_path is absolute, abs_path will become resource_path
        abs_path = self.task.path() / resource_path

        if not abs_path.exists():
            logging.debug('Not adding resource {}: no such path'.format(abs_path))
            return

        # sandbox fails to create resource with empty directory
        if abs_path.is_dir() and not any(abs_path.iterdir()):
            logging.debug('Not adding resource {}: directory is empty'.format(abs_path))
            return

        """
        По умолчанию сохраняем ресурс в статусе SUCCESS
        и храним в течении 2 недель (14 дней).
        """
        attributes = dict({
            'status': ctt.Status.SUCCESS,
            'ttl': 14,
        }, **kwargs)

        if not attributes['main']:
            del attributes['main']

        rel_path = abs_path.relative_to(self.task.path())
        resource = self.create_resource(
            relative_path=str(rel_path),
            resource_type=TRENDBOX_CI_REPORT_RESOURCE_BETA,
            description=attributes['title'],
            **attributes
        )

        if add_to_context:
            self.task.Context.report_resources.append(resource.id)

        return resource

    def create_log_resource(self, relative_path):
        """
        Create log resource.
        :param relative_path: Path to resource file or directory relative to the task root directory.
        :return: Resource descriptor instance.
        """
        return self.create_resource(
            relative_path=relative_path,
            resource_type=TASK_CUSTOM_LOGS,
            description='Trendbox job logs with lifecycle steps',
            mark_as_ready=False
        )

    def create_resource(self, relative_path, resource_type=None, description=None, mark_as_ready=True, **kwargs):
        """
        Create resource.
        :param relative_path: Path to resource file or directory relative to the task root directory.
        :param resource_type: Resource type class.
        :type resource_type: str|() -> sdk2.Resource
        :param description: Resource description.
        :type description: str
        :param mark_as_ready: Mark resource as ready.
        :type mark_as_ready: bool
        :param kwargs: Additional attributes for resource to create.
        :return: Resource descriptor instance.
        """

        resource_type = self.get_resource_type(resource_type)

        if description is None:
            description = relative_path

        res = resource_type(
            task=self.task,
            path=str(relative_path),
            description=description,
            **kwargs
        )

        # Конструктор ResourceData неявно добавляет текущий хост
        # в качестве источника для ресурсов наследников TaskLogs.
        # Это позволяет читать логи во время выполнения задачи.
        res_data = sdk2.ResourceData(res)
        if mark_as_ready:
            res_data.ready()

        return res

    def get_resource_type(self, resource_type_name):
        """
        :param resource_type_name: Resource type class.
        :type resource_type_name: str|() -> sdk2.Resource
        :return: Resource class instance.
        """

        if not resource_type_name:
            return TRENDBOX_CI_RESOURCE_BETA

        if type(resource_type_name) in (str, unicode):
            try:
                return sdk2.Resource[resource_type_name]
            except:
                # Костыль для динамического создания ресурсов, нужно для задач запущенных с бинарным архивом.
                # Подробности: FEI-12979
                logging.debug('could not create resource by name: {}, try to dynamic create', resource_type_name)

                return type(str(resource_type_name), (sdk2.Resource,), {'__default_attribute__': sdk2.parameters.String})

        return resource_type_name

    def get_resource_http_proxy(self, res_id):
        """
        :type res_id: int
        :rtype: str
        """
        res_data = self.task.server.resource[res_id].read()
        return res_data['http']['proxy']
