# -*- coding: utf-8 -*-

import logging
import pathlib2
import contextlib

logger = logging.getLogger(__name__)


@contextlib.contextmanager
def job_logs_placeholder(logs_dir, placeholder_name):
    """
    Перед запуском создаём временный файл, объясняющий содержимое директории на случай,
    когда trendbox run-job не дойдёт до выполнения пользовательского кода и логи будут пустыми.
    Иначе Sandbox откажется публиковать пустой ресурс и упадёт с исключением.
    См. FEI-12187.
    :param logs_dir:
    :type logs_dir: str
    :param placeholder_name:
    :type placeholder_name: str
    """
    logs_dir = pathlib2.Path(logs_dir)
    placeholder_path = logs_dir.joinpath(placeholder_name)

    if not logs_dir.exists():
        logger.warn('Will not create logs placeholder at "{}": target directory not exists'.format(placeholder_path))

        # выполняем пользовательский код и выходим, чистить нечего
        yield
        return

    try:
        placeholder_body = u"""
If you're reading this, it's most likely that the job code hasn't been run.
The task may have stopped abnormally at the configuration reading stage.
Please see `run_job_cli.{err,out}.txt` logs in the `log1` directory.
""".strip()

        logger.debug('Creating logs placeholder at "{}"...'.format(placeholder_path))
        placeholder_path.write_text(placeholder_body)
    except Exception:
        logger.exception('Could not create logs placeholder at "{}", ignoring'.format(placeholder_path))

    yield

    if placeholder_path.exists():
        logger.debug('Removing logs placeholder from "{}"...'.format(placeholder_path))

        placeholder_path.unlink()
    else:
        logger.warn('Skip removing logs placeholder from "{}": not exists'.format(placeholder_path))
