# -*- coding: utf-8 -*-

from requests import codes, Session
from requests.adapters import HTTPAdapter
from requests.packages.urllib3.util.retry import Retry
from requests.exceptions import ChunkedEncodingError

NON_RETRY_STATUS_CODES = frozenset([400, 401, 403, 404, 405, 413, 414, 431, 422])
STATUS_CODES_4XX_5XX = frozenset([code for code in codes.__dict__.values() if type(code) is int and code >= 400])

# Коды статусов, при которых нужно пытаться повторять запрос
RETRY_STATUS_CODES = STATUS_CODES_4XX_5XX.difference(NON_RETRY_STATUS_CODES)

# Коэффициент экспоненциального нарастания задержки. При 0.2 и 4 повторах: 0s, 0.2s, 0.6s, 1.4s, 3s
BACKOFF_FACTOR = 0.2

# HTTP методы, при которых нужно пытаться повторять запрос
RETRY_METHODS = frozenset(['POST', 'HEAD', 'TRACE', 'GET', 'PUT', 'OPTIONS', 'DELETE'])


class RetryAdapter(HTTPAdapter):
    def __init__(self, max_retries=10, backoff_factor=BACKOFF_FACTOR, retry_status_codes=RETRY_STATUS_CODES, retry_methods=RETRY_METHODS):  # noqa
        """
        :param max_retries: максимальное количество перезапросов
        :type max_retries: int
        :param backoff_factor: коэффициент экспоненциального нарастания задержки
        :type backoff_factor: float
        :param retry_status_codes: коды статусов, при которых нужно пытаться повторять запрос
        :type retry_status_codes: frozenset
        :param retry_methods: методы, при которых нужно пытаться повторять запрос
        :type retry_methods: frozenset
        """
        super(RetryAdapter, self).__init__(max_retries=max_retries)
        self.max_retries = Retry(
            total=max_retries,
            backoff_factor=backoff_factor,
            status_forcelist=retry_status_codes,
            method_whitelist=retry_methods,
            raise_on_status=False,
            raise_on_redirect=False,
        )


def make_request(method, url, headers, max_retries, backoff_factor=BACKOFF_FACTOR,
    retry_status_codes=RETRY_STATUS_CODES, retry_methods=RETRY_METHODS):  # noqa
    """
    :param method: тип запроса (get, post, put, delete)
    :type method: str
    :param url: запрашиваемая ссылка
    :type url: str
    :param headers: заголовки запроса
    :type headers: dict
    :param max_retries: максимальное количество перезапросов
    :type max_retries: int
    :param backoff_factor: коэффициент экспоненциального нарастания задержки
    :type backoff_factor: float
    :param retry_status_codes: коды статусов, при которых нужно пытаться повторять запрос
    :type retry_status_codes: frozenset
    :param retry_methods: методы, при которых нужно пытаться повторять запрос
    :type retry_methods: frozenset
    :rtype: func
    """
    if headers is None:
        headers = {}

    retry_adapter = RetryAdapter(
        max_retries=max_retries,
        backoff_factor=backoff_factor,
        retry_status_codes=retry_status_codes,
        retry_methods=retry_methods,
    )
    session = Session()
    session.mount(url, retry_adapter)
    session.headers.update(headers)

    return getattr(session, method)


def send_request(method, url, headers=None, max_retries=4, make_request=make_request,
                 retry_methods=RETRY_METHODS, **kwargs):
    """
    Отсылает запрос с перезапросами в случае неудачи

    :param method: тип запроса (get, post, put, delete)
    :type method: str
    :param url: запрашиваемая ссылка
    :type url: str
    :param headers: заголовки запроса
    :type headers: dict
    :param max_retries: максимальное количество перезапросов
    :type max_retries: int
    :param make_request функция для отправки запроса
    :type make_request: func
    :param retry_methods: методы, при которых нужно пытаться повторять запрос
    :type retry_methods: frozenset
    :return: response object
    :rtype: requests.Response
    """
    request = make_request(method, url, headers, max_retries, retry_methods=retry_methods)

    # Соединение может быть закрыто во время чтения данных,
    # попробуем сделать еще один. Если не получилось два раза подряд:
    # что-то точно пошло не так, кидаем исключение.
    try:
        return request(url=url, **kwargs)
    except ChunkedEncodingError:
        return request(url=url, **kwargs)
