# coding=utf-8
import logging
import os

from sandbox import sdk2

import sandbox.common.types.task as ctt
from sandbox.common.utils import singleton_property

from sandbox.projects.trendbox_ci.beta.constants.dist import DIST_TAG
from sandbox.projects.trendbox_ci.beta import managers
from sandbox.projects.trendbox_ci.beta.modules.tar import unpack_tar
from sandbox.projects.trendbox_ci.beta.modules.parameters import LastResource
from sandbox.projects.trendbox_ci.beta.resources import TRENDBOX_CI_JOB_CLI_BETA


class TrendboxCiBaseTask(sdk2.Task):
    """Trendbox CI Base Task"""

    name = 'TRENDBOX_CI_BASE_BETA'

    class Requirements(sdk2.Requirements):
        class Caches(sdk2.Requirements.Caches):
            # Требования к кешу отключаем в рамках FEI-6856, так-как нам нужны MULTISLOT агенты для экономии
            # квоты и пока TrendBox не умеет работать с общим кешом.
            # @see https://wiki.yandex-team.ru/sandbox/cookbook/#cores1multislot
            pass

    class Context(sdk2.Context):
        subtasks_ids = []

    class Parameters(sdk2.Parameters):
        with sdk2.parameters.Group('Identifiers') as build_request_id_block:
            build_id = sdk2.parameters.String(
                'Build ID',
                hint=True,
            )
            build_request_id = sdk2.parameters.String(
                'Trigger Request ID'
            )
        with sdk2.parameters.Group('Webhook') as webhook_block:
            webhook_url = sdk2.parameters.Url(
                'Webhook url',
                description='URL to route of Sandbox adapter. Sandbox adapter listens webhooks from Sandbox Tasks and registers Trendbox CI events',
            )
            webhook_payload = sdk2.parameters.JSON(
                'Payload',
                description='Payload will be sent in webhook body',
            )

        with sdk2.parameters.Group('Service') as service_block:
            use_last_binary_archive = sdk2.parameters.Bool('Use last binary archive', default=True)
            use_last_released_binary_archive = sdk2.parameters.Bool('Use last released binary archive', default=True)

        with sdk2.parameters.Group('Trendbox CLI package') as trendbox_cli_block:
            trendbox_cli_resource = LastResource(
                'Trendbox CLI resource',
                description='Resource with Trendbox CLI',
                resource_type=TRENDBOX_CI_JOB_CLI_BETA,
                required=True,
                owner='TRENDBOX_CI_TEAM',
                attrs={'tag': DIST_TAG},
            )

    @singleton_property
    def webhook(self):
        """
        :rtype: sandbox.projects.trendbox_ci.beta.managers.WebHookManager
        """
        return managers.WebHookManager(self)

    @singleton_property
    def reports(self):
        """
        :rtype: sandbox.projects.trendbox_ci.beta.managers.ReportsManager
        """
        return managers.ReportsManager(self)

    @singleton_property
    def zeroline_reporter(self):
        """
        :rtype: sandbox.projects.trendbox_ci.beta.managers.ZerolineReporter
        """
        return managers.ZerolineReporter()

    def send_webhook(self, status=None):
        """
        Send webhook with task info to specified URL.

        :type status: str
        """
        webhook_url = self.Parameters.webhook_url

        if status is None:
            status = self.status

        if webhook_url:
            url = self.Parameters.webhook_url
            self.webhook.send(urls=(url,), status=status)

    def set_task_archive_resource(self):
        if self.Parameters.use_last_released_binary_archive:
            self.Requirements.tasks_resource = self.find_last_released_sandbox_task_binary()

            return None

        if self.Parameters.use_last_binary_archive:
            self.Requirements.tasks_resource = self.find_last_sandbox_task_binary()

    def find_last_sandbox_task_binary(self):
        owner = 'TRENDBOX_CI_TEAM'
        attrs = {
            'project_dist_tag': DIST_TAG,
            'project': 'trendbox_ci'
        }

        return sdk2.service_resources.SandboxTasksBinary.find(attrs=attrs, owner=owner).first()

    def find_last_released_sandbox_task_binary(self):
        owner = 'TRENDBOX_CI_TEAM'
        attrs = {
            'project_dist_tag': DIST_TAG,
            'project': 'trendbox_ci',
            'released': ctt.ReleaseStatus.STABLE
        }

        return sdk2.service_resources.SandboxTasksBinary.find(attrs=attrs, owner=owner).first()

    def install_job_cli(self):
        """
        Install job cli package from sandbox resource
        """
        logging.debug('Installing trendjob cli for dist "{}"'.format(DIST_TAG))

        resource = self.Parameters.trendbox_cli_resource

        if resource is None:
            raise Exception('Could not find job cli resource with tag {dist} '.format(dist=DIST_TAG))

        os.mkdir(str(self._trendjob_modules_path))

        unpack_tar(
            sdk2.ResourceData(resource).path,
            self._trendjob_modules_path
        )

        logging.debug(
            'Installed trendjob cli to "{}"'.format(self._job_cli_path))

    @property
    def _job_cli_path(self):
        """
        :return: path to job cli bin
        :rtype: pathlib2.Path
        """
        return self._trendjob_modules_path.joinpath(
            self.Parameters.trendbox_cli_resource.bin_path
        )

    @property
    def _working_path(self):
        """
        :return: path to working directory
        :rtype: pathlib2.Path
        """
        if self._is_ramdrive_supported:
            return self.ramdrive.path
        return self.path()

    @property
    def arc_mount_path(self):
        return self._working_path.joinpath('arc_mount')

    @property
    def arc_store_path(self):
        return self._working_path.joinpath('arc_store')

    @property
    def arc_object_store_path(self):
        return self._working_path.joinpath('arc_object_store')

    @property
    def _is_ramdrive_supported(self):
        """
        :return: True if ramdrive will be used
        :rtype: bool
        """
        return self.ramdrive and self.ramdrive.path

    @property
    def _trendjob_modules_path(self):
        """
        :return: path to packages
        :rtype: pathlib2.Path
        """
        return self._working_path.joinpath('trendjob_modules')

    def on_enqueue(self):
        self.send_webhook(status=ctt.Status.ENQUEUING)

        super(TrendboxCiBaseTask, self).on_enqueue()

    def on_prepare(self):
        self.send_webhook(status=ctt.Status.PREPARING)
        self.install_job_cli()

        super(TrendboxCiBaseTask, self).on_prepare()

    def on_execute(self):
        self.send_webhook(status=ctt.Status.EXECUTING)

        super(TrendboxCiBaseTask, self).on_execute()

    def on_wait(self, prev_status, status):
        self.send_webhook(status=status)

        super(TrendboxCiBaseTask, self).on_wait(prev_status, status)

    def on_success(self, prev_status):
        self.send_webhook(status=ctt.Status.SUCCESS)

        super(TrendboxCiBaseTask, self).on_success(prev_status)

    def on_break(self, prev_status, status):
        self.send_webhook(status=status)

        super(TrendboxCiBaseTask, self).on_break(prev_status, status)

    def on_failure(self, prev_status):
        self.send_webhook(status=ctt.Status.FAILURE)

        super(TrendboxCiBaseTask, self).on_failure(prev_status)
