# -*- coding: utf-8 -*-
import logging
import os
import requests
import json

from sandbox import sdk2
from sandbox.common.utils import singleton, singleton_property
from sandbox.common.errors import TaskFailure
import sandbox.common.types.task as ctt

from sandbox.projects.sandbox.sandbox_lxc_image import SandboxLxcImage
from sandbox.projects.trendbox_ci.beta import managers


# 30 минут
WAIT_TIMEOUT = 30 * 60
TRENDBOX_CI_GITHUB_RAW_PATH = 'https://raw.github.yandex-team.ru/search-interfaces/trendbox-ci'


class TrendboxCiLXC(sdk2.Task):
    """Trendbox CI build LXC image"""

    name = 'TRENDBOX_CI_LXC_BETA'

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group('LXC') as lxc_group:
            with sdk2.parameters.CheckGroup(
                'Components',
                description='''
                    Components include into LXC (e.g. Node.js, Docker).
                    Trendbox core which is needed to run Trendbox itself is always included.
                '''
            ) as components:
                components.values['node_js'] = 'Node.js'
                components.values['docker'] = 'Docker'
                components.values['git_lfs'] = 'Git LFS'

            node_js_versions = sdk2.parameters.JSON(
                'Pre-install Node.js versions',
                description='e.g. [{ "node_js": "8.17.0", "yarn": "1.22.4" }, { "node_js": "12.13.0", "npm": "6.14.6" }]',
            )

            with sdk2.parameters.String('Ubuntu release', default='trusty') as ubuntu_release:
                ubuntu_release.values['trusty'] = 'trusty'
                ubuntu_release.values['xenial'] = 'xenial'

            custom_attrs = sdk2.parameters.Dict('Attributes to add to resulting resource')
            custom_script = sdk2.parameters.String('Shell script to execute during final stage', multiline=True)

        with sdk2.parameters.Group('Utils') as utils_group:
            trendbox_github_branch = sdk2.parameters.String(
                'Trendbox branch',
                description='GitHub branch at trendbox-ci repo to download ansible scripts from',
                default='master'
            )

    class Context(sdk2.Context):
        subtasks_ids = []
        noncritical_errors = []

    def get_image_components(self):
        """
        Optional components are explicitly sorted to produce repeatable resource search results by attributes.
        Otherwise they may come in order of UI clicks.

        `base` component should always come first because others have implicit dependencies on it.
        """
        return ['base'] + sorted(self.Parameters.components)

    @singleton_property
    def meta(self):
        return managers.MetaTaskManager(self)

    def get_build_script_template(self):
        """
        Returns template of build script
        :rtype: str
        """
        current_dir = os.path.dirname(os.path.realpath(__file__))

        with open(os.path.join(current_dir, 'build.template.sh')) as f:
            return f.read()

    def apply_template_build_script(self, template):
        """
        Returns result build script from build.sh
        """
        return template.replace(
            '__PLACE_FOR_ANSIBLE_SCRIPT__', self.get_ansible_script()
        ).replace(
            '__PLACE_FOR_TRENDBOX_BRANCH__', self.Parameters.trendbox_github_branch
        ).replace(
            '__PLACE_FOR_EXTRA_VARS__', self.get_ansible_extra_vars()
        )

    @singleton
    def get_ansible_script(self):
        """
        Returns ansible script built from base and all selected components
        :rtype: str
        """
        components = self.get_image_components()

        return '\n'.join([
            self.get_ansible_script_from_github(component) for component in components
        ])

    @singleton
    def get_ansible_extra_vars(self):
        return json.dumps(dict(
            node_js_versions=(self.Parameters.node_js_versions or []),
        )).replace("'", "\\'")

    def get_ansible_script_from_github(self, component):
        """
        Requests specified component's yml from Trendbox CI repo
        :param component: one of self.Parameters.components values
        :type component: str
        :rtype: str
        """
        branch = self.Parameters.trendbox_github_branch
        url = '{}/{}/containers/sandbox-cloud/lxc/trendbox-ci_{}.yml'.format(TRENDBOX_CI_GITHUB_RAW_PATH, branch, component)

        logging.debug('trying to get "{}" from GitHub'.format(url))

        res = requests.get(url)

        if res.status_code != 200:
            raise TaskFailure('Could not fetch "{}" from GitHub, code: {}, response: {}'.format(url, res.status_code, res.text))

        return res.text

    def get_custom_script(self):
        """
        Returns value of custom_script parameter for SANDBOX_LXC_IMAGE
        :rtype: str
        """
        return '\n'.join([
            self.apply_template_build_script(self.get_build_script_template()),
            self.Parameters.custom_script,
        ])

    def get_custom_attrs(self):
        """
        Returns value of custom_attrs parameter for SANDBOX_LXC_IMAGE
        :rtype: dict
        """
        image_components = self.get_image_components()
        attrs = {
            'owner': self.Parameters.owner,
            'ubuntu': self.Parameters.ubuntu_release,
            'components': ','.join(image_components),
            'node_js_versions': ','.join(map(lambda node_version: node_version['node_js'], self.Parameters.node_js_versions or []))
        }
        attrs.update(self.Parameters.custom_attrs)

        return attrs

    def on_execute(self):
        with self.memoize_stage.wait_build_task():
            self.meta.start_subtasks([self.meta.create_subtask(
                SandboxLxcImage,
                resource_type='TRENDBOX_CI_LXC_IMAGE_BETA',
                description=self.Parameters.description,
                custom_image=True,
                custom_script=self.get_custom_script(),
                custom_attrs=self.get_custom_attrs(),
                ubuntu_release=self.Parameters.ubuntu_release,
                test_result_lxc=False,
            )])

            raise sdk2.WaitTask(
                self.Context.subtasks_ids,
                ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
                timeout=max(self.Parameters.kill_timeout, WAIT_TIMEOUT),
            )

        with self.memoize_stage.check_build_task():
            if sdk2.Task[self.Context.subtasks_ids[0]].status != ctt.Status.SUCCESS:
                raise TaskFailure('LXC Build Failed')
