# -*- coding: utf-8 -*-

import logging

from sandbox import sdk2
import sandbox.common.types.task as ctt

from sandbox.projects.resource_types import TASK_CUSTOM_LOGS
from sandbox.projects.trendbox_ci.stable.resources import TRENDBOX_CI_REPORT_RESOURCE, TRENDBOX_CI_RESOURCE


class ResourcesManager(object):
    """
    Manager that helps to work with resources.
    """

    def __init__(self, task):
        """
        :param task: Task instance.
        """
        self.task = task

    def create_report(self, resource_path=None, add_to_context=False, **kwargs):
        """
        :param resource_path: path to resource relative to project or absolute path to resource
        :param add_to_context: appends generated resource id to `self.task.Context.report_resources`
        :param kwargs: additional attributes of created resource
        :return: Resource descriptor instance
        """
        # if resource_path is absolute, abs_path will become resource_path
        abs_path = self.task.path() / resource_path

        if not abs_path.exists():
            logging.debug('Not adding resource {}: no such path'.format(abs_path))
            return

        # sandbox fails to create resource with empty directory
        if abs_path.is_dir() and not any(abs_path.iterdir()):
            logging.debug('Not adding resource {}: directory is empty'.format(abs_path))
            return

        """
        По умолчанию сохраняем ресурс в статусе SUCCESS
        и храним в течении 2 недель (14 дней)
        """
        attributes = dict({
            'status': ctt.Status.SUCCESS,
            'ttl': 14,
        }, **kwargs)

        rel_path = abs_path.relative_to(self.task.path())
        resource = self.create_resource(
            relative_path=str(rel_path),
            resource_type=TRENDBOX_CI_REPORT_RESOURCE,
            **attributes
        )

        if add_to_context:
            self.task.Context.report_resources.append(resource.id)

        return resource

    def create_log_resource(self, relative_path, **kwargs):
        """
        Create log resource. `checksum` attribute will be treated special way and go to the resource description.
        :param relative_path: Path to resource file or directory relative to the task root directory.
        :param kwargs: Additional attributes for resource to create.
        :return: Resource descriptor instance.
        """
        return self.create_resource(relative_path, TASK_CUSTOM_LOGS, **kwargs)

    def create_resource(self, relative_path, resource_type=TRENDBOX_CI_RESOURCE, **kwargs):
        """
        Create resource. `checksum` attribute will be treated special way and go to the resource description.
        :param relative_path: Path to resource file or directory relative to the task root directory.
        :param resource_type: Resource type class.
        :type resource_type: str|() -> sdk2.Resource
        :param kwargs: Additional attributes for resource to create.
        :return: Resource descriptor instance.
        """

        if type(resource_type) in (str, unicode):
            resource_type = sdk2.Resource[resource_type]

        checksum = kwargs.get('checksum')
        kwargs.update(path=str(relative_path))
        res = resource_type(
            task=self.task,
            description='{}#{}'.format(relative_path, checksum) if checksum else relative_path,
            **kwargs
        )

        res_data = sdk2.ResourceData(res)
        res_data.ready()

        return res

    def get_resource_http_proxy(self, res_id):
        """
        :type res_id: int
        :rtype: str
        """
        res_data = self.task.server.resource[res_id].read()
        return res_data['http']['proxy']
