# -*- coding: utf-8 -*-

import logging

from sandbox import sdk2

VAULTS_LIMIT = 100


class VaultManager(object):
    """
    Manager that helps to work with vault.
    """

    def __init__(self, task):
        """
        :param task: Task instance.
        :type task: sdk2.Task
        """
        self.task = task

    def get_all_vault_env(self):
        """
        :return: dict of environments
        :rtype: dict
        """
        logging.debug('reading env variables from vault')

        owner = self.task.owner

        vault_items = self.read_all_vault(owner)
        env_vault_items = filter(self._is_env, vault_items)

        vault_env_dict = {}
        for item in env_vault_items:
            vault_env_dict[self._strip_env(item)] = self.read_vault(item['name'], owner)

        return vault_env_dict

    def read_vault(self, name, owner):
        """
        :param name: Vault name.
        :type name: str
        :param owner: Owner task instance
        :type owner: str
        :return: vault value
        :rtype: str
        """
        return sdk2.Vault.data(owner, name)

    def read_all_vault(self, owner):
        """
        :param owner: Owner task instance
        :type owner: str
        :return: list of vault item
        :rtype: list of dict
        """
        logging.debug('reading everything from {} vault'.format(owner))

        total = 1
        offset = 0
        items = []

        while total > offset:
            response = self.task.server.vault.read(owner=owner, offset=offset, limit=VAULTS_LIMIT)
            total = response.get('total', 0)
            items += response.get('items', [])
            offset += len(items)

        logging.debug('read {items_count} records from {owner} vault'.format(owner=owner, items_count=len(items)))

        return items

    def _is_env(self, vault_item):
        """
        :param vault_item: Vault item
        :type vault_item: dict
        :return: true if vault_item is environment variable
        :rtype: bool
        """
        return vault_item['name'].startswith('env.')

    def _strip_env(self, vault_item):
        """
        Сut off the .env at the beginning of the string
        :param vault_item: Vault item
        :type vault_item: dict
        :return: environment name
        :rtype: str
        """
        return vault_item['name'][4:]
