# -*- coding: utf-8 -*-
import os
import shlex
import logging

from sandbox.sdk2.helpers import ProcessLog, subprocess
from sandbox.common.errors import TaskFailure

INNER_EXCEPTION_RETURN_CODE = 125


class SubprocessError(TaskFailure):
    """ Error working with subprocess """
    task_context_field = 'subprocess_error'

    def __init__(
        self, process_name='', cmd=None, returncode=None,
        logs_resource=None, log_path=None,
        stdout_path=None, stderr_path=None,
    ):
        message = self._build_error_message(process_name, returncode)
        super(SubprocessError, self).__init__(message)

        self.process_name = process_name
        self.returncode = returncode
        self.cmd = cmd
        self.logs_resource = logs_resource
        self.log_path = log_path
        self.stdout_path = stdout_path
        self.stderr_path = stderr_path

    def _build_error_message(self, process_name, returncode):
        """
        :type process_name: str
        :type returncode: int
        :rtype: str
        """
        return 'subprocess "{process}" exit with code {returncode}'.format(
            process=process_name,
            returncode=returncode
        )

    def get_task_context_value(self):
        """
            Returns dict to add exeption info to task context
        """
        return {
            'error': self.message,
            'cmd': self.cmd,
            'returncode': self.returncode,
            'resource_id': self.logs_resource.id,
            'stdout': str(self.stdout_path),
            'stderr': str(self.stderr_path),
        }

    def get_task_info(self):
        """
            Returns description to task section "info"
        """
        stdout_link = self._format_log_html_link('stdout', str(self.stdout_path))
        stderr_link = self._format_log_html_link('stderr', str(self.stderr_path))
        log_links = filter(bool, [stdout_link, stderr_link])

        return u'🔴 Subprocess <b>{process}</b> exit with code {returncode}, see logs: {log_links}'.format(
            process=self.process_name,
            returncode=self.returncode,
            log_links=', '.join(log_links)
        )

    def _format_log_html_link(self, log_name, file_name):
        """
        :type log_name: str
        :type file_name: str
        :return: html link
        :rtype: str
        """
        if self._is_not_empty_log(file_name):
            log_link = '{}/{}'.format(self.logs_resource.http_proxy, file_name)
            return self._format_html_link(log_name, log_link)

    def _is_not_empty_log(self, file_name):
        """
        :type file_name: str
        :rtype: bool
        """
        file_path = str(self.log_path.joinpath(file_name))

        return os.path.exists(file_path) and os.path.getsize(file_path) > 0

    def _format_html_link(self, text, link):
        """
        :type text: str
        :type link: str
        :return: html link
        :rtype: str
        """
        return '<a href="{link}">{text}</a>'.format(text=text, link=link)


def run_process(task, process_name, cmd, **kwargs):
    """
    Runs subprocess, write logs to task log resource.

    :param task: Task instance
    :type task: sandbox.sdk2.Task
    :param process_name: process process_name
    :type process_name: str
    :param cmd: command to run in process
    :type cmd: str
    :return: returncode
    :rtype: int
    """
    logging.debug(u'Running "{process}" process: {cmd}'.format(process=process_name, cmd=cmd))

    if isinstance(cmd, unicode):
        cmd = cmd.encode('utf-8')

    args = shlex.split(cmd)
    returncode = INNER_EXCEPTION_RETURN_CODE
    log_files = {}

    with ProcessLog(task, logger=process_name) as process_log:
        returncode = subprocess.Popen(args, stdout=process_log.stdout, stderr=process_log.stderr, **kwargs).wait()
        log_files = _get_log_files(task, process_log)

    if returncode > 0:
        raise SubprocessError(
            process_name=process_name,
            cmd=cmd,
            returncode=returncode,
            stdout_path=log_files['stdout'],
            stderr_path=log_files['stderr'],
            logs_resource=task.log_resource,
            log_path=task.log_path(),
        )

    return returncode


def _get_log_files(task, process_log):
    """
    :type task: sandbox.sdk2.Task
    :type process_log: sandbox.sdk2.helpers.ProcessLog
    :rtype: dict
    """
    log_path = task.log_path()

    return {
        'stdout': os.path.relpath(str(process_log.stdout.path), start=str(log_path)),
        'stderr': os.path.relpath(str(process_log.stderr.path), start=str(log_path)),
    }
