# -*- coding: utf-8 -*-
import json
import logging
import os

from sandbox import sdk2
from sandbox.common import config
import sandbox.common.types.misc as ctm
from sandbox.common.types import task as ctt
from sandbox.common.types import notification as notify_types
from sandbox.common.errors import TaskFailure
from sandbox.sandboxsdk import environments

from sandbox.projects.trendbox_ci.stable.modules.process import run_process, INNER_EXCEPTION_RETURN_CODE
from sandbox.projects.trendbox_ci.stable.modules.schedulers import is_scheduler_allowed_to_run, is_task_able_to_run
from sandbox.projects.trendbox_ci.stable.managers.resources import ResourcesManager
from sandbox.projects.trendbox_ci.stable.managers.vault import VaultManager
from sandbox.projects.trendbox_ci.stable.resources import TRENDBOX_CI_LXC_IMAGE, TRENDBOX_CI_RESOURCE


def escape_json_str(json_str):
    """
    Escapes json string to run in subprocess.

    :type json_str: str
    :rtype: str
    """
    return u'{}'.format(json_str).replace("'", "'\\''")


class TrendboxJobException(Exception):
    pass


class TrendboxCiJob(sdk2.Task):
    name = 'TRENDBOX_CI_JOB'

    class Requirements(sdk2.Requirements):
        dns = ctm.DnsType.DNS64
        disk_space = 5 * 1024
        environments = (
            environments.NodeJS('8.4.0'),
        )

    class Parameters(sdk2.Parameters):
        with sdk2.parameters.Group('Container') as container_block:
            _container = sdk2.parameters.Container(
                'LXC resource',
                description='Commands of lifecycle will be launched in specified Linux Container',
                resource_type=TRENDBOX_CI_LXC_IMAGE,
                platform='linux_ubuntu_14.04_trusty',
                required=True,
            )

        with sdk2.parameters.Group('Trendbox CLI package') as trendbox_cli_block:
            trendbox_source = sdk2.parameters.String(
                'Package name',
                description='NPM package name with Trendbox CLI',
                default='@yandex-int/trendbox-ci.sandbox-job-cli',
                required=True,
            )
            trendbox_version = sdk2.parameters.String(
                'Package version',
                description='NPM package version of Trendbox CLI',
                default='stable',
                required=True,
            )
            trendbox_registry = sdk2.parameters.String(
                'NPM registry',
                description='URL to NPM registry',
                default='http://npm.yandex-team.ru',
                required=True,
            )
            trendbox_cli = sdk2.parameters.String(
                'bin',
                description='Path to bin of Trendbox CLI',
                default='trendbox',
                required=True,
            )

        with sdk2.parameters.Group('Config') as trendbox_block:
            trendbox_config = sdk2.parameters.String(
                'Job configuration',
                description='JSON with job info (lifecycle steps, environments, etc)',
                required=True,
                multiline=True,
            )

        with sdk2.parameters.Output():
            with sdk2.parameters.Group('Job API') as public_api:
                logs_urls = sdk2.parameters.String('Logs urls', description='URLs to log files of lifecycle steps')
                resources = sdk2.parameters.String('Resources', description='Resources meta info')
                reports = sdk2.parameters.String('Reports', description='Report resources meta info')
                exit_code = sdk2.parameters.String('Exit code')

    def on_prepare(self):
        if self.author != 'robot-trendbot':
            notify_body=u"""Привет, это письмо отправлено автоматически автору таски #{id}.
            После 30 июня таски TRENDBOX_CI_JOB, созданные не из CI, перестали запускаться. Подробнее в посте — https://clubs.at.yandex-team.ru/trendbox-ci/396
            Тикет, в котором мы помогаем переезжать: https://st.yandex-team.ru/TRENDBOX-148""".format(id=self.id)

            # https://docs.yandex-team.ru/sandbox/dev/notification#email
            self.server.notification(
                subject='Задачи TRENDBOX_CI_JOB перестали запускаться после 30 июня',
                body=notify_body,
                recipients=[self.author],
                transport=notify_types.Transport.EMAIL,
                urgent=True  # send mail from sandbox-urgent@ instead of sandbox-noreply@
            )

            # Проверяем таску по вайтлисту, если в вайтлисте, то после проверим, не новый ли это планировщик
            if not is_task_able_to_run(self):
                raise Exception("Trendbox CI закрывается, запуски таски не из CI запрещены. Подробнее — https://clubs.at.yandex-team.ru/trendbox-ci/308.")

        if self.scheduler and not is_scheduler_allowed_to_run(self):
            raise Exception("Trendbox CI закрывается, новые планировщики запрещены. Подробнее — https://clubs.at.yandex-team.ru/trendbox-ci/308")

        logging.debug('Prepared working directory: {}'.format(self._working_path))

        if self._is_ramdrive_supported:
            logging.debug('Working directory mounted in ramdrive: {}'.format(self.ramdrive.path))

        logging.info('Node.js installed to "{}"'.format(self._node_path))
        logging.info('npm installed to "{}"'.format(self._npm_path))

        self.install_job_cli()

    @property
    def _working_path(self):
        """
        :return: path to working directory
        :rtype: pathlib2.Path
        """
        if self._is_ramdrive_supported:
            return self.ramdrive.path
        return self.path()

    @property
    def _is_ramdrive_supported(self):
        """
        :return: True if ramdrive will be used
        :rtype: bool
        """
        return self.ramdrive and self.ramdrive.path

    @property
    def _node_environment(self):
        """
        :rtype: sandbox.sandboxsdk.environments.NodeJs
        """
        return self.Requirements.environments[0]

    @property
    def _node_path(self):
        """
        :return: path to Node.js bin
        :rtype: pathlib2.Path
        """
        return self._node_environment.node

    @property
    def _npm_path(self):
        """
        :return: path to npm bin
        :rtype: pathlib2.Path
        """
        return self._node_environment.npm

    @property
    def _node_modules_path(self):
        """
        :return: path to node_modules
        :rtype: pathlib2.Path
        """
        return self._working_path.joinpath('node_modules')

    def install_job_cli(self):
        """
        Install npm package with job cli
        """
        logging.debug('Installing trendjob cli')

        self._run_process(
            process_name='install_job_cli',
            cmd='{npm} install {source}@{version} --registry {registry}'.format(
                npm=self._npm_path,
                source=self.Parameters.trendbox_source,
                version=self.Parameters.trendbox_version,
                registry=self.Parameters.trendbox_registry,
            )
        )

        logging.debug('Installed trendjob cli to "{}"'.format(self._job_cli_path))

    def _run_process(self, *args, **kwargs):
        """
        Runs subprocess, write logs to task log resource.

        :return: exit code
        :rtype: int
        """
        return run_process(task=self, *args, **kwargs)

    @property
    def _job_cli_path(self):
        """
        :return: path to job cli bin
        :rtype: pathlib2.Path
        """
        return self._node_modules_path.joinpath('.bin', self.Parameters.trendbox_cli)

    def on_execute(self):
        self.init_environ()
        self.run_job_cli()

        metainfo = self.process_job_meta()

        exit_code = self._get_exit_code(metainfo)
        if exit_code > 0:
            raise TaskFailure('Job lifecycle exit with code {}'.format(exit_code))

    def init_environ(self):
        """
        Init environment variables for job
        """
        logging.debug('Initing environ')

        os.environ['CI'] = 'true'
        os.environ['TRENDBOX_CI'] = 'true'
        os.environ['TRENDBOX_JOB_HOST'] = config.Registry().this.id
        os.environ['TRENDBOX_JOB_INTERNAL_NODE_PATH'] = str(self._node_path)
        os.environ['TRENDBOX_JOB_INTERNAL_NODE_MODULES_PATH'] = str(self._node_modules_path)
        os.environ['PATH'] = '{}:{}'.format(str(self._node_modules_path.joinpath('.bin')), os.environ['PATH'])
        os.environ['SANDBOX_TASK_ID'] = str(self.id)
        os.environ['SANDBOX_SESSION_TOKEN'] = str(self.agentr.token)
        os.environ['SANDBOX_SYNCHROPHAZOTRON_PATH'] = str(self.synchrophazotron)

        logging.debug('Inited environ: {}'.format(os.environ))

    def run_job_cli(self):
        """
        Run job cli
        """
        logging.debug('Running job cli')

        vault_manager = VaultManager(self)
        vault_envs = vault_manager.get_all_vault_env().items()

        self._create_log_dir()
        self._run_process(
            process_name='run_job_cli',
            cmd=u'{trendbox} run-job --config \'{config}\' --log-dir={log_dir} --meta-file={meta_file} --silent-lifecycle'.format(
                trendbox=self._job_cli_path,
                config=escape_json_str(self.Parameters.trendbox_config),
                log_dir=self._get_log_dir(),
                meta_file=self._get_meta_file_path(),
            ),
            env=dict(os.environ.items() + vault_envs),
        )

        logging.debug('Finished job cli')

    def _create_log_dir(self):
        self._get_log_dir().mkdir(mode=0o777)

    def _check_log_dir_empty(self):
        return len(list(self._get_log_dir().iterdir())) > 0

    def _get_log_dir(self):
        return self._working_path.joinpath('job_logs')

    def _get_meta_file_path(self):
        return self._working_path.joinpath('job_meta.json')

    def process_job_meta(self):
        resource_manager = ResourcesManager(self)

        meta = self._read_job_meta()
        exit_code = self._get_exit_code(meta)
        self._output_meta(meta)

        logs = meta.get('logs')
        resources = meta.get('resources')
        reports = meta.get('reports')

        logs_output = []
        resources_output = []
        reports_output = []

        if not logs:
            raise TrendboxJobException('Logs not found in meta file')

        if not self._check_log_dir_empty():
            raise TrendboxJobException('Logs dir is empty')

        res = resource_manager.create_log_resource(str(self._get_log_dir()))
        res_url = resource_manager.get_resource_http_proxy(res.id)
        for log in logs:
            logs_output.append('{}/{}'.format(res_url, os.path.basename(log)))

        for r in resources:
            r_attrs = r['attrs'] or {}
            res = resource_manager.create_resource(
                relative_path=r['path'],
                resource_type=r['type'] or TRENDBOX_CI_RESOURCE,
                **r_attrs
            )
            if res:
                resources_output.append(dict(
                    id=res.id,
                    url=resource_manager.get_resource_http_proxy(res.id),
                    name=r['name'],
                    attrs=r_attrs,
                ))

        for r in reports:
            res_attrs = r['attrs']
            res_status = ctt.Status.SUCCESS if exit_code == 0 else ctt.Status.FAILURE
            res_attrs['status'] = res_status
            res = resource_manager.create_report(r['path'], **r['attrs'])
            if res:
                reports_output.append(dict(
                    status=res_status,
                    url=resource_manager.get_resource_http_proxy(res.id),
                    id=res.id,
                    title=r['title'],
                    main=r['main'],
                ))

        self.Parameters.logs_urls = json.dumps(logs_output)
        self.Parameters.resources = json.dumps(resources_output)
        self.Parameters.reports = json.dumps(reports_output)

        return meta

    def _read_job_meta(self):
        meta_file = self._get_meta_file_path()

        if not meta_file.exists():
            raise TrendboxJobException('Job meta file does not exist')

        with meta_file.open() as f:
            data = json.load(f)
            if not data:
                raise TrendboxJobException('Job meta file is empty')

            logging.debug('Meta file content: {}'.format(data))

            return data

    def _get_exit_code(self, meta_content):
        return int(meta_content.get('exitCode', -1))

    def _output_meta(self, meta_content):
        """
        :type meta_content: dict
        """
        self.Parameters.exit_code = self._get_exit_code(meta_content)

    def on_finish(self, prev_status, status):
        self._fill_output()

        super(TrendboxCiJob, self).on_finish(prev_status, status)

    def on_break(self, prev_status, status):
        self._fill_output()

        super(TrendboxCiJob, self).on_break(prev_status, status)

    def _fill_output(self):
        """
        Fills empty output parameters
        """
        empty_json_str = json.dumps([])

        if self.Parameters.logs_urls == '':
            self.Parameters.logs_urls = empty_json_str

        if self.Parameters.resources == '':
            self.Parameters.resources = empty_json_str

        if self.Parameters.reports == '':
            self.Parameters.reports = empty_json_str

        if self.Parameters.exit_code == '':
            self.Parameters.exit_code = str(INNER_EXCEPTION_RETURN_CODE)
