# encoding: utf-8

from sandbox import sdk2
import datetime
import logging
import re

STARTREK_API_URL = "https://st-api.yandex-team.ru/"
DEFAULT_START_TIME = datetime.datetime(2020, 1, 1)
SPI_DATE_RE = re.compile(r"\[\d{4}\.\d{2}\.\d{2}\]")

YT_URL = "hahn.yt.yandex.net"
TABLE = "//home/trust/analytics/downtime/dtable"
FULL_TABLE = "//home/trust/analytics/downtime/full/fulltable"

INESSENTIAL_TICKET_TAGS = ("test", "monitoring", "prod_not_affected")


class Command(object):
    def execute(self):
        raise NotImplementedError()


class ExtractMetrics(Command):
    """
    Extracts Data about incidents from BALANCEINC and SPI queues in startrack.
    """

    def __init__(self, context):
        from startrek_client import Startrek

        self.st_client = Startrek(
            useragent="trust", base_url=STARTREK_API_URL, token=context["st_token"]
        )
        self.context = context
        start_time = context.get("start_time")
        if not start_time:
            start_time = DEFAULT_START_TIME.strftime("%Y-%m-%d")
        self.start_time = start_time

    @property
    def balanceinc_query(self):
        return 'Queue:BALANCEINC Components:trust Created: >="{}" "Sort By": Created'.format(  # noqa
            self.start_time
        )

    @property
    def spi_query(self):
        return 'Queue:SPI Components:trust Created: >="{}" "Sort By": Created'.format(
            self.start_time
        )

    @staticmethod
    def parse_datetime(value):
        dt = datetime.datetime.strptime(value[:-5], "%Y-%m-%dT%H:%M:%S.%f")
        return dt.strftime("%Y-%m-%d")

    @staticmethod
    def parse_is_broken(description):
        mandatory_text = (
            "TrustDowntime=",
            "TrustDowntime'=",
            "MoneyLoss_{trust}=",
            "MoneyLossNonRefundable_{trust}=",
        )
        for text in mandatory_text:
            if text not in description:
                return True
        return False

    def parse_kpi_metrics(self, description):
        downtime_re = re.compile(r"TrustDowntime=.*")
        downtime_derivative_re = re.compile(r"TrustDowntime'=.*")
        loss_re = re.compile(r"MoneyLoss_{trust}=.*")
        not_refund_loss_re = re.compile(r"MoneyLossNonRefundable_{trust}=.*")

        downtime = self.extract_value(downtime_re.findall(description))
        downtime_derivative = self.extract_value(
            downtime_derivative_re.findall(description)
        )
        loss = self.extract_value(loss_re.findall(description))
        not_refund_loss = self.extract_value(not_refund_loss_re.findall(description))
        metrics = [downtime, downtime_derivative, loss, not_refund_loss]
        return metrics

    @staticmethod
    def extract_value(input_value):
        result = 0.0
        if not input_value:
            return result
        number = re.findall(r"\d+[.,]?\d*", input_value[0])
        if not number:
            return result
        try:
            result = float(number[0].replace(",", "."))
        except (IndexError, ValueError, TypeError) as err:
            print(number, input_value, type(err))
        return result

    @classmethod
    def get_issue_date_from_title(cls, text):
        issue_date = None
        results = SPI_DATE_RE.findall(text)
        if results:
            issue_date = results[0].strip("[").strip("]").replace(".", "-")
        return issue_date

    @classmethod
    def choose_issue_date(cls, issue):
        issue_date = cls.get_issue_date_from_title(issue.summary)
        if not issue_date:
            issue_date = cls.parse_datetime(issue.createdAt)
        return issue_date

    @staticmethod
    def is_duplicated(issue):
        if "SPI" in issue.key:
            for link in issue.links:
                if "BALANCEINC" in link.object.key:
                    return True
        return False

    def execute(self):
        balanceinc_issues = self.st_client.issues.find(query=self.balanceinc_query)
        spi_issues = self.st_client.issues.find(query=self.spi_query)
        issues = list(balanceinc_issues) + list(spi_issues)
        result = []

        for issue in issues:
            date = self.choose_issue_date(issue)
            ticket = issue.key
            tags = ",".join(issue.tags)
            metrics = self.parse_kpi_metrics(issue.description)
            is_broken = self.parse_is_broken(issue.description)
            assignee = issue.assignee.login if issue.assignee else ""
            is_duplicated = self.is_duplicated(issue)
            row = {
                "date": date,
                "tags": tags,
                "ticket": ticket,
                "assignee": assignee,
                "downtime": metrics[0],
                "downtime_der": metrics[1],
                "loss": metrics[2],
                "not_refund_loss": metrics[3],
                "is_broken": is_broken,
                "is_duplicated": is_duplicated,
            }
            result.append(row)
        self.context["metrics"] = result
        return result


class UploadResultToStorage(Command):
    """
    Upload data about incidents into YT.
    """

    def __init__(self, context):
        import yt.wrapper as yt

        data = context.get("metrics", [])
        self.full_data = data
        self.filtered_data = self.filter_sensitive_data(data)
        self.yt = yt.YtClient(YT_URL, context["yt_token"])

    @classmethod
    def filter_sensitive_data(cls, full_data):
        sensitive_fields = {"assignee", "loss", "not_refund_loss", "warnings"}
        filtered_data = []
        for record in full_data:
            item = {
                field: value
                for field, value in record.items()
                if field not in sensitive_fields
            }
            filtered_data.append(item)
        return filtered_data

    def execute(self):
        if not self.yt.exists(TABLE):  # noqa
            schema = [
                {"name": "date", "type": "string"},
                {"name": "tags", "type": "string"},
                {"name": "ticket", "type": "string"},
                {"name": "downtime", "type": "double"},
                {"name": "downtime_der", "type": "double"},
            ]
            self.yt.create("table", TABLE, attributes={"schema": schema})  # noqa
        if not self.yt.exists(FULL_TABLE):  # noqa
            schema = [
                {"name": "date", "type": "string"},
                {"name": "tags", "type": "string"},
                {"name": "ticket", "type": "string"},
                {"name": "assignee", "type": "string"},
                {"name": "downtime", "type": "double"},
                {"name": "downtime_der", "type": "double"},
                {"name": "loss", "type": "double"},
                {"name": "not_refund_loss", "type": "double"},
                {"name": "warnings", "type": "string"},
            ]
            self.yt.create("table", FULL_TABLE, attributes={"schema": schema})  # noqa
        if not self.filtered_data:
            return
        self.yt.write_table(TABLE, self.filtered_data, raw=False)  # noqa
        if not self.full_data:
            return
        self.yt.write_table(FULL_TABLE, self.full_data, raw=False)  # noqa


class AnalyzeMetrics(Command):
    """
    Enrich data with warnings,
    """

    def __init__(self, context):
        self.data = context.get("metrics", [])

    @classmethod
    def enrich_with_warnings(cls, row):
        warnings = []
        if row["is_broken"]:
            warning = "Ticket doesn't have proper KPI metrics."
            warnings.append(warning)
            del row["is_broken"]
            return
        del row["is_broken"]  # Won't store in YT table

        for tag in INESSENTIAL_TICKET_TAGS:
            if tag not in row["tags"]:
                continue
            row["warnings"] = "\n".join(warnings)
            return
        if not any(
            (row["downtime"], row["downtime_der"], row["loss"], row["not_refund_loss"])
        ):
            warning = "Empty metrics."
            warnings.append(warning)
        elif not (bool(row["downtime"]) or bool(row["downtime_der"])):
            warning = "No downtime filled."
            warnings.append(warning)
        if bool(row["downtime"]) ^ bool(row["downtime_der"]):
            warning = "Only one downtime filled."
            warnings.append(warning)
        if row["loss"] < row["not_refund_loss"]:
            warning = "Loss is less, than NonRefundable Loss."
            warnings.append(warning)
        row["warnings"] = "\n".join(warnings)

    def execute(self):
        for row in self.data:
            self.enrich_with_warnings(row)


class HideDuplicates(Command):
    """
    Delete duplicated rows.
    """

    def __init__(self, context):
        self.context = context

    def execute(self):
        data = self.context.get("metrics", [])
        filtered_metrics = []
        for row in data:
            if row["is_duplicated"]:
                continue
            del row["is_duplicated"]
            filtered_metrics.append(row)
        self.context["metrics"] = filtered_metrics


class CalculateTrustDowntime(sdk2.Task):
    class Requirements(sdk2.Requirements):
        environments = [
            sdk2.environments.PipEnvironment("yandex-yt", version="0.9.34"),
            sdk2.environments.PipEnvironment("yandex-yt-yson-bindings"),
            sdk2.environments.PipEnvironment("yandex-yt-yson-bindings-skynet"),
            # See https://st.yandex-team.ru/DEVTOOLSSUPPORT-5219#5fd35214aa62c1459c71c34b for details
            sdk2.environments.PipEnvironment("startrek_client", version="2.5",
                                             custom_parameters=['--upgrade-strategy only-if-needed']),
            sdk2.environments.PipEnvironment("python-dateutil"),
        ]

    class Parameters(sdk2.Parameters):
        secret = sdk2.parameters.YavSecret(
            "Tokens for YT and startrek",
            required=True,
            description="Required keys: st_token, yt_token",
        )
        start_time = sdk2.parameters.String(
            "Start date", default=None, description="Example: 2020-06-01"
        )

    class Context(sdk2.Context):
        pass

    def on_execute(self):
        logging.info("Task: Start")
        execution_line = (
            ExtractMetrics,
            AnalyzeMetrics,
            HideDuplicates,
            UploadResultToStorage,
        )
        context = {
            "start_time": self.Parameters.start_time,
            "yt_token": self.Parameters.secret.data()["yt_token"],
            "st_token": self.Parameters.secret.data()["st_token"],
        }
        for cmd in execution_line:
            cmd(context).execute()
        logging.info("Task: End")
