# -*- coding: utf-8 -*-

import datetime
import logging
import time

import requests
import yaml

from sandbox import sdk2


SCHEDULE = [
    {
        'section': 'trust_payments_prod',
        'quota_key': 'get-payments-mediaservices',
        'condition': lambda current_hour: 6 <= current_hour < 23,
        'quota_value_if_condition_true': 80,
        'quota_value_if_condition_false': 160,
    },
    {
        'section': 'trust_payments_prod',
        'quota_key': 'post-orders-mediaservices',
        'condition': lambda current_hour: 6 <= current_hour < 23,
        'quota_value_if_condition_true': 50,
        'quota_value_if_condition_false': 100,
    },
    {
        'section': 'trust_payments_prod',
        'quota_key': 'post-payments-mediaservices',
        'condition': lambda current_hour: 6 <= current_hour < 23,
        'quota_value_if_condition_true': 55,
        'quota_value_if_condition_false': 110,
    },
    {
        'section': 'trust_payments_prod',
        'quota_key': 'post-payments-start-mediaservices',
        'condition': lambda current_hour: 6 <= current_hour < 23,
        'quota_value_if_condition_true': 50,
        'quota_value_if_condition_false': 100,
    },
    {
        'section': 'trust_payments_prod',
        'quota_key': 'post-topup-music-mediaservices-new',
        'condition': lambda current_hour: 6 <= current_hour < 23,
        'quota_value_if_condition_true': 2,
        'quota_value_if_condition_false': 4,
    },
    {
        'section': 'trust_payments_prod',
        'quota_key': 'post-topup-start-music-mediaservices-new',
        'condition': lambda current_hour: 6 <= current_hour < 23,
        'quota_value_if_condition_true': 2,
        'quota_value_if_condition_false': 4,
    },
    {
        'section': 'trust_payments_prod',
        'quota_key': 'post-topup-music-new',
        'condition': lambda current_hour: 6 <= current_hour < 23,
        'quota_value_if_condition_true': 2,
        'quota_value_if_condition_false': 4,
    },
    {
        'section': 'trust_payments_prod',
        'quota_key': 'post-topup-start-music-new',
        'condition': lambda current_hour: 6 <= current_hour < 23,
        'quota_value_if_condition_true': 2,
        'quota_value_if_condition_false': 4,
    },
    {
        'section': 'trust_payments_prod',
        'quota_key': 'get-payments-tickets',
        'condition': lambda current_hour: 6 <= current_hour < 23,
        'quota_value_if_condition_true': 41,
        'quota_value_if_condition_false': 80,
    },
    {
        'section': 'trust_payments_prod',
        'quota_key': 'post-topup-start-tickets',
        'condition': lambda current_hour: 6 <= current_hour < 23,
        'quota_value_if_condition_true': 5,
        'quota_value_if_condition_false': 10,
    },
    {
        'section': 'trust_payments_prod',
        'quota_key': 'post-topup-tickets',
        'condition': lambda current_hour: 6 <= current_hour < 23,
        'quota_value_if_condition_true': 2,
        'quota_value_if_condition_false': 4,
    },
    {
        'section': 'trust_payments_prod',
        'quota_key': 'post-refunds-start-tickets',
        'condition': lambda current_hour: 6 <= current_hour < 23,
        'quota_value_if_condition_true': 3,
        'quota_value_if_condition_false': 6,
    },
]


class OAuth(requests.auth.AuthBase):
    def __init__(self, token):
        self._header = 'OAuth ' + token

    def __call__(self, r):
        r.headers['Authorization'] = self._header
        return r


class RpsLimiterClient(object):
    base_url = 'https://rpslimiter.z.yandex-team.ru/api/rpslimiter.Model/'

    def __init__(self, installation, section, token):
        self.installation = installation
        self.section = section
        self.auth = OAuth(token)

    def _request(self, http_method, method, **kwargs):
        kwargs.setdefault('timeout', 60)
        kwargs['auth'] = self.auth
        url = self.base_url + method

        logging.info('%s %s', http_method.upper(), url)
        if 'params' in kwargs:
            logging.info('PARAMS: %s', kwargs['params'])
        if 'json' in kwargs:
            logging.info('JSON: %s', kwargs['json'])
        N = 5
        for i in range(N):
            try:
                response = requests.request(http_method, url, **kwargs)
                logging.info('http_code = %s, content length = %s', response.status_code, len(response.content))
                if 'x-req-id' in response.headers:
                    logging.info('Request id: "%s"', response.headers['x-req-id'])
                if response.headers.get('grpc-status', -1) != 0:
                    logging.info('Grpc status: %s', response.headers.get('grpc-status', -1))
                    logging.info('Message: %s', response.headers.get('grpc-message', ''))

                response.raise_for_status()
                result = response.json()
                if result.get('error'):
                    raise RuntimeError(result['error'])
                return result
            except:
                logging.exception('Request failed')
                if i + 1 == N:
                    raise
                logging.exception('Wait before next request')
                time.sleep(15)

    def get_quota(self, quota_name):
        result = self._request('POST', 'fetchConfig', json={'parent': self.installation, 'id': self.section})
        quota_configs = yaml.safe_load(result['content']['quotas'])
        return quota_configs[quota_name]['quota']

    def update_quota(self, quota_name, new_value, comment=None):
        result = self._request('POST', 'fetchConfig', json={'parent': self.installation, 'id': self.section})
        result.pop('author', None)
        result.pop('unixtimeMcs', None)
        quota_configs = yaml.safe_load(result['content']['quotas'])
        quota_configs[quota_name]['quota'] = new_value
        result['content']['quotas'] = yaml.dump(quota_configs)
        if comment is not None:
            result['content']['quotas'] = '# {}\n{}'.format(comment.strip(), result['content']['quotas'])

        self._request('POST', 'updateConfig', json=result)


class DynamicRpsLimiter(sdk2.Task):
    class Parameters(sdk2.Parameters):
        oauth_token_secret = sdk2.parameters.YavSecret('Yav secret for oauth token', required=True)
        oauth_token_key = sdk2.parameters.String('Name of record in secret', required=True)

        rps_limiter_installation = sdk2.parameters.String('Rps limiter installation', default='main', required=True)

    def process_schedule_item(self, schedule_item, token):
        rps_limiter_client = RpsLimiterClient(self.Parameters.rps_limiter_installation, schedule_item['section'], token)
        current_value = rps_limiter_client.get_quota(schedule_item['quota_key'])
        condition_func = schedule_item['condition']
        if condition_func(datetime.datetime.now().hour):
            target_value = schedule_item['quota_value_if_condition_true']
        else:
            target_value = schedule_item['quota_value_if_condition_false']
        if current_value != target_value:
            comment = 'Changed by DynamicRpsLimiter sandbox task with id: {} {}'.format(self.id, datetime.datetime.now())
            rps_limiter_client.update_quota(schedule_item['quota_key'], target_value, comment=comment)

    def on_execute(self):
        token = self.Parameters.oauth_token_secret.data()[self.Parameters.oauth_token_key]
        for i, schedule_item in enumerate(SCHEDULE):
            logging.info('Processing schedule item: %s', i)
            try:
                self.process_schedule_item(schedule_item, token)
            except Exception:
                logging.exception('Error while processing schedule item')
                self.set_info('Error while processing schedule items, see logs for more info')
