import logging
from sandbox import sdk2

import sandbox.common.errors as errors
from sandbox.common.types.resource import State
from sandbox.common.types.task import Status
from sandbox.projects.common import binary_task
from sandbox.projects.tank.load_resources.resources import YANDEX_TANKAPI_LXC_CONTAINER
from sandbox.projects.tank.ShootViaTankapi import ShootViaTankapi

logger = logging.getLogger(__name__)


class TrustAmmoResource(sdk2.Resource):
    """
    Ammo for Trust perf testing.
    """
    auto_backup = True
    releasable = True
    releasers = ["TRUST"]


class TrustScenariosPandoraGunExecutable(sdk2.Resource):
    """
    Executable of a trust/trust_load_gun/ for Trust perf testing.
    """
    auto_backup = True
    releasable = True
    releasers = ["TRUST"]


ALL_TANKS = {
    'man': [
        {
            'host': 'abrams.tanks.yandex.net',
            'port': '8083'
        },
        {
            'host': 'armata.tanks.yandex.net',
            'port': '8083'
        },
        {
            'host': 'comet.tanks.yandex.net',
            'port': '8083'
        },

    ],
    'sas': [
        {
            'host': 'hellcat.tanks.yandex.net',
            'port': '8083'
        },
        {
            'host': 'lee.tanks.yandex.net',
            'port': '8083'
        },
        {
            'host': 'scorpion.tanks.yandex.net',
            'port': '8083'
        },
        {
            'host': 'vickers.tanks.yandex.net',
            'port': '8083'
        },
        {
            'host': 'chaffee.tanks.yandex.net',
            'port': '8083'
        },
        {
            'host': 'stuart.tanks.yandex.net',
            'port': '8083'
        },
    ],
    'myt': [
        {
            'host': 'bumblebee.tanks.yandex.net',
            'port': '8083'
        },
        {
            'host': 'steam.tanks.yandex.net',
            'port': '8083'
        },
        {
            'host': 'buratino.tanks.yandex.net',
            'port': '8083'
        },
        {
            'host': 'kv1.tanks.yandex.net',
            'port': '8083'
        },
    ],
    'iva': [
        {
            'host': 'tulip.tanks.yandex.net',
            'port': '8083'
        },
        {
            'host': 'violet.tanks.yandex.net',
            'port': '8083'
        },
        {
            'host': 'matilda.tanks.yandex.net',
            'port': '8083'
        }
    ]
}


def link_format(href, text, description=''):
    return "{}<a href=\"{}\">{}</a>".format(description, href, text)


class StartTrustPerfTesting(binary_task.LastBinaryTaskRelease, sdk2.Task):
    """
    Task that starts the perfomance testing of Trust, which is a shooting to two targets (stable and testing) with
    a Yandex.Tank and comparing their results with lunapark compare.
    """
    class Requirements(sdk2.Task.Requirements):
        ram = 10 * 1024  # 10 Gb
        disk_space = 30 * 1024  # 10 Gb

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 5 * 60 * 60
        startrek_ticket_id = sdk2.parameters.String(
            "StarTrek ticket id.",
            required=False,
            description="StarTrek ticket id"
        )
        ammo_resource = sdk2.parameters.Resource(
            "Ammo resource.",
            resource_type=TrustAmmoResource,
            required=True,
            description="Ammo resource of type TRUST_AMMO_RESOURCE ."
        )
        gun_resource = sdk2.parameters.Resource(
            "Pandora gun executable resource. Leave it empty to use the last released:stable.",
            resource_type=TrustScenariosPandoraGunExecutable,
            required=False,
            description="Pandora gun executable resource. Leave it empty to use the last released:stable."
        )
        startup_load_profile = sdk2.parameters.String(
            "Tank startup load profile",
            required=True,
            default="[{type: once, times: 1}]",
            description="Tank startup load profile"
        )
        rps_load_profile = sdk2.parameters.String(
            "Tank rps load profile",
            required=True,
            default="[{type: const, duration: 10s, ops: 10}]",
            description="Tank rps load profile",
        )
        autostop_criteria = sdk2.parameters.String(
            "Autostop condition",
            required=True,
            default="['http(4xx, 10%, 60s)']",
            description="Autostop condition",
        )
        trust_url = sdk2.parameters.String(
            "Url for TRUST(host:port)",
            description="Target to use as Trust backend",
            default_value='http://uao52ote22lreg2w.man.yp-c.yandex.net',
            required=True,
        )
        diehard_url = sdk2.parameters.String(
            "Url for DieHard(host:port)",
            description="Target to use as DieHard backend",
            default_value='http://jj5thvn5siatlxoj.man.yp-c.yandex.net',
            required=True,
        )
        paysys_url = sdk2.parameters.String(
            "Url for Paysys(host:port)",
            description="Target to use as Paysys backend",
            default_value='http://65kvwpl66vzdlhjy.man.yp-c.yandex.net',
            required=True,
        )
        stable_lunapark_job_id = sdk2.parameters.String('Stable lunapark job id.', default_value='')
        ext_params = binary_task.LastBinaryReleaseParameters(stable=True)

        with sdk2.parameters.Output:
            testing_lunapark_job_id = sdk2.parameters.String('Testing lunapark job id.', default_value='')
            testing_lunapark_url = sdk2.parameters.String('Testing lunapark url.', default_value='')
            compare_url = sdk2.parameters.String('Testing vs Stable shootings compare url.', default_value='')

    def on_save(self):
        super(StartTrustPerfTesting, self).on_save()

    def on_execute(self):
        super(StartTrustPerfTesting, self).on_execute()

        with self.memoize_stage.find_tank_lxc_container:
            self._container = YANDEX_TANKAPI_LXC_CONTAINER.find(state=State.READY).order(-YANDEX_TANKAPI_LXC_CONTAINER.id).first().id
            logger.info("Found a YANDEX_TANKAPI_LXC_CONTAINER, id={}".format(str(self._container)))

        with self.memoize_stage.find_all_tanks:
            self.Context._all_tanks = ALL_TANKS

        with self.memoize_stage.start_shootings:
            self.Context._testing_shooting_id = self._start_shooting("TESTING")
            self.Context._stable_shooting_id = self.Parameters.stable_lunapark_job_id

        with self.memoize_stage.wait_shootings:
            ids_to_wait = filter(lambda id: id is not None, [self.Context._testing_shooting_id])
            raise sdk2.WaitTask(ids_to_wait, Status.Group.FINISH | Status.Group.BREAK, wait_all=True, timeout=14400)

        with self.memoize_stage.find_shooting_results:
            testing_result = self._find_shooting_result(self.Context._testing_shooting_id)
            self.Parameters.testing_lunapark_job_id = testing_result.Parameters.lunapark_job_id
            self.Parameters.testing_lunapark_url = testing_result.Parameters.lunapark_link
            self.set_info(
                link_format(self.Parameters.testing_lunapark_url, self.Parameters.testing_lunapark_url, description='Testing shooting: '),
                do_escape=False
            )

            if self.Context._stable_shooting_id is not None:
                stable_shooting_link = 'https://lunapark.yandex-team.ru/{}'.format(self.Parameters.stable_lunapark_job_id)
                self.set_info(
                    link_format(stable_shooting_link, stable_shooting_link, description='Stable shooting: '),
                    do_escape=False
                )

                self.Parameters.compare_url = "https://lunapark.yandex-team.ru/compare/" \
                                              "#jobs={stable_id},{testing_id}&tab=test_data&mainjob={stable_id}" \
                                              "&helper=all&cases=&plotGroup=additional&metricGroup=&target=" \
                                              .format(testing_id=self.Parameters.testing_lunapark_job_id,
                                                      stable_id=self.Parameters.stable_lunapark_job_id)
                self.set_info(
                    link_format(self.Parameters.compare_url, "url", description='Stable vs Testing compare: '),
                    do_escape=False
                )

    def _start_shooting(self, tag):
        tanks = self._find_tanks_in_same_dc(self.Parameters.trust_url)
        logger.info("Tanks to be used: {}".format(str(tanks)))

        gun_resource = self.Parameters.gun_resource
        if gun_resource is None:
            gun_resource = TrustScenariosPandoraGunExecutable.find(
                state=State.READY,
                attrs=dict(
                    released='stable',
                ),
            ).first()

        config_content = self._make_config_content(tag, gun_resource)

        logger.info("Preparing shooting task...")
        shooting_task = ShootViaTankapi(
            self,
            description="Shooting to {} TRUST:{}, DIEHARD:{}, Paysys: {}".format(tag, self.Parameters.trust_url, self.Parameters.diehard_url, self.Parameters.paysys_url),
            override_nanny_port=80,
            use_public_tanks=False,
            tanks=tanks,
            ammo_source="in_config",
            config_source="file",
            config_content=config_content,
            container=self._container,
        ).enqueue()
        logger.info("Started shooting task id={}".format(str(shooting_task.id)))
        return shooting_task.id

    def _find_shooting_result(self, shooting_task_id):
        logger.info("Searching for shooting result by id={}".format(shooting_task_id))
        shooting_result = sdk2.Task.find(
            id=shooting_task_id,
            children=True
        ).first()
        logger.info("Shooting result found, id={}".format(str(shooting_result.id)))
        if shooting_result.status not in Status.Group.SUCCEED:
            raise errors.TaskError("Child task did not succeed.")
        return shooting_result

    def _find_tanks_in_same_dc(self, target_url):
        dc = self._detect_dc(target_url)
        if dc not in ALL_TANKS:
            dc = 'sas'
        tank_infos = ALL_TANKS[dc]
        return ["{host}:{port}".format(host=tank_info["host"], port=tank_info["port"]) for tank_info in tank_infos]

    def _detect_dc(self, target_url):
        if ".man.yp-c.yandex.net" in target_url:
            return "man"
        if ".sas.yp-c.yandex.net" in target_url:
            return "sas"
        if ".vla.yp-c.yandex.net" in target_url:
            return "vla"
        if ".myt.yp-c.yandex.net" in target_url:
            return "myt"
        return "unknown"

    def _make_config_content(self, tag, gun_resource):
        config_template = \
"""
autostop:
    enabled: true
    package: yandextank.plugins.Autostop
    autostop: {autostop_criteria}
phantom: {{enabled: false}}
uploader:
    operator: lunapark
    task: {startrek_ticket_id}
    job_name: {tag}
    job_dsc: 'load testing started from Sandbox'
    package: yandextank.plugins.DataUploader
    enabled: true
    api_address: https://lunapark.yandex-team.ru/
pandora:
    enabled: true
    expvar: false
    pandora_cmd: {gun_download_url}
    resources:
      - src: {ammo_download_url}
        dst: ./ammo.json
    config_content:
        pools:
          - id: trust_load
            gun:
                type: trust_load
                global: 600
                trust: {trust_url}
                diehard: {diehard_url}
                paysys: {paysys_url}
            ammo:
                type: trust_load_provider
                source:
                      type: file
                      path: ./ammo.json
            result:
                type: phout
                destination: ./phout.log
            rps:
              {rps_load_profile}
            startup:
              {startup_load_profile}
        log:
          level: debug
"""
        return config_template.format(
            ammo_download_url=self.Parameters.ammo_resource.http_proxy,
            gun_download_url=gun_resource.http_proxy,
            startup_load_profile=self.Parameters.startup_load_profile,
            rps_load_profile=self.Parameters.rps_load_profile,
            startrek_ticket_id=self.Parameters.startrek_ticket_id,
            tag=tag,
            trust_url=self.Parameters.trust_url,
            diehard_url=self.Parameters.diehard_url,
            paysys_url=self.Parameters.paysys_url,
            autostop_criteria=self.Parameters.autostop_criteria
        )
