# coding: utf-8

from sandbox import sdk2
from sandbox.projects.turbo.component import TurboComponent
from sandbox.projects.turbo import resources
import sandbox.projects.common.dolbilka2 as dolbilka2
from sandbox.projects.common.search import bugbanner2
from sandbox.common.types.client import Tag
from sandbox.projects.common import error_handlers as eh
from sandbox.sandboxsdk import paths
from sandbox.sdk2.helpers import subprocess as sp


class PerformanceTurbo(bugbanner2.BugBannerTask):
    class Requirements(sdk2.Task.Requirements):

        client_tags = Tag.GENERIC & Tag.Group.LINUX & ~Tag.LXC

    class Parameters(sdk2.Parameters):
        package = sdk2.parameters.Resource("Package with turbo", resource_type=resources.TURBO_MERGER_PACKAGE)
        plan = sdk2.parameters.Resource("Queries plan", resource_type=resources.TURBO_PLAN, required=True)
        dolbilka_param = dolbilka2.DolbilkaExecutor2.Parameters
        critical = sdk2.parameters.Integer("Min expected rps", required=True, default=4500)
        threads = sdk2.parameters.Integer("Turbo merger threads", required=True, default=16)

    def on_enqueue(self):
        self.Context.result = []

    def on_execute(self):

        self.add_bugbanner(bugbanner2.Banners.Turbo)

        package_res_path = str(sdk2.ResourceData(self.Parameters.package).path)
        paths.make_folder('turbo_res', delete_content=True)
        unpack_cmd = ["tar", "-xvf", str(package_res_path), "-C", "turbo_res"]

        with sdk2.helpers.ProcessLog(self, logger='tar_actions') as pl:
            return_code = sp.Popen(
                unpack_cmd,
                shell=False, stdout=pl.stdout, stderr=sp.STDOUT
            ).wait()
            if return_code != 0:
                raise Exception('Failed to unpack actions repos')

        turbo_res = resources.TURBO_EXECUTABLE(
            self,
            "turbo_binary",
            "turbo_res/bin/turbo_merger"
        )
        turbo_conf = resources.TURBO_CONFIG(
            self,
            "turbo_config",
            "turbo_res/config.json"
        )

        d_exec = dolbilka2.DolbilkaExecutor2(self.Parameters.dolbilka_param)

        tank_plan = resources.TURBO_PLAN(
            self,
            "plan_for_tank",
            "plan_for_tank.txt",
            auto_update=False
        )

        d_exec.planner.create_plan(
            queries_path=str(sdk2.ResourceData(self.Parameters.plan).path),
            result_path=str(sdk2.ResourceData(tank_plan).path),
            loader_type='phantom',
        )

        turbo = TurboComponent(
            self,
            binary=turbo_res,
            config=turbo_conf,
            threads=self.Parameters.threads,
        )

        self.Context.result = d_exec.run_sessions(
            self,
            plan=str(sdk2.ResourceData(tank_plan).path),
            target=turbo,
            host="localhost",
            save_answers=True,
        )
        for data in self.Context.result:
            eh.ensure(data["rps"] > self.Parameters.critical, "RPS is too low")
            eh.ensure(
                data["requests_ok"] > float(self.Parameters.dolbilka_param.dolbilka_executor_requests_limit) * 98,
                "too many errors"
            )
        self.Context.result = [{"rps": "rps", "requests_ok": "OK requests"}] + self.Context.result

    @sdk2.header()
    def report(self):
        return {
            num: "<pre>{:12}</pre><br><pre>{:12}</pre><br>".format(data["rps"], data["requests_ok"])
            if "rps" in data else "wait" for num, data in enumerate(self.Context.result)
        }
