from sandbox.sandboxsdk import process, paths, network, errors
from sandbox import sdk2
from sandbox.projects import resource_types
from sandbox.projects.common import utils
import time
import tempfile
import logging


TURBO_MERGER = "turbo_merger"
TURBO_DDM = "turbo_ddm"
TURBO_INIT = "turbo_init"


class TurboComponent(object):

    name = 'turbo_merger'
    port = None

    def __init__(
            self,
            task=None,
            binary=None,
            port=None,
            config=None,
            threads=4,
            fake_init=False,
            tv_headers=None,
            comp_type=TURBO_MERGER,
            site_names_file=None,
    ):
        self.process = None
        self.task = task or None
        self.binary = binary
        self.config = config
        self.threads = threads
        self.port = port or utils.gimme_port()
        self.tv_headers = tv_headers
        self.comp_type = comp_type
        self.evlog_file = "turbo_logs/eventlog-{}.evlog".format(self.port)
        self.__endpoints = [("localhost", self.port)]
        self.__wait_timeout = 60
        self.fake_init = fake_init
        self.site_names_file = site_names_file

    def __enter__(self):
        if not self.fake_init:
            self.start()
            self.wait()
        return self

    def __exit__(self, exc_type, exc_value, traceback):
        if not self.fake_init:
            self.stop()

    def use_component(self, work_func):
        return work_func()

    def stop(self):
        if self.process:
            self.process.kill()

    def wait(self):
        logging.info("Waiting for component on endpoints: {}".format(self.__endpoints))
        wait_for = time.time() + self.__wait_timeout
        while time.time() < wait_for:
            if all(not network.is_port_free(port, host) for host, port in self.__endpoints):
                return
            time.sleep(2)
        raise errors.SandboxTaskFailureError(
            "Failed to connect to endpoints {} in {} seconds".format(self.__endpoints, self.__wait_timeout)
        )

    def is_running(self):
        if self.process:
            return self.process.poll() is None
        return False

    def start(self):
        binary_path = str(sdk2.ResourceData(self.binary).path)
        config_path = str(sdk2.ResourceData(self.config).path)
        paths.make_folder('turbo_logs', delete_content=True)
        self.logs = resource_types.PLAIN_TEXT(
            self.task,
            "turbo_logs",
            "turbo_logs"
        )
        cmd = '{} --port {} --config {} --threads {}'.format(
            binary_path,
            self.port,
            config_path,
            self.threads,
        )
        if self.comp_type != TURBO_INIT:
            cmd += ' --eventlog {}'.format(self.evlog_file)

        if self.site_names_file:
            cmd += ' --site-names-file {}'.format(self.site_names_file)

        with open("turbo_logs/{}_output.txt".format(self.comp_type), "w") as stdout_file:
            with open("turbo_logs/{}_err.txt".format(self.comp_type), "w") as stderr_file:
                tv_headers_file = None
                if self.tv_headers and self.comp_type == TURBO_DDM:
                    tv_headers_file = tempfile.NamedTemporaryFile(mode='w')
                    for name, value in self.tv_headers.iteritems():
                        logging.info('Got header "%s"', name)
                        tv_headers_file.write('{name}: {value}\n'.format(name=name, value=value))
                    tv_headers_file.flush()
                    cmd += ' --tv_headers {tv_headers_file_name}'.format(tv_headers_file_name=tv_headers_file.name)

                stdout_file.write("Started by sandbox, cmd:{0}\n".format(cmd))
                self.process = process.run_process(cmd, wait=False, stdout=stdout_file, stderr=stderr_file, shell=True)

                if tv_headers_file:
                    tv_headers_file.close()
        logging.info('Search component %s pid: %s', self.name, self.process.pid)

        return self.process
