import collections
import logging

import requests

PLATFORMS = [
    "cv",
    "cvte",
    "hikeen"
]

ALLOWED_CUSTOMERS = [
    "ARTEL",
    "CAIXUN",
    "CENTEK",
    "CHANGHONG",
    "CHAOYE",
    "DCZN",
    "ECON",
    "EXPRESSLUCK",
    "HIPER",
    "HUIDI",
    "HUIXIONG",
    "JINPIN",
    "KONKA",
    "KVANT",
    "LEARN",
    "MINGCAI",
    "MINGXIN",
    "MTIMES",
    "SHYS",
    "SQY",
    "UNIOTA",
    "YANDEX",
    "YANDEX_UNI",
    "YINGSHI",
    "ZHAOCHI",

    "TEST"
]

CLIDS_BY_BRAND = {
    "DEXP": {
        "clid1": 2399678,
        "clid100010": 2399674,
        "clid100011": 2399675,
        "clid100012": 2399677,
        "clid100013": 2399676,
        "clid100014": 2399679
    },
    "TELEFUNKEN": {
        "clid1": 2399683,
        "clid100010": 2399684,
        "clid100011": 2399685,
        "clid100012": 2399681,
        "clid100013": 2399682,
        "clid100014": 2399680
    },
    "MERLION": {
        "clid1": 2399830,
        "clid100010": 2399826,
        "clid100011": 2399828,
        "clid100012": 2399829,
        "clid100013": 2399825,
        "clid100014": 2399827
    },
    "BBK": {
        "clid1": 2399690,
        "clid100010": 2399686,
        "clid100011": 2399691,
        "clid100012": 2399687,
        "clid100013": 2399688,
        "clid100014": 2399689
    },
    "LEFF": {
        "clid1": 2399692,
        "clid100010": 2399697,
        "clid100011": 2399693,
        "clid100012": 2399694,
        "clid100013": 2399695,
        "clid100014": 2399696
    },
    "MVIDEO": {
        "clid1": 2406206,
        "clid100010": 2406203,
        "clid100011": 2406204,
        "clid100012": 2406205,
        "clid100013": 2406208,
        "clid100014": 2406207
    },
    "AMCV": {
        "clid1": 2407686,
        "clid100010": 2407689,
        "clid100011": 2407687,
        "clid100012": 2407691,
        "clid100013": 2407690,
        "clid100014": 2407688
    },
    "DOFFLER": {
        "clid1": 2407686,
        "clid100010": 2407689,
        "clid100011": 2407687,
        "clid100012": 2407691,
        "clid100013": 2407690,
        "clid100014": 2407688
    },
    "HARTENS": {
        "clid1": 2416930,
        "clid100010": 2416928,
        "clid100011": 2416932,
        "clid100012": 2416931,
        "clid100013": 2416929,
        "clid100014": 2416927
    },
    "DENN": {
        "clid1": 2416937,
        "clid100010": 2416934,
        "clid100011": 2416938,
        "clid100012": 2416933,
        "clid100013": 2416936,
        "clid100014": 2416935
    },
    "ECON": {
        "clid1": 2420783,
        "clid100010": 2420779,
        "clid100011": 2420782,
        "clid100012": 2420778,
        "clid100013": 2420780,
        "clid100014": 2420781
    },
    "POLARLINE": {
        "clid1": 2454614,
        "clid100010": 2454612,
        "clid100011": 2454613,
        "clid100012": 2454615,
        "clid100013": 2454616,
        "clid100014": 2454617
    },
    "VEKTA": {
        "clid1": 2460226,
        "clid100010": 2460229,
        "clid100011": 2460227,
        "clid100012": 2460225,
        "clid100013": 2460228,
        "clid100014": 2460224
    },
    "IRBIS": {
        "clid1": 2463797,
        "clid100010": 2463798,
        "clid100011": 2463800,
        "clid100012": 2463799,
        "clid100013": 2463801,
        "clid100014": 2463802
    },
    "KVANT": {
        "clid1": 2469381,
        "clid100010": 2469380,
        "clid100011": 2469382,
        "clid100012": 2469383,
        "clid100013": 2469384,
        "clid100014": 2469385
    },
    "QWATT": {
        "clid1": 2485079,
        "clid100010": 2485080,
        "clid100011": 2485077,
        "clid100012": 2485078,
        "clid100013": 2485076,
        "clid100014": 2485081
    },
    "ARTEL": {
        "clid1": 2527763,
        "clid100010": 2527766,
        "clid100011": 2527765,
        "clid100012": 2527764,
        "clid100013": 2527762,
        "clid100014": 2527767
    },
    "ASTROBT": {
        "clid1": 2527763,
        "clid100010": 2527766,
        "clid100011": 2527765,
        "clid100012": 2527764,
        "clid100013": 2527762,
        "clid100014": 2527767
    },
    "DIORIT": {
        "clid1": 2533017,
        "clid100010": 2533015,
        "clid100011": 2533019,
        "clid100012": 2533020,
        "clid100013": 2533016,
        "clid100014": 2533018
    },
    "BAFF": {
        "clid1": 2533017,
        "clid100010": 2533015,
        "clid100011": 2533019,
        "clid100012": 2533020,
        "clid100013": 2533016,
        "clid100014": 2533018
    },
    "HIBERG": {
        "clid1": 2533017,
        "clid100010": 2533015,
        "clid100011": 2533019,
        "clid100012": 2533020,
        "clid100013": 2533016,
        "clid100014": 2533018
    },
    "PRESTIGIO": {
        "clid1": 2561834,
        "clid100010": 2561839,
        "clid100011": 2561838,
        "clid100012": 2561837,
        "clid100013": 2561836,
        "clid100014": 2561835
    },
    "VITYAZ": {
        "clid1": 2406206,
        "clid100010": 2406203,
        "clid100011": 2406204,
        "clid100012": 2406205,
        "clid100013": 2406208,
        "clid100014": 2406207
    },
    "STARWIND": {
        "clid1": 2399830,
        "clid100010": 2399826,
        "clid100011": 2399828,
        "clid100012": 2399829,
        "clid100013": 2399825,
        "clid100014": 2399827
    },
    "ASANO": {
        "clid1": 2399692,
        "clid100010": 2399697,
        "clid100011": 2399693,
        "clid100012": 2399694,
        "clid100013": 2399695,
        "clid100014": 2399696
    },
    "HYUNDAI": {
        "clid1": 2399830,
        "clid100010": 2399826,
        "clid100011": 2399828,
        "clid100012": 2399829,
        "clid100013": 2399825,
        "clid100014": 2399827
    },
    "YUNO": {
        "clid1": 2399690,
        "clid100010": 2399686,
        "clid100011": 2399691,
        "clid100012": 2399687,
        "clid100013": 2399688,
        "clid100014": 2399689
    },
    "V-HOME": {
        "clid1": 2406206,
        "clid100010": 2406203,
        "clid100011": 2406204,
        "clid100012": 2406205,
        "clid100013": 2406208,
        "clid100014": 2406207
    },
    "HI": {
        "clid1": 2406206,
        "clid100010": 2406203,
        "clid100011": 2406204,
        "clid100012": 2406205,
        "clid100013": 2406208,
        "clid100014": 2406207
    },
    "NOVEX": {
        "clid1": 2406206,
        "clid100010": 2406203,
        "clid100011": 2406204,
        "clid100012": 2406205,
        "clid100013": 2406208,
        "clid100014": 2406207
    },
    "EMERALD": {
        "clid1": 2469381,
        "clid100010": 2469380,
        "clid100011": 2469382,
        "clid100012": 2469383,
        "clid100013": 2469384,
        "clid100014": 2469385
    },
    "HORIZONT": {
        "clid1": 2460226,
        "clid100010": 2460229,
        "clid100011": 2460227,
        "clid100012": 2460225,
        "clid100013": 2460228,
        "clid100014": 2460224
    },
    "NEKO": {
        "clid1": 2454614,
        "clid100010": 2454612,
        "clid100011": 2454613,
        "clid100012": 2454615,
        "clid100013": 2454616,
        "clid100014": 2454617
    },
    "SSMART": {
        "clid1": 2613061,
        "clid100010": 2613062,
        "clid100011": 2613059,
        "clid100012": 2613063,
        "clid100013": 2613064,
        "clid100014": 2613060
    },
    "HIPER": {
        "clid1": 2460226,
        "clid100010": 2460229,
        "clid100011": 2460227,
        "clid100012": 2460225,
        "clid100013": 2460225,
        "clid100014": 2460224
    },
    "CENTEK": {
        "clid1": 2633481,
        "clid100010": 2633478,
        "clid100011": 2633476,
        "clid100012": 2633479,
        "clid100013": 2633477,
        "clid100014": 2633480
    },
    "ERISSON": {
        "clid1": 2460226,
        "clid100010": 2460229,
        "clid100011": 2460227,
        "clid100012": 2460225,
        "clid100013": 2460228,
        "clid100014": 2460224
    },
    "SUNWIND": {
        "clid1": 2399830,
        "clid100010": 2399826,
        "clid100011": 2399828,
        "clid100012": 2399829,
        "clid100013": 2399825,
        "clid100014": 2399827
    },
    "POLAR": {
        "clid1": 2454614,
        "clid100010": 2454612,
        "clid100011": 2454613,
        "clid100012": 2454615,
        "clid100013": 2454616,
        "clid100014": 2454617
    },
    "HARPER": {
        "clid1": 2469381,
        "clid100010": 2469380,
        "clid100011": 2469382,
        "clid100012": 2469383,
        "clid100013": 2469384,
        "clid100014": 2469385
    },
    "ACER": {
        "clid1": 2399692,
        "clid100010": 2399697,
        "clid100011": 2399693,
        "clid100012": 2399694,
        "clid100013": 2399695,
        "clid100014": 2399696
    },
    "SKYLINE": {
        "clid1": 2469381,
        "clid100010": 2469380,
        "clid100011": 2469382,
        "clid100012": 2469383,
        "clid100013": 2469384,
        "clid100014": 2469385
    },
    "OLTO": {
        "clid1": 2469381,
        "clid100010": 2469380,
        "clid100011": 2469382,
        "clid100012": 2469383,
        "clid100013": 2469384,
        "clid100014": 2469385
    },
    "SAMTRON": {
        "clid1": 2454614,
        "clid100010": 2454612,
        "clid100011": 2454613,
        "clid100012": 2454615,
        "clid100013": 2454616,
        "clid100014": 2454617
    },
    "MAUNFELD": {
        "clid1": 2663097,
        "clid100010": 2663095,
        "clid100011": 2663096,
        "clid100012": 2663099,
        "clid100013": 2663098,
        "clid100014": 2663100
    },
    "TEST": {}
}

PLATFORMS_WITH_UNIOTA = ["cv6681", "cv9632"]
ALLOWED_UNIOTA_PROPS = {
    "cv6681": {
        "board": [
            "BD_MST151B_10ABQHAT_CV6681HA42",
            "BD_MST151B_10ABQHAT_CV6681HB42",
            "BD_MST151B_10ABQHAT_CV6681HB42_3AF",
            "BD_MST151B_10ABQHAT_CV6681K24",
            "BD_MST151B_10ABQHAT_CV6681K32",
            "BD_MST151B_10ABQHAT_CV6681K42",
            "BD_MST151B_10ABQHAT_CV6681L42_Earphone"
        ],
        "wifi": [
            "RTK8723DU"
        ]
    },
    "cv9632": {
        "board": [
            "BD_MT165B_10AT_CV9632AH",
            "BD_MT165B_10AT_CV9632HA50",
            "BD_MT165B_10AT_CV9632K55",
            "BD_MT165B_10AT_CV9632KH",
            "BD_MT165B_10AT_CV9632L55",
            "BD_MT165B_10AT_CV9632M42",
            "BD_MT165B_10AT_CV9632N55"
        ],
        "wifi": [
            "MT7668",
            "RTK88X2CU"
        ]
    }
}

DEFAULT_ROBOT_KEY = "sec-01d1ehgnbf5981b4f8dkg78ndj"


def notify_duty(message, ticket, secret):
    abc_req = requests.get(
        "https://abc-back.yandex-team.ru/api/v4/duty/on_duty/?service=34352&schedule__slug=android_tv_duty&fields=person.login",
        headers={'Authorization': 'OAuth {}'.format(secret.data()["abc-token"])}
    )
    current_duty = abc_req.json()[0]["person"]["login"]
    st_req = requests.post(
        "https://st-api.yandex-team.ru/v2/issues/{}/comments".format(ticket),
        headers={'Authorization': 'OAuth {}'.format(secret.data()["st-token"])},
        json={
            "text": message,
            "summonees": [current_duty]
        }
    )
    logging.info("Comment creation status: {}".format(st_req.status_code))
    logging.debug("Response: {}".format(st_req.content))


def get_resource_search_arguments(task):
    import sandbox.common.types.resource as ctr
    return {
        "attrs": {"task_bundle": "TV_BUNDLE", "released": task.Parameters.binary_executor_release_type},
        "state": [ctr.State.READY]
    }


class ConfigParsingException(Exception):
    """Exception raised for errors in provided data.
    """
    pass


class Config(object):
    def __init__(self, customer, target, boot_logo, recovery_logo, error_logo,
                 platform, ticket, panel, sw_name, props, vendor_config=None,
                 clids=None, uniota=False, uniota_props=None, last_modified=None):
        self.customer = customer
        self.target = target
        self.boot_logo = boot_logo
        self.recovery_logo = recovery_logo
        self.error_logo = error_logo
        self.platform = platform
        self.ticket = ticket
        self.panel = panel
        self.sw_name = sw_name
        self.props = props
        self.vendor_config = vendor_config
        self.clids = clids
        self.uniota = uniota
        self.uniota_props = uniota_props
        self.last_modified = last_modified

    @staticmethod
    def from_json(data, sw_name):
        clids = None
        if "clids" in data:
            clids = Config.Clids.from_json(data["clids"])

        vendor_config = None
        if "vendor_config" in data:
            vendor_config = Config.VendorConfig.from_json(data["vendor_config"])

        Config.verify_fields(data, ["customer",
                                    "target",
                                    "boot_logo",
                                    "recovery_logo",
                                    "error_logo",
                                    "platform",
                                    "ticket",
                                    "panel"])

        uniota = False
        if "uniota" in data:
            uniota = data["uniota"]

        uniota_props = None
        if data["platform"] in PLATFORMS_WITH_UNIOTA:
            Config.verify_fields(data, ["uniota_props"])
            json_props = data["uniota_props"]
            allowed_uniota_props = ALLOWED_UNIOTA_PROPS[data["platform"]]
            Config.verify_value(json_props, "board", allowed_uniota_props["board"])
            Config.verify_value(json_props, "wifi", allowed_uniota_props["wifi"])
            uniota_props = Config.UniotaProps.from_json(json_props)

        last_modified = None
        if "last_modified" in data:
            last_modified = data["last_modified"]

        return Config(
            data["customer"],
            data["target"],
            data["boot_logo"],
            data["recovery_logo"],
            data["error_logo"],
            data["platform"],
            data["ticket"],
            data["panel"],
            sw_name,
            Config.Props.from_json(data["props"]),
            vendor_config=vendor_config,
            clids=clids,
            uniota=uniota,
            uniota_props=uniota_props,
            last_modified=last_modified,
        )

    def to_json(self):
        result = collections.OrderedDict()

        result["customer"] = self.customer
        result["target"] = self.target
        result["boot_logo"] = self.boot_logo
        result["recovery_logo"] = self.recovery_logo
        result["error_logo"] = self.error_logo
        result["platform"] = self.platform

        if self.vendor_config is not None:
            result["vendor_config"] = self.vendor_config.to_json()

        if self.clids is not None:
            result["clids"] = self.clids.to_json()

        result["props"] = self.props.to_json()

        result["ticket"] = self.ticket
        result["panel"] = self.panel

        if self.uniota:
            result["uniota"] = self.uniota
        if self.uniota_props is not None:
            result["uniota_props"] = self.uniota_props.to_json()

        if self.last_modified:
            result["last_modified"] = self.last_modified

        return result

    @staticmethod
    def verify_fields(data, expected_fields):
        missed_fields = []

        for field in expected_fields:
            if field not in data:
                missed_fields.append(field)

        if missed_fields:
            raise ConfigParsingException("missing {}".format(", ".join(missed_fields)))

    @staticmethod
    def verify_value(data, key, allowed):
        value = data.get(key)
        if value not in allowed:
            raise ConfigParsingException("Value {} is not allowed for field {}. Allowed values: {}".format(value, key, ", ".join(allowed)))

    class VendorConfig(object):
        def __init__(self, pult, banners, feedback):
            self.pult = pult
            self.banners = banners
            self.feedback = feedback

        @staticmethod
        def from_json(data):
            return Config.VendorConfig(
                data.get("pult"),
                data.get("banners"),
                data.get("feedback")
            )

        def to_json(self):
            result = collections.OrderedDict()

            if self.pult is not None:
                result["pult"] = self.pult

            if self.banners is not None:
                result["banners"] = self.banners

            if self.feedback is not None:
                result["feedback"] = self.feedback

            return result

    class Clids(object):
        def __init__(self, clid1, clid100010, clid100011, clid100012,
                     clid100013, clid100014):
            self.clid1 = clid1
            self.clid100010 = clid100010
            self.clid100011 = clid100011
            self.clid100012 = clid100012
            self.clid100013 = clid100013
            self.clid100014 = clid100014

        @staticmethod
        def from_json(data):
            Config.verify_fields(data, ["clid1",
                                        "clid100010",
                                        "clid100011",
                                        "clid100012",
                                        "clid100013",
                                        "clid100014"])

            return Config.Clids(
                data["clid1"],
                data["clid100010"],
                data["clid100011"],
                data["clid100012"],
                data["clid100013"],
                data["clid100014"]
            )

        def to_json(self):
            result = collections.OrderedDict()
            result["clid1"] = self.clid1
            result["clid100010"] = self.clid100010
            result["clid100011"] = self.clid100011
            result["clid100012"] = self.clid100012
            result["clid100013"] = self.clid100013
            result["clid100014"] = self.clid100014
            return result

    class Props(object):
        def __init__(self, device, model, manufacturer, brand, name):
            self.device = device
            self.model = model
            self.manufacturer = manufacturer
            self.brand = brand
            self.name = name

        @staticmethod
        def from_json(data):
            Config.verify_fields(data, ["device",
                                        "model",
                                        "manufacturer",
                                        "brand",
                                        "name"])

            return Config.Props(
                data["device"],
                data["model"],
                data["manufacturer"],
                data["brand"],
                data["name"]
            )

        def to_json(self):
            result = collections.OrderedDict()
            result["brand"] = self.brand
            result["name"] = self.name
            result["device"] = self.device
            result["model"] = self.model
            result["manufacturer"] = self.manufacturer
            return result

    class UniotaProps(object):
        def __init__(self, board, wifi):
            self.board = board
            self.wifi = wifi

        @staticmethod
        def from_json(data):
            Config.verify_fields(data, ["board", "wifi"])
            return Config.UniotaProps(
                data["board"],
                data["wifi"]
            )

        def to_json(self):
            result = collections.OrderedDict()
            result["board"] = self.board
            result["wifi"] = self.wifi
            return result

        def __eq__(self, other):
            if isinstance(other, Config.UniotaProps):
                return self.board == other.board and self.wifi == other.wifi
            return False


def set_ticket_status(status, ticket, token):
    req_headers = {'Authorization': 'OAuth {}'.format(token)}
    st_req = requests.get(
        "https://st-api.yandex-team.ru/v2/issues/{}/transitions".format(ticket),
        headers=req_headers
    )
    logging.info("Transition getting status: {}".format(st_req.status_code))
    logging.debug("Response: {}".format(st_req.content))

    if any([x["id"] == status for x in st_req.json()]):
        st_req = requests.post(
            "https://st-api.yandex-team.ru/v2/issues/{}/transitions/{}/_execute".format(ticket, status),
            headers=req_headers
        )
        logging.info("Transition status: {}".format(st_req.status_code))
        logging.debug("Response: {}".format(st_req.content))
