import logging
import os

import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm

from sandbox.common.errors import TaskFailure

import sandbox.projects.common.binary_task as binary_task

from sandbox import sdk2

from sandbox.sdk2.helpers import ProcessLog
from sandbox.sdk2.helpers import subprocess as sp

from sandbox.projects.common.vcs.arc import Arc
from sandbox.projects.common.binary_task import LastRefreshableBinary

from sandbox.projects.mobile_apps.utils.android_sdk_env import AndroidSdkEnvironment

from sandbox.projects.tv.common import DEFAULT_ROBOT_KEY, get_resource_search_arguments


class TvDocsWarden(LastRefreshableBinary, sdk2.Task):

    ARC_PATH = "arcadia"
    TV_SOURCES_REPO = "smart_devices/android"
    TV_DOCS_PATH = "tv/docs/flags.md"
    ANDROID_SDK_REQUIREMENTS = "platforms(28,29,30,31)+tools(31.0.0)"

    DEPENDENCY_EXCLUSION_LIST = [
        "./tv/common/lintcheck/build.gradle",
        "./tv/common/docs-processor/build.gradle"
    ]

    class Requirements(sdk2.Requirements):
        dns = ctm.DnsType.DNS64
        client_tags = ctc.Tag.LINUX_BIONIC
        container_resource = 2466247729

        cores = 4
        ram = 4096

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        _lbrp = binary_task.binary_release_parameters(stable=True)

        revision = sdk2.parameters.String("Change revision", required=True)
        pr_id = sdk2.parameters.Integer("Related PR id")

    @property
    def binary_executor_query(self):
        return get_resource_search_arguments(self)

    def on_execute(self):
        secret = sdk2.yav.Secret(DEFAULT_ROBOT_KEY)
        arc_token = secret.data()["arc-token"]
        arc_client = Arc(arc_oauth_token=arc_token)

        mount_point = arc_client.mount_path(None, None, fetch_all=False)
        mp = mount_point._work_path
        arc_client.checkout(mp, str(self.Parameters.revision))

        os.chdir(os.path.join(mp, self.TV_SOURCES_REPO))

        self.check_dependency()
        self.check_docs()

    def check_dependency(self):
        gradle_file = "build.gradle"
        gradle_build_files = [
            os.path.join(x[0], gradle_file) for x in os.walk(".") if any([y for y in x[2] if y == gradle_file])
        ]

        for build_file_path in gradle_build_files:
            if build_file_path in self.DEPENDENCY_EXCLUSION_LIST:
                continue
            with open(build_file_path) as build_file:
                file_content = build_file.read()
                contains_annotations = "docs-annotations" in file_content
                contains_processor = "docs-processor" in file_content
                if contains_annotations != contains_processor:
                    raise TaskFailure(
                        "{} should contain both :tv:common:docs-annotations and :tv:common:docs-processor".format(
                            os.path.join(self.TV_SOURCES_REPO, build_file_path)
                        )
                    )

    def check_docs(self):
        if not os.path.exists(self.TV_DOCS_PATH):
            return

        with ProcessLog(self, logger=logging.getLogger('gradle')) as pl:
            android_sdk_environment = AndroidSdkEnvironment(self.ANDROID_SDK_REQUIREMENTS)
            android_sdk_environment.prepare(None, None, None)
            sp.check_call("./gradlew -Pproduct=tv -p tv publishFlags", shell=True, stderr=pl.stderr, stdout=pl.stdout)

        with ProcessLog(self, logger=logging.getLogger('arc')) as pl:
            out = sp.check_output("arc diff {}".format(self.TV_DOCS_PATH), shell=True, stderr=pl.stderr)

        logging.info(out)

        if len(out) > 0:
            raise TaskFailure("Flags had been changed. Please run gradle publishFlags and commit result\n\n\n{}".format(out))
