import json
import os

import sandbox.projects.common.binary_task as binary_task
import sandbox.projects.quasar.platform as quasar_platform
from sandbox import common
from sandbox import sdk2
from sandbox.common.types.task import Status as TaskStatus
from sandbox.projects.common.binary_task import LastRefreshableBinary
from sandbox.projects.tv.common import Config
from sandbox.projects.tv.common import get_resource_search_arguments, DEFAULT_ROBOT_KEY
from sandbox.projects.tv.target_check.common.get_properties import get_properties, save_properties_to_file
from sandbox.projects.tv.target_check.common.st_client import create_comment_in_startrek
from sandbox.projects.tv.target_check.common.test import compare_props
from sandbox.projects.tv.target_check.common.utils import echo_title, copy_files
from sandbox.sdk2.vcs.svn import Arcadia


class TargetCheckReports(sdk2.Resource):
    tv_platform = sdk2.resource.Attributes.String("tv_platform")
    tv_target = sdk2.resource.Attributes.String("tv_target")
    startrek_task_number = sdk2.resource.Attributes.String("startrek_task_number")


class TargetCheck(LastRefreshableBinary, sdk2.Task):
    ARC_PATH = "arcadia"
    ARC_BASE_URL = "https://a.yandex-team.ru/arcadia"
    TV_FIRMWARE_REPO_DIR = "smart_devices/tv/platforms"

    class Parameters(sdk2.Parameters):
        _lbrp = binary_task.binary_release_parameters(stable=True)
        report_resource = sdk2.parameters.Resource('Build TV firmware report', required=True)
        kill_timeout = 3600

    @property
    def binary_executor_query(self):
        return get_resource_search_arguments(self)

    def on_prepare(self):
        self.secret = sdk2.yav.Secret(DEFAULT_ROBOT_KEY)
        self.private_ssh_key = self.secret.data()["priv-key"]
        self.tv_platform = self.Parameters.platform
        self.tv_target = sdk2.Resource[self.Parameters.report_resource].tv_target

        self.properties_keys_list = self.Parameters.properties_keys_list

        tv_platform = quasar_platform.TvPlatform(self.tv_platform)
        self.target_config_folder_path = os.path.join(tv_platform.odm, tv_platform.board_number)

        config_path = os.path.join(self.target_config_folder_path, self.tv_target) + ".json"
        self.target_json_config_url = os.path.join(self.ARC_BASE_URL, self.TV_FIRMWARE_REPO_DIR, config_path)
        arcadia_url = os.path.join(Arcadia.ARCADIA_TRUNK_URL, self.TV_FIRMWARE_REPO_DIR)
        Arcadia.checkout(arcadia_url, self.ARC_PATH)

        with open(os.path.join(self.ARC_PATH, config_path)) as json_file:
            self.target_json_config = Config.from_json(json.load(json_file),
                                                       json_file.name.split("/")[-1].replace(".json", ""))

    def on_execute(self):
        echo_title("Init input parameters")
        report_resource_path = sdk2.ResourceData(sdk2.Resource[self.Parameters.report_resource]).path
        report_resource_task_id = sdk2.Resource[self.Parameters.report_resource].task_id

        echo_title("Prepare resource for results")
        results = TargetCheckReports(self, "Target check reports", "target_check_reports")
        results.tv_platform = self.tv_platform
        results.tv_target = self.tv_target
        results.json_config_link = self.target_json_config_url
        data = sdk2.ResourceData(results)
        data.path.mkdir(mode=0o775, parents=True, exist_ok=True)

        echo_title("Copy files from input resource")
        copy_files(report_resource_path, data.path)

        echo_title("Parse all properties from build rom configs")
        properties_from_build_rom = get_properties(resource_path=report_resource_path,
                                                   properties_keys_list=self.Parameters.properties_keys_list,
                                                   result_path=data.path)

        echo_title("Generate properties.txt")
        save_properties_to_file(properties=properties_from_build_rom,
                                platform=self.tv_platform,
                                target=self.tv_target,
                                result_path=data.path)

        echo_title("Compare properties")
        compare_props(json_config=self.target_json_config,
                      build_props=properties_from_build_rom,
                      report_path=data.path,
                      platform=self.tv_platform,
                      target=self.tv_target)

        echo_title("Create comment in startrek")
        create_comment_in_startrek(issue_task_number=self.target_json_config.ticket,
                                   properties=properties_from_build_rom,
                                   reports_path=data.path,
                                   json_url=self.target_json_config_url,
                                   sandbox_task_id=self.id,
                                   report_resource_task_id=report_resource_task_id)

        data.ready()

        echo_title("FINISH")

    @sdk2.header(title='Test results:')
    def header(self):
        try:
            if self.status not in TaskStatus.Group.FINISH | TaskStatus.Group.BREAK:
                return ''
            test_report = sdk2.Resource.find(
                type='TARGET_CHECK_REPORTS',
                task_id=self.id,
                state=common.types.resource.State.READY
            ).first()

            result = '<div style="font-size: 15px;">'
            path = 'https://proxy.sandbox.yandex-team.ru/{0}'.format(test_report.id)
            result += """<p> Clids, build props and logo images:
                                                <a href="{0}/properties.html">properties.html</a>
                                                </p>""".format(path)

            result += """<p> Target test results:
                                                <a href="{0}/test_results.html">test_results.html</a>
                                                </p>""".format(path)

            return result

        except Exception as ex:
            return getattr(ex, 'message', repr(ex))
