import logging

from sandbox import sdk2
from sandbox.projects.tv.common import DEFAULT_ROBOT_KEY
from sandbox.projects.tv.target_check.common.utils import generate_report_text, generate_attachments_list


def api_call_with_retry(tracker_method):
    from retrying import retry

    @retry(wait_exponential_multiplier=5000, wait_exponential_max=60000)
    def _api_call():
        try:
            tracker_method()
            return True
        except Exception as e:
            logging.info("safe_api_call: Failed to call tracker api:")
            logging.info(e)
            return False

    return _api_call()


def get_startrek_client():
    from startrek_client import Startrek
    secret = sdk2.yav.Secret(DEFAULT_ROBOT_KEY)
    st_token = secret.data()["st-token"]
    return Startrek(useragent="python client", base_url="https://st-api.yandex-team.ru", token=st_token)


def move_issue_to_testing(task_number):
    st_issue_client = get_startrek_client().issues[task_number]
    available_transitions = [transition.id for transition in st_issue_client.transitions.get_all()]
    if "readyForBuild" in available_transitions:
        logging.info("move_issue_to_testing: \nMove issue {}  from In progress to readyForBuild".format(task_number))
        api_call_with_retry(lambda: st_issue_client.transitions["readyForBuild"].execute())
        logging.info("move_issue_to_testing: \nMove issue {}  from readyForBuild to readyForTest".format(task_number))
        api_call_with_retry(lambda: st_issue_client.transitions["readyForTest"].execute())
    if "readyForTest" in available_transitions:
        logging.info("move_issue_to_testing: \nMove issue {}  from readyForBuild to readyForTest".format(task_number))
        api_call_with_retry(lambda: st_issue_client.transitions["readyForTest"].execute())


def get_task_status(task_number):
    st_issue_client = get_startrek_client().issues[task_number]
    return st_issue_client.status.key


def create_comment(report_text, attachments, task_number):
    st_issue_client = get_startrek_client().issues[task_number]
    return api_call_with_retry(lambda: st_issue_client.comments.create(text=report_text, attachments=attachments,
                                                                       summonees=["prokin-lev", "e-galkin"]))


def create_comment_without_attachments(report_text, task_number):
    st_issue_client = get_startrek_client().issues[task_number]
    api_call_with_retry(lambda: st_issue_client.comments.create(text=report_text, summonees=["prokin-lev", "e-galkin"]))


def create_comment_in_startrek(issue_task_number, properties, reports_path, sandbox_task_id, json_url,
                               report_resource_task_id):

    if issue_task_number is not None and issue_task_number != "TVANDROID-NONE":
        # no add comment if status is: tested, waitingForApproval, released, closed
        closed_task_statuses = ["tested", "waitingForApproval", "released", "closed"]
        actual_task_status = get_task_status(issue_task_number)
        if actual_task_status in closed_task_statuses:
            logging.info("Don't add a comment. {0} is {1}".format(issue_task_number, actual_task_status))
        else:
            report_text = generate_report_text(properties=properties,
                                               sandbox_task_id=sandbox_task_id,
                                               json_url=json_url,
                                               report_resource_task_id=report_resource_task_id)
            attachments = generate_attachments_list(properties=properties, reports_path=reports_path)

            logging.info(report_text)

            if attachments:
                logging.info("Create comment with attachments")
                success_call = create_comment(report_text, attachments, issue_task_number)

                if not success_call:
                    logging.info("Failed to send with attachment, try without")
                    create_comment_without_attachments(report_text, issue_task_number)
            else:
                logging.info("Create comment without attachments")
                create_comment_without_attachments(report_text, issue_task_number)

            move_issue_to_testing(issue_task_number)
    else:
        logging.info("Don't add a comment. TASK_NUMBER not provided")
