# coding=utf-8
import datetime
import json
import logging
import os

import requests

from requests.auth import HTTPBasicAuth

import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm

import sandbox.projects.common.binary_task as binary_task
from sandbox import sdk2
from sandbox.sdk2.vcs.svn import Arcadia
from sandbox.sdk2.helpers import ProcessLog
from sandbox.sdk2.helpers import subprocess as sp

from sandbox.projects.common.binary_task import LastRefreshableBinary

from sandbox.projects.tv.common import DEFAULT_ROBOT_KEY, notify_duty, get_resource_search_arguments, set_ticket_status
from sandbox.projects.tv.common import Config


class TargetatorException(Exception):
    pass


def _isascii(s):
    try:
        s.decode('ascii')
    except UnicodeDecodeError:
        return False
    except UnicodeEncodeError:
        return False
    else:
        return True


def _safe_read(file):
    return str(filter(lambda x: _isascii(x), file.read()))


class Targetator(LastRefreshableBinary, sdk2.Task):
    ROBOT_ALIAS = "robot-edi"

    ARC_PATH = "arcadia"
    STORE_PATH = "store"
    TV_FIRMWARE_REPO_DIR = "smart_devices/tv/platforms"
    RESOURCES_DIR = "resources"

    RELEASE_CONFIG_FILE = "release.json"
    RELEASE_DEFAULT_BRANCHES_KEY = "default_branches"
    RELEASE_CUSTOM_BRANCHES_KEY = "{}_branches"
    RELEASE_PLATFORM_BRANCH_KEY = "platform"
    RELEASE_APPS_BRANCH_KEY = "apps"

    TARGETS_REPO = "ssh://git@bb.yandex-team.ru/yandex-tv/targets_lists.git"
    TARGETS_REPO_DIR = "targets_lists"
    TARGETS_FILE_PATTERN = "targets_{}.json"

    REPO_HOOK_TEMPLATE = "{}/.git/hooks/"

    STAGE_TWO_ENABLED = False

    class Requirements(sdk2.Requirements):
        dns = ctm.DnsType.DNS64
        client_tags = ctc.Tag.LINUX_BIONIC

        # Multislot requirements
        cores = 2
        ram = 4096

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        _lbrp = binary_task.binary_release_parameters(stable=True)

        revision = sdk2.parameters.Integer("Change revision", required=True)
        configuration_path = sdk2.parameters.String("Configuration file path", description="smart_devices/tv/...", required=True)
        trigger_teamcity_build = sdk2.parameters.Bool("Trigger build on teamcity", required=False, default=True)
        branches = sdk2.parameters.List("Factory branch(-es)", required=False, default=[])

    @property
    def binary_executor_query(self):
        return get_resource_search_arguments(self)

    def on_prepare(self):
        self.secret = sdk2.yav.Secret(DEFAULT_ROBOT_KEY)
        self.private_ssh_key = self.secret.data()["priv-key"]

        arcadia_url = os.path.join(Arcadia.ARCADIA_TRUNK_URL, self.TV_FIRMWARE_REPO_DIR)
        Arcadia.checkout(arcadia_url, self.ARC_PATH, revision=self.Parameters.revision)

        with open(os.path.join(self.ARC_PATH, self.Parameters.configuration_path)) as json_file:
            self.configuration = Config.from_json(json.load(json_file), json_file.name.split("/")[-1].replace(".json", ""))

        with open(os.path.join(self.ARC_PATH, self.RELEASE_CONFIG_FILE)) as json_file:
            self.release_config = json.load(json_file)

    def link_to_ticket(self):
        st_req = requests.post(
            "https://st-api.yandex-team.ru/v2/issues/{}/remotelinks?notifyAuthor=false&backlink=false".format(self.configuration.ticket),
            headers={
                'Authorization': 'OAuth {}'.format(self.secret.data()["st-token"]),
                'Content-Type': 'application/json'
            },
            json={
                "relationship": "relates",
                "key": str(self.id),
                "origin": "ru.yandex.sandbox"
            }
        )
        logging.info("Sandbox linking status: {}".format(st_req.status_code))
        logging.debug("Response: {}".format(st_req.content))

    def repo_initialize(self, repo, path):
        hook = self.REPO_HOOK_TEMPLATE.format(path)
        repo_link = "ssh://{}@gerrit.yandex-team.ru/{}".format(self.ROBOT_ALIAS, repo)
        try:
            with sdk2.ssh.Key(private_part=self.private_ssh_key), ProcessLog(self, logger=logging.getLogger('git')) as pl:
                sp.check_call("git clone \"{}\" {}".format(repo_link, path), shell=True, stderr=pl.stderr, stdout=pl.stdout)
                sp.check_call(
                    "scp -p -P 22 {}@gerrit.yandex-team.ru:hooks/commit-msg \"{}\"".format(self.ROBOT_ALIAS, hook),
                    shell=True, stderr=pl.stderr, stdout=pl.stdout
                )
                prev_dir = os.getcwd()
                os.chdir(path)
                sp.check_call("git config user.email \"{}@yandex-team.ru\"".format(self.ROBOT_ALIAS), shell=True, stderr=pl.stderr, stdout=pl.stdout)
                sp.check_call("git config user.name \"{}\"".format(self.ROBOT_ALIAS), shell=True, stderr=pl.stderr, stdout=pl.stdout)
                sp.check_call("git checkout -b {}".format(self.configuration.ticket), shell=True, stderr=pl.stderr, stdout=pl.stdout)
                os.chdir(prev_dir)
        except sp.CalledProcessError as e:
            logging.error('git clone error: %s', e)
            raise e

    def repo_initialize_sparse(self, repo, path, directory):
        hook = self.REPO_HOOK_TEMPLATE.format(path)
        repo_link = "ssh://{}@gerrit.yandex-team.ru/{}".format(self.ROBOT_ALIAS, repo)
        try:
            with sdk2.ssh.Key(private_part=self.private_ssh_key), ProcessLog(self, logger=logging.getLogger('git')) as pl:
                sp.check_call("git init {}".format(path), shell=True, stderr=pl.stderr, stdout=pl.stdout)
                sp.check_call(
                    "scp -p -P 22 {}@gerrit.yandex-team.ru:hooks/commit-msg \"{}\"".format(self.ROBOT_ALIAS, hook),
                    shell=True, stderr=pl.stderr, stdout=pl.stdout
                )

                prev_dir = os.getcwd()

                os.chdir(path)

                sp.check_call("git remote add origin {}".format(repo_link), shell=True, stderr=pl.stderr, stdout=pl.stdout)
                sp.check_call("git config core.sparsecheckout true", shell=True, stderr=pl.stderr, stdout=pl.stdout)
                sp.check_call("echo \"{}\" >> .git/info/sparse-checkout".format(directory), shell=True, stderr=pl.stderr, stdout=pl.stdout)
                sp.check_call("git pull --depth 1 origin master", shell=True, stderr=pl.stderr, stdout=pl.stdout)

                for branches in self.get_release_branches():
                    platform_branch = branches[self.RELEASE_PLATFORM_BRANCH_KEY]
                    branch_exists = sp.check_output("git ls-remote --heads origin {}".format(platform_branch), shell=True)

                    if branch_exists:
                        sp.check_call("git fetch --depth 1 origin {}".format(platform_branch), shell=True, stderr=pl.stderr, stdout=pl.stdout)

                sp.check_call("git config user.email \"{}@yandex-team.ru\"".format(self.ROBOT_ALIAS), shell=True, stderr=pl.stderr, stdout=pl.stdout)
                sp.check_call("git config user.name \"{}\"".format(self.ROBOT_ALIAS), shell=True, stderr=pl.stderr, stdout=pl.stdout)
                sp.check_call("git checkout -b {}".format(self.configuration.ticket), shell=True, stderr=pl.stderr, stdout=pl.stdout)

                os.chdir(prev_dir)
        except sp.CalledProcessError as e:
            logging.error('git clone error: %s', e)
            raise e

    def gerrit_commit_and_push(self, path):
        config = self.configuration
        try:
            with sdk2.ssh.Key(private_part=self.private_ssh_key), ProcessLog(self, logger=logging.getLogger('git')) as pl:
                prev_dir = os.getcwd()
                os.chdir(path)
                has_changes = sp.call("git diff HEAD --no-ext-diff --quiet", shell=True, stderr=pl.stderr, stdout=pl.stdout)
                if has_changes:
                    sp.check_call(
                        "git commit -a -m \"[{}] Added/Edited target for {} {} {}\"".format(
                            config.ticket,
                            config.customer,
                            config.props.brand,
                            config.props.name
                        ),
                        shell=True,
                        stderr=pl.stderr,
                        stdout=pl.stdout
                    )
                    out = sp.check_output(
                        "git push origin HEAD:refs/for/master -o topic=\"{}\" 2>&1".format(config.ticket),
                        shell=True
                    )

                    if self.STAGE_TWO_ENABLED:
                        sp.check_call(
                            "ssh -p 22 {}@gerrit.yandex-team.ru gerrit review --code-review +2 $(git rev-parse HEAD)".format(self.ROBOT_ALIAS),
                            shell=True,
                            stderr=pl.stderr,
                            stdout=pl.stdout
                        )
                        sp.check_call(
                            "ssh -p 22 {}@gerrit.yandex-team.ru gerrit review --submit $(git rev-parse HEAD)".format(self.ROBOT_ALIAS),
                            shell=True,
                            stderr=pl.stderr,
                            stdout=pl.stdout
                        )
                        CHERRY_PICK_TOOL = "gerrit-cherry-pick"
                        sp.check_call(
                            "scp -p -P 22 {}@gerrit.yandex-team.ru:bin/gerrit-cherry-pick \"{}\"".format(self.ROBOT_ALIAS, CHERRY_PICK_TOOL),
                            shell=True,
                            stderr=pl.stderr,
                            stdout=pl.stdout
                        )
                        out = sp.check_output("echo \"{}\" | grep -oP \"/\\+/\\K\\w+\" 2>&1".format(out), shell=True)

                        for branches in self.get_release_branches():
                            platform_branch = branches[self.RELEASE_PLATFORM_BRANCH_KEY]
                            branch_exists = sp.check_output("git ls-remote --heads origin {0}".format(platform_branch), shell=True)
                            if not branch_exists:
                                logging.info("branch {0} does not exist, skip cherry-pick".format(platform_branch))
                                continue
                            sp.check_call(
                                "git checkout {0}"
                                " && ./{1} origin {2}"
                                " && git push origin HEAD:refs/for/{0} -o topic=\"{3}\""
                                " && rm -rf .git/rebase-apply".format(
                                    platform_branch,
                                    CHERRY_PICK_TOOL,
                                    out.replace('\n', ''),
                                    config.ticket
                                ),
                                shell=True,
                                stderr=pl.stderr,
                                stdout=pl.stdout
                            )
                            sp.check_call(
                                "ssh -p 22 {}@gerrit.yandex-team.ru gerrit review --code-review +2 $(git rev-parse HEAD)".format(self.ROBOT_ALIAS),
                                shell=True,
                                stderr=pl.stderr,
                                stdout=pl.stdout
                            )
                            sp.check_call(
                                "ssh -p 22 {}@gerrit.yandex-team.ru gerrit review --submit $(git rev-parse HEAD)".format(self.ROBOT_ALIAS),
                                shell=True,
                                stderr=pl.stderr,
                                stdout=pl.stdout
                            )
                            sp.check_call(
                                "git checkout {}".format(config.ticket),
                                shell=True,
                                stderr=pl.stderr,
                                stdout=pl.stdout
                            )
                else:
                    logging.info("No changes detected in {} repo".format(os.path.basename(path)))
                os.chdir(prev_dir)
        except sp.CalledProcessError as e:
            logging.error(e.output)
            logging.error('git commit and push failed: %s', e)
            raise e

    def load_image(self, image_path, target_path):
        try:
            with ProcessLog(self, logger=logging.getLogger('cp')) as pl:
                sp.check_call(
                    "cp {} {}".format(os.path.join(self.ARC_PATH, self.RESOURCES_DIR, image_path), target_path),
                    shell=True,
                    stderr=pl.stderr,
                    stdout=pl.stdout
                )
        except sp.CalledProcessError as e:
            logging.error('image loading failed: %s', e)
            raise e

    def add_remove_target(self, config, branch_name, pl, auth, remove_target):
        message_str = "Added/Edited"
        if remove_target:
            message_str = "Removed"
        message = "[{}] {} target for {} {} {}".format(
            config.ticket,
            message_str,
            config.customer,
            config.props.brand,
            config.props.name
        )

        sp.check_call("git commit -a -m \"{}\"".format(message), shell=True, stderr=pl.stderr, stdout=pl.stdout)
        sp.check_call("git push origin \"{}\"".format(branch_name), shell=True, stderr=pl.stderr, stdout=pl.stdout)

        body = {
            "title": message,
            "description": message,
            "state": "OPEN",
            "open": True,
            "closed": False,
            "fromRef": {
                "id": "refs/heads/{}".format(branch_name),
                "repository": {
                    "slug": "targets_lists",
                    "name": None,
                    "project": {
                        "key": "YANDEX-TV"
                    }
                }
            },
            "toRef": {
                "id": "refs/heads/master",
                "repository": {
                    "slug": "targets_lists",
                    "name": None,
                    "project": {
                        "key": "YANDEX-TV"
                    }
                }
            },
            "locked": False,
            "reviewers": [],
            "close_source_branch": True
        }

        request = requests.post(
            'https://bb.yandex-team.ru/rest/api/1.0/projects/YANDEX-TV/repos/targets_lists/pull-requests/',
            auth=auth,
            json=body
        )
        logging.info("BB response code: {}".format(request.status_code))
        logging.debug("Response: {}".format(request.content))
        return request

    def edit_targets_list(self, config):
        branch_name = "{}-{}".format(config.ticket, datetime.datetime.today().strftime('%Y-%m-%d-%H-%M-%S'))
        prev_dir = os.getcwd()
        try:
            with sdk2.ssh.Key(private_part=self.private_ssh_key), ProcessLog(self, logger=logging.getLogger('git')) as pl:
                sp.check_call(
                    "git clone \"{}\" {}".format(self.TARGETS_REPO, self.TARGETS_REPO_DIR),
                    shell=True,
                    stderr=pl.stderr,
                    stdout=pl.stdout
                )
                os.chdir(self.TARGETS_REPO_DIR)
                sp.check_call(
                    "git config user.email \"{}@yandex-team.ru\"".format(self.ROBOT_ALIAS),
                    shell=True,
                    stderr=pl.stderr,
                    stdout=pl.stdout
                )
                sp.check_call(
                    "git config user.name \"{}\"".format(self.ROBOT_ALIAS),
                    shell=True,
                    stderr=pl.stderr,
                    stdout=pl.stdout
                )
                sp.check_call(
                    "git checkout -b {}".format(branch_name),
                    shell=True,
                    stderr=pl.stderr,
                    stdout=pl.stdout
                )
        except sp.CalledProcessError as e:
            logging.error('git error: %s', e)
            raise e

        with open(self.TARGETS_FILE_PATTERN.format(config.platform), 'r+') as file:
            data = json.loads(file.read())
            file.seek(0)

            try:
                prev_sw_name = dict(zip(data["targets"].values(), data["targets"].keys()))[config.target]
                data["targets"].pop(prev_sw_name, None)
            except Exception:
                pass

            data["targets"][config.sw_name] = config.target

            file.write(json.dumps(data, indent=2, sort_keys=True))
            file.truncate()

        try:
            with sdk2.ssh.Key(private_part=self.private_ssh_key), ProcessLog(self, logger=logging.getLogger('git')) as pl:
                has_changes = sp.call("git diff --no-ext-diff --quiet", shell=True, stderr=pl.stderr, stdout=pl.stdout)
                auth = HTTPBasicAuth(self.ROBOT_ALIAS, str(self.secret.data()["bb-personal-token"]))
                if has_changes:
                    request = self.add_remove_target(config, branch_name, pl, auth, remove_target=False)
                else:
                    logging.info("No changes detected in target_lists repo")

                if self.STAGE_TWO_ENABLED:
                    if has_changes:
                        pr_id = request.json()["id"]
                        request = requests.post(
                            'https://bb.yandex-team.ru/rest/api/1.0/projects/YANDEX-TV/repos/targets_lists/pull-requests/{}/merge'.format(pr_id),
                            auth=auth,
                            json={"version": 0}
                        )
                        logging.info("BB merge response code: {}".format(request.status_code))
                        logging.debug("Merge response: {}".format(request.content))

                    set_ticket_status("readyForBuild", config.ticket, self.secret.data()["st-token"])

                    if self.Parameters.trigger_teamcity_build:
                        self.trigger_teamcity_builds(config)
                    else:
                        logging.info("teamcity build skipped")

        except sp.CalledProcessError as e:
            logging.error('git commit and push failed: %s', e)
            raise e

        os.chdir(prev_dir)

    def trigger_teamcity_builds(self, config):
        for branches in self.get_release_branches():
            platform_branch = branches[self.RELEASE_PLATFORM_BRANCH_KEY]
            apps_branch = branches[self.RELEASE_APPS_BRANCH_KEY]
            build_request_json = {
                "branchName": "refs/heads/{}".format(platform_branch),
                "buildType": {
                    "id": self.get_build_task(config.platform)
                },
                "comment": {
                    "text": "Targetator automated build for {} {} {}".format(
                        config.customer,
                        config.props.brand,
                        config.props.name
                    )
                },
                "properties": {
                    "property": [{
                        "name": "build.PRODUCT_TARGET",
                        "value": config.target
                    }, {
                        "name": "task_number",
                        "value": config.ticket
                    }, {
                        "name": "yandex.apps.branch",
                        "value": apps_branch
                    }, {
                        "name": "vcs.default.BRANCH",
                        "value": platform_branch
                    }]
                }
            }

            request = requests.post(
                'https://teamcity.yandex-team.ru/app/rest/buildQueue',
                headers={'Authorization': 'OAuth {}'.format(self.secret.data()["teamcity-oauth-token"])},
                json=build_request_json
            )
            logging.info("Request https://teamcity.yandex-team.ru/app/rest/buildQueue {}".format(build_request_json))
            logging.info("Build response code: {}".format(request.status_code))
            logging.debug("Build response: {}".format(request.content))
            if request.status_code / 100 in [4, 5]:
                raise Exception("Failed to start teamcity build")

    def get_release_branches(self):
        if self.Parameters.branches:
            # todo return just self.Parameters.branches
            # needed for backward compatibility until we stop using teamcity builds
            return map(lambda b: {'platform': b}, self.Parameters.branches)

        branches = self.release_config.get(self.RELEASE_CUSTOM_BRANCHES_KEY.format(self.configuration.platform))
        if branches is None:
            branches = self.release_config[self.RELEASE_DEFAULT_BRANCHES_KEY]

        return branches

    def get_build_task(self, platform):
        pass

    def process_target(self, config):
        pass

    def publish_result(self, config):
        pass

    def on_execute(self):
        config = self.configuration

        try:
            self.link_to_ticket()

            logging.info("[1/3] Processing target")
            if not config.uniota:
                self.process_target(config)
                logging.info("[1/3] Processing target completed")
            else:
                logging.info("[1/3] Skip processing for uniota target")

            logging.info("[2/3] Publishing results")
            if not config.uniota:
                self.publish_result(config)
                logging.info("[2/3] Publishing completed")
            else:
                logging.info("[2/3] Skip publishing for uniota target")

            # still, all targets should be in target list
            logging.info("[3/3] Editing target list")
            self.edit_targets_list(config)
            logging.info("[3/3] Editing completed")

            logging.info("[3/3] All done!")
        except Exception as e:
            logging.info("Error occurred, notifying current duty")
            message = "**Не удалось создать таргет**\n\n!!{}!!\n\nhttps://sandbox.yandex-team.ru/task/{}/view".format(
                str(e),
                self.id
            )
            notify_duty(message, config.ticket, self.secret)
            set_ticket_status("failed", config.ticket, self.secret.data()["st-token"])
            raise
