import logging
import os
import codecs

import sandbox.common.types.task as ctt

from sandbox.common.errors import TaskFailure
from sandbox.projects.tv.targetator.common import Targetator, _safe_read


class CvBuildInfo(object):
    def __init__(self, mi_conf, platform_dir, wifi):
        self.mi_conf = mi_conf
        self.platform_dir = platform_dir
        self.wifi = wifi


class TargetatorCultraview(Targetator):

    STAGE_TWO_ENABLED = True

    BUILD_REPO_PATH = "build"
    STBMI_REPO_PATH = "stbmi"
    BUILD_REPO = "mtk9632/dtv/platform/build"
    STBMI_REPO = "mtk9632/dtv/fusion-bsp2/mstar/stbmi"
    PLATFORM_REPO = "mtk9632/dtv/fusion-an/platform/vendor/mstar/{}"
    PROP_PATTERN = "ro.product.{}"
    CLID_PATTERN = "yndx.config.{}"
    SUB_CUSTOMER_FOLDER_KEY = "SUB_CUSTOMER_MODEL"

    SUPPORTED_PLATFORMS = {
        "cv9632": {
            "repo_path": "m7332",
            "build_task": "android_tv_Cultraview9632_SandboxRomCultraview9632user"
        },
        "cv6681": {
            "repo_path": "m5621",
            "build_task": "android_tv_Cultraview6681_SandboxRomCultraview6681user"
        },
    }

    class Requirements(Targetator.Requirements):
        semaphores = ctt.Semaphores(acquires=[ctt.Semaphores.Acquire(name='CV_TARGETATOR_SEMAPHORE')])

    def set_property(self, data, prop, value, kv_separator, line_separator, append_to_end=False):
        start_pos = data.find(prop + kv_separator)
        if start_pos == -1:
            separator = line_separator if line_separator != "\n" else ""
            if append_to_end:
                data = data + "\n{}{}{}{}".format(prop, kv_separator, value, separator)
            else:
                data = "{}{}{}{}\n".format(prop, kv_separator, value, separator) + data
        else:
            end_pos = data.find(line_separator, start_pos)
            data = data[:start_pos + len(prop) + len(kv_separator)] + value + data[end_pos:]
        return data

    def remove_property(self, data, prop):
        start_pos = data.find(prop)
        if start_pos == -1:
            return data
        end_pos = data.find('\n', start_pos)
        if end_pos == -1:
            return data[:start_pos]
        else:
            return data[:start_pos] + data[end_pos:]

    def get_platform_repo_path(self, platform):
        return self.SUPPORTED_PLATFORMS[platform]["repo_path"]

    def get_build_task(self, platform):
        return self.SUPPORTED_PLATFORMS[platform]["build_task"]

    def create_in_build(self, config, build_repo_path):
        logging.info("Making changes in build repo")

        customers_folder = "{}/build_config".format(build_repo_path)
        path = "{}/{}".format(
            customers_folder,
            [x for x in os.listdir(customers_folder) if x.upper() == config.customer][0]
        )

        target_files = [fn for fn in os.listdir(path) if config.target in fn or fn in config.target]
        if len(target_files) > 1:
            raise TaskFailure("Ambiguous target file in build repo")
        elif len(target_files) == 0:
            raise TaskFailure("No target file found in build repo")

        target_file = target_files[0]
        logging.info("Using " + target_file)

        with codecs.open("{}/{}".format(path, target_file), 'r+', encoding='utf-8', errors='ignore') as file:
            data = _safe_read(file)
            file.seek(0)

            mi_conf = self.find_in_config(data, "miconf")
            subcustomer_model = self.find_in_config(data, "subCustomerModel")
            wifi = self.find_in_config(data, "WifiModule")

            data = self.set_property(data, "FINGERPRINT_BRAND", config.props.brand, kv_separator=":", line_separator=";")
            data = self.set_property(data, "FINGERPRINT_PRODUCT", config.props.device, kv_separator=":", line_separator=";")
            data = self.set_property(data, "FINGERPRINT_DEVICE", config.props.device, kv_separator=":", line_separator=";")

            vc = config.vendor_config
            if vc is not None:
                vendor_overlay = " ".join([x for x in [vc.pult, vc.banners, vc.feedback] if x is not None])
                if len(vendor_overlay) > 0:
                    data = self.set_property(data, "vendorOverlays", vendor_overlay, kv_separator=":", line_separator=";")
                else:
                    data = self.remove_property(data, "vendorOverlays")
            else:
                data = self.remove_property(data, "vendorOverlays")
                logging.info("No vendor config")

            file.write(data)
            file.truncate()

        logging.info("Done")
        return CvBuildInfo(mi_conf, subcustomer_model, wifi)

    def create_in_stbmi(self, config, stbmi_repo_path, mi_conf):
        logging.info("Making changes in stbmi repo")

        miconf_path = "{}/miapp/midemo/build/cfg/{}".format(stbmi_repo_path, mi_conf)
        with codecs.open(miconf_path, 'r+', encoding='utf-8', errors='ignore') as file:
            data = _safe_read(file)
            customer = self.find_in_miconf(data, 'CUSTOMER_MODEL')
            target_folder = self.find_in_miconf(data, 'SUB_CUSTOMER_MODEL')
            board = self.find_in_miconf(data, 'CFG01_BOARD')

        logging.info("Using " + target_folder)
        path = "{}/mi/platform/common/ini/{}/{}/misc".format(stbmi_repo_path, customer, target_folder)
        with codecs.open("{}/boot.ini".format(path), 'r+', encoding='utf-8', errors='ignore') as file:
            data = _safe_read(file)
            file.seek(0)

            data = self.set_property(data, "MUSIC_ON", "0", kv_separator=" = ", line_separator=";")
            data = self.remove_property(data, "MUSIC_NAME1")
            data = self.remove_property(data, "CAP1_NAME")
            data = self.remove_property(data, "CAP2_NAME")

            file.write(data)
            file.truncate()

        self.safe_remove("{}/boot_error.jpg".format(path))
        self.safe_remove("{}/boot_recovery.jpg".format(path))
        self.safe_remove("{}/boot0.jpg".format(path))
        self.safe_remove("{}/boot1.jpg".format(path))

        self.load_image(config.boot_logo, "{}/boot0.jpg".format(path))
        self.load_image(config.error_logo, "{}/boot_error.jpg".format(path))
        self.load_image(config.recovery_logo, "{}/boot_recovery.jpg".format(path))

        logging.info("Done")
        return board

    def create_in_platform(self, config, platform_repo_path, target_folder):
        logging.info("Making changes in platform repo")

        prop_path = "{}/CustomerConfig/ZEASN/{}/preinstall/ctvbuild.prop".format(platform_repo_path, target_folder)
        logging.info("Using " + prop_path)
        with codecs.open(prop_path, 'r+', encoding='utf-8', errors='ignore') as file:
            data = _safe_read(file)
            file.seek(0)

            if data[-1] != '\n':
                data += '\n'

            if config.props.device is not None:
                data = self.set_property(data, self.PROP_PATTERN.format("device"), config.props.device, kv_separator="=", line_separator="\n")
            if config.props.model is not None:
                data = self.set_property(data, self.PROP_PATTERN.format("model"), config.props.model, kv_separator="=", line_separator="\n")
            if config.props.manufacturer is not None:
                data = self.set_property(data, self.PROP_PATTERN.format("manufacturer"), config.props.manufacturer, kv_separator="=", line_separator="\n")
            if config.props.brand is not None:
                data = self.set_property(data, self.PROP_PATTERN.format("brand"), config.props.brand, kv_separator="=", line_separator="\n")
            if config.props.name is not None:
                data = self.set_property(data, self.PROP_PATTERN.format("name"), config.props.name, kv_separator="=", line_separator="\n")

            if config.clids is not None:
                if config.clids.clid1 is not None:
                    data = self.set_property(data, self.CLID_PATTERN.format("clid1"), str(config.clids.clid1), kv_separator="=", line_separator="\n", append_to_end=True)
                if config.clids.clid100014 is not None:
                    data = self.set_property(data, self.CLID_PATTERN.format("clid100014"), str(config.clids.clid100014), kv_separator="=", line_separator="\n", append_to_end=True)
                if config.clids.clid100010 is not None:
                    data = self.set_property(data, self.CLID_PATTERN.format("clid100010"), str(config.clids.clid100010), kv_separator="=", line_separator="\n", append_to_end=True)
                if config.clids.clid100011 is not None:
                    data = self.set_property(data, self.CLID_PATTERN.format("clid100011"), str(config.clids.clid100011), kv_separator="=", line_separator="\n", append_to_end=True)
                if config.clids.clid100012 is not None:
                    data = self.set_property(data, self.CLID_PATTERN.format("clid100012"), str(config.clids.clid100012), kv_separator="=", line_separator="\n", append_to_end=True)
                if config.clids.clid100013 is not None:
                    data = self.set_property(data, self.CLID_PATTERN.format("clid100013"), str(config.clids.clid100013), kv_separator="=", line_separator="\n", append_to_end=True)
            else:
                self._clear_clids(data)
                logging.info("No clids")

            if data[-1] != '\n':
                data += '\n'

            file.write(data)
            file.truncate()

        logging.info("Done")

    def find_in_config(self, data, key):
        start = data.find(key)
        if start < 0:
            raise TaskFailure("Failed to find {} in config".format(key))
        end = data.find(";", start)
        return data[start + len(key) + 1: end]

    def find_in_miconf(self, data, key):
        start = data.find(key)
        if start < 0:
            raise TaskFailure("Failed to find {} in config".format(key))
        end = data.find("\n", start)
        return data[start + len(key) + 1: end]

    def safe_remove(self, path):
        try:
            os.remove(path)
        except Exception:
            pass

    def _clear_clids(self, data):
        while True:
            saved_data = data
            data = self.remove_property(saved_data, self.CLID_PATTERN.format(""))
            if saved_data == data:
                break

    def process_target(self, config):
        self.repo_initialize(self.BUILD_REPO, self.BUILD_REPO_PATH)
        self.repo_initialize(self.STBMI_REPO, self.STBMI_REPO_PATH)
        platform_repo_path = self.get_platform_repo_path(config.platform)
        self.repo_initialize(self.PLATFORM_REPO.format(platform_repo_path), platform_repo_path)
        self.make_changes(config, self.BUILD_REPO_PATH, self.STBMI_REPO_PATH, platform_repo_path)

    def make_changes(self, config, build_repo_path, stbmi_repo_path, platform_repo_path):
        build_info = self.create_in_build(config, build_repo_path)
        board = self.create_in_stbmi(config, stbmi_repo_path, build_info.mi_conf)
        self.create_in_platform(config, platform_repo_path, build_info.platform_dir)

        wifi = build_info.wifi
        if board != config.uniota_props.board or wifi != config.uniota_props.wifi:
            raise TaskFailure(
                "Uniota props check failed. " +
                "expected: [ board={}, wifi={} ] ".format(config.uniota_props.board, config.uniota_props.wifi) +
                "actual: [ board={}, wifi={} ]".format(board, wifi)
            )

    def publish_result(self, config):
        self.gerrit_commit_and_push(self.BUILD_REPO_PATH)
        self.gerrit_commit_and_push(self.STBMI_REPO_PATH)
        self.gerrit_commit_and_push(self.get_platform_repo_path(config.platform))
