import filecmp
import logging
import os
import re
import sys

import sandbox.common.types.task as ctt

from sandbox.common.errors import TaskFailure
from sandbox.sdk2.helpers import ProcessLog
from sandbox.sdk2.helpers import subprocess as sp

from sandbox.projects.tv.targetator.common import Targetator, TargetatorException


class TargetatorCVTE(Targetator):

    STAGE_TWO_ENABLED = True

    REPO_PATTERN = "{}/customers"
    PROP_PATTERN = "#define {0: <64}"
    PROP_SHORT_PATTERN = "#define {0} "
    LABEL_PATTERN = "// {0}\n"
    CLID_PATTERN = "CVT_DEF_YNDX_CLID{}"
    LOGO_PATTERN = "ID_LOGO_{}"
    PLATFORM_REPO = "{}/customers"

    SUPPORTED_PLATFORMS = {
        "cvte351": {
            "repo_path": "cvte351",
            "build_task": "android_tv_Hisilicon351_SandboxRomCvte351user"
        },
        "cvte6681": {
            "repo_path": "cvte9632",
            "build_task": "yandex_android_tv_Hisilicon6681_SandboxRomCvte6681user"
        },
        "cvte9632": {
            "repo_path": "cvte9632",
            "build_task": "yandex_android_tv_Hisilicon9632_SandboxRomCvte9632user"
        },
        "cvte9256": {
            "repo_path": "cvte9256",
            "build_task": "yandex_android_tv_cvte9256_user"
        },
        "cvte9632_11": {
            "repo_path": "cvte9256",
            "build_task": "yandex_android_tv_Mtk963211_SandboxRomCvte963211user"
        },
    }

    class Requirements(Targetator.Requirements):
        semaphores = ctt.Semaphores(acquires=[ctt.Semaphores.Acquire(name='CVTE_TARGETATOR_SEMAPHORE')])

    def get_platform_repo_path(self, platform):
        return self.SUPPORTED_PLATFORMS[platform]["repo_path"]

    def get_build_task(self, platform):
        return self.SUPPORTED_PLATFORMS[platform]["build_task"]

    def set_property(self, data, prop, value, escape=True):
        logging.info("set_property " + prop + " to " + value)

        data = self.remove_property(data, prop)
        prop = self.PROP_PATTERN.format(prop)

        pattern = "{}\"{}\"\n" if escape else "{}{}\n"
        data = pattern.format(prop, value) + data

        return data

    def remove_property(self, data, prop):
        logging.info("remove_property " + prop)
        prop = self.PROP_SHORT_PATTERN.format(prop)

        while True:
            start_pos = data.find(prop)

            if start_pos == -1:
                break

            end_pos = data.find('\n', start_pos)

            if end_pos == -1:
                data = data[:start_pos]
            else:
                # +1 to remove new line
                data = data[:start_pos] + data[end_pos+1:]

        return data

    def set_label(self, data, label):
        logging.info("set_label " + label)

        data = self.remove_label(data, label)
        label = self.LABEL_PATTERN.format(label)

        data = label + data

        return data

    def remove_label(self, data, label, recursive=True):
        logging.info("remove_label " + label)
        label = self.LABEL_PATTERN.format(label)

        while True:
            start_pos = data.find(label)

            if start_pos == -1:
                break

            end_pos = data.find('\n', start_pos)

            if end_pos == -1:
                data = data[:start_pos]
            else:
                data = data[:start_pos] + data[end_pos+1:]

            if not recursive:
                break

        return data

    def get_logo_id(self, platform_repo, config):
        tmp_file = "tmp.jpg"
        self.load_image(config.boot_logo, tmp_file)

        logo_path = "{}/customer/customer_{}/logo".format(platform_repo, config.customer.lower())
        enum_file = "{}/enum_list/id_logo.h".format(platform_repo)

        files = os.listdir(logo_path)
        result_file = ""
        for file in files:
            if filecmp.cmp(os.path.join(logo_path, file), tmp_file, shallow=False):
                result_file = file
                break

        with open(enum_file, 'r+') as enum_file:
            enum_data = enum_file.read()
            enum_file.seek(0)

            tmp_logo = "{}_{}".format(
                config.customer.upper(),
                config.boot_logo.replace("bootlogo_", "")
                .replace(".jpg", "").replace("-", "_").upper()
            )
            logo_id = self.LOGO_PATTERN.format(tmp_logo)

            start = enum_data.find(logo_id)
            logging.info("found " + logo_id + " at " + str(start))
            end = enum_data.find('\n', enum_data.rfind(logo_id))

            if start != -1:
                logos = [x for x in re.split('[ ,\n]*', enum_data[start:end]) if
                         len(x) > 0]
                logos += [logo_id]
                logos = list(set(logos))
                logos.sort()
                logging.info("logos " + str(logos))
                logos_txt = ",\n    ".join(logos) + ","

                enum_file.write(enum_data[:start] + logos_txt + enum_data[end:])
            else:
                pos = enum_data.find("} enum_LOGO;")
                enum_file.write(
                    enum_data[:pos] + "    {},\n".format(logo_id) + enum_data[
                                                                    pos:])

            enum_file.truncate()

            if len(result_file) == 0:
                try:
                    with ProcessLog(self, logger=logging.getLogger('Image adding')) as pl:
                        sp.check_call(
                            "mv {} {}".format(tmp_file, os.path.join(logo_path, "{}.jpg".format(logo_id))),
                            shell=True,
                            stderr=pl.stderr,
                            stdout=pl.stdout
                        )
                        prev_dir = os.getcwd()
                        os.chdir(platform_repo)
                        tmp_path = logo_path.replace("{}/".format(platform_repo), "")
                        sp.check_call(
                            "git add {}".format(os.path.join(tmp_path, "{}.jpg".format(logo_id))),
                            shell=True,
                            stderr=pl.stderr,
                            stdout=pl.stdout
                        )
                        os.chdir(prev_dir)
                except sp.CalledProcessError as e:
                    logging.error('image copying failed: %s', e)
                    raise TargetatorException('image copying failed', e, sys.exc_info()[2])
            else:
                logo_id = result_file.replace(".jpg", "")

        return logo_id

    def process_target(self, config):
        logging.info("Preparing platform ({}) repo".format(config.platform))

        platform_repo = self.get_platform_repo_path(config.platform)

        self.repo_initialize(self.PLATFORM_REPO.format(platform_repo), platform_repo)

        path = "{0}/customer/customer_{1}/customer_{1}.h".format(platform_repo, config.customer.lower())

        with open(path, 'r+') as file:
            data = file.read()
            file.seek(0)

            target_match = re.search("if\s*\(\s*IsModelID\({}\)".format(config.target), data)

            if target_match == None:
                raise TaskFailure("Target not found: {}".format(config.target))

            target_start = target_match.span()[0]

            target_start = data.find('\n', target_start) + 1

            target_end = data.find("#el", target_start + 1)

            target = data[target_start:target_end]

            target = self.set_property(target, "CVT_DEF_LOGO_TYPE",
                                       self.get_logo_id(platform_repo, config),
                                       escape=False)
            target = self.set_label(target, "logo")

            vc = config.vendor_config
            if vc is not None:
                vendor_overlay = " ".join(
                    [x for x in [vc.pult, vc.banners, vc.feedback] if
                     x is not None])
                if len(vendor_overlay) > 0:
                    test_prop = self.PROP_PATTERN.format(
                        "CVT_DEF_YNDX_VENDORCONFIG_OVERLAYS")
                    if target.find(test_prop) == -1:
                        target = '\n' + target

                    target = self.set_property(target,
                                               "CVT_DEF_YNDX_VENDORCONFIG_OVERLAYS",
                                               vendor_overlay)
                    target = self.set_label(target, "overlay")
                else:
                    target = self.remove_property(target,
                                                  "CVT_DEF_YNDX_VENDORCONFIG_OVERLAYS")
                    target = self.remove_label(target, "overlay")
                    logging.info("Vendor config is empty")
            else:
                target = self.remove_property(target,
                                              "CVT_DEF_YNDX_VENDORCONFIG_OVERLAYS")
                target = self.remove_label(target, "overlay")
                logging.info("No vendor config")

            clids = config.clids
            if clids is not None:
                # Mind reversed nature of file filling
                if clids.clid100013 is not None:
                    target = self.set_property(target, self.CLID_PATTERN.format(
                        "100013"), str(clids.clid100013), escape=False)
                if clids.clid100012 is not None:
                    target = self.set_property(target, self.CLID_PATTERN.format(
                        "100012"), str(clids.clid100012), escape=False)
                if clids.clid100011 is not None:
                    target = self.set_property(target, self.CLID_PATTERN.format(
                        "100011"), str(clids.clid100011), escape=False)
                if clids.clid100010 is not None:
                    target = self.set_property(target, self.CLID_PATTERN.format(
                        "100010"), str(clids.clid100010), escape=False)
                if clids.clid100014 is not None:
                    target = self.set_property(target, self.CLID_PATTERN.format(
                        "100014"), str(clids.clid100014), escape=False)
                if clids.clid1 is not None:
                    target = self.set_property(target,
                                               self.CLID_PATTERN.format(""),
                                               str(clids.clid1), escape=False)
                target = self.set_label(target, "clids")
            else:
                target = self.remove_label(target, "clids")
                self._clear_clids(data)

            test_prop = self.PROP_PATTERN.format("CVT_DEF_PRODUCT_BRAND")
            if target.find(test_prop) == -1:
                target = '\n' + target

            # Mind reversed nature of file filling
            target = self.set_property(target, "CVT_DEF_PRODUCT_MANUFACTURER",
                                       config.props.manufacturer)
            target = self.set_property(target, "CVT_DEF_PRODUCTION_MODEL",
                                       config.props.model)
            target = self.set_property(target, "CVT_DEF_PRODUCT_DEVICE",
                                       config.props.device)
            target = self.set_property(target, "CVT_DEF_PRODUCT_NAME",
                                       config.props.name)
            target = self.set_property(target, "CVT_DEF_PRODUCT_BRAND",
                                       config.props.brand)
            target = self.set_label(target, "build properties")

            data = data[:target_start] + target + data[target_end:]

            # remove all empty lines or empty lines with spaces
            data = "".join([s for s in data.strip().splitlines(True) if s.strip()])

            file.write(data)
            file.truncate()

        logging.info("Done")

    def _clear_clids(self, data):
        while True:
            saved_data = data
            data = self.remove_property(saved_data,
                                        self.CLID_PATTERN.format(""))
            if saved_data == data:
                break

    def publish_result(self, config):
        self.gerrit_commit_and_push(self.get_platform_repo_path(config.platform))
