# coding=utf-8
import datetime
import json
import os
import re

import requests

import sandbox.projects.common.binary_task as binary_task

from sandbox import sdk2

from sandbox.projects.common.vcs.arc import Arc
from sandbox.projects.common.binary_task import LastRefreshableBinary

from sandbox.projects.tv.common import CLIDS_BY_BRAND, notify_duty, get_resource_search_arguments, set_ticket_status
from sandbox.projects.tv.common import DEFAULT_ROBOT_KEY
from sandbox.projects.tv.common import Config
from sandbox.projects.tv.common import ConfigParsingException


class Cyberdyne(LastRefreshableBinary, sdk2.Task):

    AUTOMATION_START_MARKER = "<{targetator_automation\n%%(json)\n"
    AUTOMATION_END_MARKER = "%%"

    PULT_DEFAULT = "default"

    TV_FIRMWARE_REPO_DIR = "smart_devices/tv/platforms"

    class Parameters(sdk2.Parameters):
        _lbrp = binary_task.binary_release_parameters(stable=True)
        push_tasks_resource = True

        ticket = sdk2.parameters.String("Ticket", required=True)

    @property
    def binary_executor_query(self):
        return get_resource_search_arguments(self)

    def on_execute(self):
        secret = sdk2.yav.Secret(DEFAULT_ROBOT_KEY)
        arc_token = secret.data()["arc-token"]
        st_token = secret.data()["st-token"]
        try:
            arc_client = Arc(arc_oauth_token=arc_token)

            mount_point = arc_client.mount_path(None, None, fetch_all=False)
            mp = mount_point._work_path

            description = requests.get(
                "https://st-api.yandex-team.ru/v2/issues/{}".format(self.Parameters.ticket),
                headers={'Authorization': 'OAuth {}'.format(st_token)}
            ).json()["description"]

            marker_start = description.find(self.AUTOMATION_START_MARKER) + len(self.AUTOMATION_START_MARKER)
            marker_end = description.find(self.AUTOMATION_END_MARKER, marker_start)

            config = json.loads(description[marker_start:marker_end])

            config["boot_logo"] = config["boot_logo"].lower()
            config["ticket"] = self.Parameters.ticket
            config["target"] = config["target"].replace('.txt', '').replace('.cfg', '')

            if "vendor_config" in config:
                if config["vendor_config"]["pult"] == self.PULT_DEFAULT:
                    del config["vendor_config"]["pult"]
                if not config["vendor_config"]:
                    del config["vendor_config"]

            brand = config["props"]["brand"]
            if brand.upper() in CLIDS_BY_BRAND:
                clids = CLIDS_BY_BRAND[brand.upper()]

                if len(clids) > 0:
                    config["clids"] = clids
            else:
                notify_duty(
                    "Нет клидов для бренда {}\n\nhttps://sandbox.yandex-team.ru/task/{}/view".format(brand.upper(), self.id),
                    self.Parameters.ticket,
                    secret
                )
                raise ConfigParsingException("no clids for: {}".format(brand))

            platform, board, _ = re.split(r'(\d+_?\d*)', config["platform"])
            if platform == "rt":
                platform = "hikeen"
            if platform == "cvte" and board != "351":
                config["customer"] = "YANDEX_UNI"

            config["last_modified"] = datetime.datetime.now().strftime('%Y-%m-%dT%H:%M:%S')

            new_config_path_relative = os.path.join(platform, board, "{}.json".format(config["sw_name"]))
            new_config_path = os.path.join(mp, self.TV_FIRMWARE_REPO_DIR, new_config_path_relative)
            new_config = Config.from_json(config, config["sw_name"])

            arc_client.checkout(mp, new_config.ticket, True)

            with open(new_config_path, 'w+') as file:
                file.write(json.dumps(new_config.to_json(), indent=4, separators=(',', ': ')))
                file.write('\n')

            message = "[{}] Added config for {} {} {}".format(
                new_config.ticket,
                new_config.customer,
                new_config.props.brand,
                new_config.props.name
            )

            arc_client.add(mp, new_config_path)
            arc_client.commit(mp, message)
            arc_client.push(mp, "users/robot-edi/{}".format(new_config.ticket), force=True)
            arc_client.pr_create(mp, "trunk", message, publish=True, auto=True)

            mount_point.unmount()
        except Exception:
            set_ticket_status("failed", self.Parameters.ticket, st_token)
            raise
