import json
import logging
import os

from collections import OrderedDict

import sandbox.common.types.task as ctt

from sandbox.common.errors import TaskFailure
from sandbox.sdk2.helpers import ProcessLog
from sandbox.sdk2.helpers import subprocess as sp

from sandbox.projects.tv.targetator.common import Targetator


class TargetatorHikeen(Targetator):

    STAGE_TWO_ENABLED = True

    PLATFORM_REPO_PATH = "platform"
    PLATFORM_REPO = "hikeen2842/platform"
    DIRECTORY_PATH = "hikeen/order"
    LOGO_PATH = "platform/hikeen/order/{}/resources/bootfile.raw"
    LOGO_PATH_FOR_GIT = "hikeen/order/{}/resources/bootfile.raw"
    JSON_PATH = "platform/hikeen/order/{}/order.json"
    MAKEFILE_PATH = "platform/hikeen/order/{}/device.mk"
    OVERLAY_PATH = "vendor/yandex/packages/VendorConfig/{}/overlay"

    BUILD_TASK = "yandex_android_tv_HikeenBuilds_Realtek2842_SandboxRomHikeenRt2842user"

    YANDEX_LOGO = "bootlogo_yandex.jpg"

    LOGO_SIZE = (1280, 720)

    class Requirements(Targetator.Requirements):
        semaphores = ctt.Semaphores(acquires=[ctt.Semaphores.Acquire(name="HIKEEN_TARGETATOR_SEMAPHORE")])

    def get_build_task(self, platform):
        return self.BUILD_TASK

    def update_logo(self, config):
        logging.info("Updating logo")

        if config.boot_logo == self.YANDEX_LOGO:
            # Remove the boot logo file to use the default boot logo so it's easier to update it in future
            try:
                os.remove(self.LOGO_PATH.format(config.target))
            except Exception:
                pass
        else:
            tmp_file = "tmp.jpg"
            self.load_image(config.boot_logo, tmp_file)

            from PIL import Image

            with Image.open(tmp_file) as image:
                image_resized = image.resize(self.LOGO_SIZE, resample=Image.BICUBIC)
                with open(self.LOGO_PATH.format(config.target), "wb") as file:
                    file.write(image_resized.tobytes())

            with ProcessLog(self, logger=logging.getLogger("Adding logo to git")) as pl:
                sp.check_call(
                    "git -C {} add {}".format(self.PLATFORM_REPO_PATH, self.LOGO_PATH_FOR_GIT.format(config.target)),
                    shell=True,
                    stderr=pl.stderr,
                    stdout=pl.stdout
                )

    def update_json(self, config):
        logging.info("Updating Json")

        with open(self.JSON_PATH.format(config.target), "r") as file:
            data = json.load(file, object_pairs_hook=OrderedDict)

        # Remove the properties from order.json to have a single definition in device.mk

        try:
            product_node = data["config"]["order"]["-ro"]["product"]

            product_node.pop("brand", None)
            product_node.pop("name", None)
            product_node.pop("device", None)
            product_node.pop("model", None)
            product_node.pop("manufacturer", None)
        except Exception:
            pass

        try:
            config_node = data["config"]["order"]["-yndx"]["config"]

            config_node.pop("clid1", None)
            config_node.pop("clid100010", None)
            config_node.pop("clid100011", None)
            config_node.pop("clid100012", None)
            config_node.pop("clid100013", None)
            config_node.pop("clid100014", None)
        except Exception:
            pass

        with open(self.JSON_PATH.format(config.target), "w") as file:
            file.write(json.dumps(data, indent=4, separators=(",", ": ")))
            file.write("\n")

    def update_makefile(self, config):
        logging.info("Updating makefile")

        with open(self.MAKEFILE_PATH.format(config.target), "r") as file:
            data = file.read()

        data = self.update_vendor_overlay(config, data)
        data = self.update_product_variables(config, data)
        data = self.update_yandex_properties(config, data)

        with open(self.MAKEFILE_PATH.format(config.target), "w") as file:
            file.write(data)

    def update_vendor_overlay(self, config, data):
        logging.info("Updating vendor overlay")

        vc = config.vendor_config
        if vc is not None:
            vendor_overlay = " ".join([self.OVERLAY_PATH.format(x) for x in [vc.pult, vc.banners, vc.feedback] if x is not None])
            if len(vendor_overlay) > 0:
                vendor_overlay = "PRODUCT_PACKAGE_OVERLAYS += " + vendor_overlay + "\n"
        else:
            vendor_overlay = ""

        start = data.find("PRODUCT_PACKAGE_OVERLAYS +=")

        if start == -1:
            data = vendor_overlay + data
        else:
            end = data.find("\n", start)
            data = data[:start] + vendor_overlay + data[end + 1:]

        return data

    def update_product_variables(self, config, data):
        logging.info("Updating product variables")

        data = self.set_variable(data, "HIKEEN_PRODUCT_BRAND", config.props.brand)
        data = self.set_variable(data, "HIKEEN_PRODUCT_NAME", config.props.name)
        data = self.set_variable(data, "HIKEEN_PRODUCT_DEVICE", config.props.device)
        data = self.set_variable(data, "HIKEEN_PRODUCT_MODEL", config.props.model)
        data = self.set_variable(data, "HIKEEN_PRODUCT_MANUFACTURER", config.props.manufacturer)

        return data

    def set_variable(self, data, name, value):
        start = data.find(name + " =")

        if start == -1:
            last_statement_start = data.rfind("HIKEEN_PRODUCT_")

            if last_statement_start == -1:
                last_statement_start = data.find("PRODUCT_PACKAGE_OVERLAYS +=")

            if last_statement_start == -1:
                last_statement_end = -1
            else:
                last_statement_end = data.find("\n", last_statement_start)

            data = data[:last_statement_end + 1] + "{} = {}\n".format(name, value) + data[last_statement_end + 1:]
        else:
            end = data.find("\n", start)
            data = data[:start] + "{} = {}\n".format(name, value) + data[end + 1:]

        return data

    def update_yandex_properties(self, config, data):
        logging.info("Updating Yandex properties")

        start = data.find("HIKEEN_BUILD_PROPERTIES +=")

        if start == -1:
            manufacturer_start = data.find("HIKEEN_PRODUCT_MANUFACTURER =")
            manufacturer_end = data.find("\n", manufacturer_start)
            data = data[:manufacturer_end + 1] + "HIKEEN_BUILD_PROPERTIES +=\n" + data[manufacturer_end + 1:]
            start = manufacturer_end + 1

        end = data.find("\n", start)

        while data[start:end].rstrip().endswith("\\"):
            end = data.find("\n", end + 1)

            if end == -1:
                raise TaskFailure("Incorrect properties structure")

        properties = data[start:end + 1]

        # Remove the properties from HIKEEN_BUILD_PROPERTIES to have a single definition in the HIKEEN_PRODUCT_... variables

        properties = self.remove_property(properties, "ro.product.brand")
        properties = self.remove_property(properties, "ro.product.name")
        properties = self.remove_property(properties, "ro.product.device")
        properties = self.remove_property(properties, "ro.product.model")
        properties = self.remove_property(properties, "ro.product.manufacturer")

        if config.clids is not None:
            properties = self.set_property(properties, "yndx.config.clid1", config.clids.clid1)
            properties = self.set_property(properties, "yndx.config.clid100010", config.clids.clid100010)
            properties = self.set_property(properties, "yndx.config.clid100011", config.clids.clid100011)
            properties = self.set_property(properties, "yndx.config.clid100012", config.clids.clid100012)
            properties = self.set_property(properties, "yndx.config.clid100013", config.clids.clid100013)
            properties = self.set_property(properties, "yndx.config.clid100014", config.clids.clid100014)
        else:
            properties = self.remove_property(properties, "yndx.config.clid1")
            properties = self.remove_property(properties, "yndx.config.clid100010")
            properties = self.remove_property(properties, "yndx.config.clid100011")
            properties = self.remove_property(properties, "yndx.config.clid100012")
            properties = self.remove_property(properties, "yndx.config.clid100013")
            properties = self.remove_property(properties, "yndx.config.clid100014")

        data = data[:start] + properties + data[end + 1:]

        return data

    def set_property(self, data, prop, value):
        start = data.find(prop + "=")

        if start == -1:
            if data.rstrip().endswith("\\"):
                data = data[:-1] + "\n    {}={}\n".format(prop, value)
            else:
                data = data[:-1] + " \\\n    {}={}\n".format(prop, value)
        else:
            start = data.rfind("\n", 0, start) + 1
            end = data.find("\n", start)
            has_backslash = "\\" in data[start:end]
            pattern = "    {}={} \\\n" if has_backslash else "    {}={}\n"
            data = data[:start] + pattern.format(prop, value) + data[end + 1:]

        return data

    def remove_property(self, data, prop):
        start = data.find(prop + "=")

        if start != -1:
            start = data.rfind("\n", 0, start) + 1
            end = data.find("\n", start)

            if end == len(data) - 1:
                backslash_index = data.rfind("\\", 0, start)
                data = data[:backslash_index].rstrip() + "\n"
            else:
                data = data[:start] + data[end + 1:]

        return data

    def process_target(self, config):
        logging.info("Preparing platform repo")

        self.repo_initialize_sparse(self.PLATFORM_REPO, self.PLATFORM_REPO_PATH, self.DIRECTORY_PATH)

        self.update_logo(config)
        self.update_json(config)
        self.update_makefile(config)

        logging.info("Done")

    def publish_result(self, config):
        self.gerrit_commit_and_push(self.PLATFORM_REPO_PATH)
