import logging
import os
import requests

import sandbox.common.types.client as ctc
import sandbox.common.types.task as ctt
import sandbox.common.types.misc as ctm

import sandbox.projects.common.binary_task as binary_task
from sandbox import sdk2
from sandbox.sdk2.vcs.svn import Arcadia

from sandbox.projects.common.binary_task import LastRefreshableBinary

from sandbox.projects.tv.common import DEFAULT_ROBOT_KEY, get_resource_search_arguments
from sandbox.projects.tv.targetator.cv import TargetatorCultraview
from sandbox.projects.tv.targetator.cvte import TargetatorCVTE
from sandbox.projects.tv.targetator.hikeen import TargetatorHikeen
from sandbox.projects.tv.targetator.undertaker import TargetatorUndertaker


class TvSkynet(LastRefreshableBinary, sdk2.Task):

    ARC_PATH = "arcadia"
    STORE_PATH = "store"
    TV_FIRMWARE_REPO_DIR = "smart_devices/tv/platforms"

    TAG_ADDED = 'A'
    TAG_MODIFIED = 'M'
    TAG_DELETED = 'D'

    TASK_DESCRIPTION_PATTERN = "Target: {}\n\nReason: {}"

    TARGETATOR_BY_PLATFORM = {
        "cv": TargetatorCultraview,
        "cvte": TargetatorCVTE,
        "hikeen": TargetatorHikeen
    }

    KEYWORD_SKYNET_SKIP = "[skynet:skip]"

    class Requirements(sdk2.Requirements):
        dns = ctm.DnsType.DNS64
        client_tags = ctc.Tag.LINUX_BIONIC

        # Multislot requirements
        cores = 2
        ram = 4096

    class Parameters(sdk2.Parameters):
        _lbrp = binary_task.binary_release_parameters(stable=True)
        push_tasks_resource = True

        revision = sdk2.parameters.Integer("Change revision", required=True)
        flow_url = sdk2.parameters.String("Related flow url")
        pr_id = sdk2.parameters.Integer("Related PR id")
        path_filter = sdk2.parameters.String("Path filter", default="")
        trigger_targetators = sdk2.parameters.Bool("Trigger targetators", default=True)
        with sdk2.parameters.Output:
            changed_targets = sdk2.parameters.List("New and modified targets", default=[])

    @property
    def binary_executor_query(self):
        return get_resource_search_arguments(self)

    def on_execute(self):
        revision = self.Parameters.revision
        flow_url = self.Parameters.flow_url
        pr_id = self.Parameters.pr_id

        with self.memoize_stage.write_comment:
            if self.Parameters.trigger_targetators:
                if pr_id is not None and flow_url is not None and pr_id != 0 and len(str(flow_url)) > 0:
                    self.create_task_comment(str(flow_url), pr_id)

        new_and_edited, deleted = self.get_changed_files(revision)
        reason = self.get_log(revision)["msg"].split('\n\n')[0]

        logging.info("Revision: {}\nReason:{}".format(revision, reason))

        if self.KEYWORD_SKYNET_SKIP in reason:
            logging.info("Skip skynet")
            return

        self.Parameters.changed_targets = self.get_changed_targets(revision)

        if not self.Parameters.trigger_targetators:
            return

        with self.memoize_stage.undertakers:
            undertakers = [TargetatorUndertaker(
                self,
                description=self.TASK_DESCRIPTION_PATTERN.format(x.split('/')[-1].replace(".json", ""), reason),
                revision=revision,
                configuration_path=x
            ) for x in deleted]
            for undertaker in undertakers:
                undertaker.enqueue()
            raise sdk2.WaitTask(undertakers, [ctt.Status.Group.FINISH, ctt.Status.TIMEOUT, ctt.Status.EXCEPTION], wait_all=True)

        with self.memoize_stage.targetators:
            targetators = [TvSkynet.get_targetator_for_file(
                x,
                self.TASK_DESCRIPTION_PATTERN.format(x.split('/')[-1].replace(".json", ""), reason),
                self
            ) for x in new_and_edited]
            for targetator in targetators:
                targetator.enqueue()
            raise sdk2.WaitTask(targetators, [ctt.Status.Group.FINISH, ctt.Status.TIMEOUT, ctt.Status.EXCEPTION], wait_all=True)

    def get_changed_files(self, revision):
        changed_files = [x for x in self.get_log(revision)["paths"] if self.get_local_file_name(x[1]).split('/')[0] in self.TARGETATOR_BY_PLATFORM]

        new_and_added_targets = [self.get_local_file_name(x[1]) for x in changed_files if x[0] == self.TAG_ADDED or x[0] == self.TAG_MODIFIED]
        deleted_targets = [self.get_local_file_name(x[1]) for x in changed_files if x[0] == self.TAG_DELETED]

        logging.info("New and edited files: {}".format(new_and_added_targets))
        logging.info("Deleted files: {}".format(deleted_targets))

        return new_and_added_targets, deleted_targets

    def get_changed_targets(self, revision):
        changed_targets = []
        log = Arcadia.log(Arcadia.ARCADIA_TRUNK_URL, revision_from=revision, limit=1)[0]
        for tag, svn_path in log['paths']:
            if tag == self.TAG_ADDED or tag == self.TAG_MODIFIED:
                arc_path = TvSkynet.remove_prefix(svn_path, '/trunk/arcadia/')
                if arc_path.startswith(self.Parameters.path_filter):
                    tv_target = os.path.basename(arc_path).replace('.json', '')
                    changed_targets.append(tv_target)
        return changed_targets

    def get_log(self, revision):
        url = os.path.join(Arcadia.ARCADIA_TRUNK_URL, self.TV_FIRMWARE_REPO_DIR)
        return Arcadia.log(url, revision_from=revision, limit=1)[0]

    def get_local_file_name(self, file_name):
        return file_name.replace("/trunk/arcadia/smart_devices/tv/platforms/", "")

    @staticmethod
    def remove_prefix(str, prefix):
        if str.startswith(prefix):
            return str[len(prefix):]
        return str

    @staticmethod
    def get_targetator_class_for_file(file):
        """
        :param file: str, path relative to smart_devices/tv/platforms/
        """
        platform = file.split('/')[0]
        if platform in TvSkynet.TARGETATOR_BY_PLATFORM:
            return TvSkynet.TARGETATOR_BY_PLATFORM[platform]
        return None

    @staticmethod
    def get_targetator_for_file(file, reason, parent_task):
        """
        :param file: str, path relative to smart_devices/tv/platforms/
        :param reason: str, any comment
        :param parent_task: sdk2.Task or None
        """
        targetator = TvSkynet.get_targetator_class_for_file(file)
        return targetator(
            parent_task,
            description=TvSkynet.TASK_DESCRIPTION_PATTERN.format(file.split('/')[-1].replace(".json", ""), reason),
            configuration_path=file
        )

    def create_task_comment(self, flow_url, pr_id):
        self.secret = sdk2.yav.Secret(DEFAULT_ROBOT_KEY)

        sandbox_url = "https://sandbox.yandex-team.ru/task/{}/view".format(self.id)

        req_headers = {
            'Authorization': 'OAuth {}'.format(self.secret.data()["arc-token"]),
            'Content-Type': 'application/json'
        }

        comment = "# Skynet has been started\n### Flow:\n{}\n### Sandbox:\n{}".format(flow_url, sandbox_url)

        logging.info("Publishing comment: {}".format(comment))

        response = requests.post(
            "https://a.yandex-team.ru/api/v1/pull-requests/{}/comments".format(pr_id),
            headers=req_headers,
            json={
                "content": comment,
                "issue": False
            }
        )
        logging.info("Comment creation status: {}".format(response.status_code))
        logging.debug("Response: {}".format(response.content))
