import datetime
import json
import logging
import os

import requests

from requests.auth import HTTPBasicAuth

import sandbox.common.types.task as ctt

from sandbox import sdk2
from sandbox.sdk2.vcs.svn import Arcadia
from sandbox.sdk2.helpers import ProcessLog
from sandbox.sdk2.helpers import subprocess as sp

from sandbox.projects.tv.common import DEFAULT_ROBOT_KEY
from sandbox.projects.tv.common import Config

from sandbox.projects.tv.targetator.common import Targetator


class TargetatorUndertaker(Targetator):

    STAGE_TWO_ENABLED = True

    class Requirements(Targetator.Requirements):
        semaphores = ctt.Semaphores(acquires=[
            ctt.Semaphores.Acquire(name='CV_TARGETATOR_SEMAPHORE'),
            ctt.Semaphores.Acquire(name='CVTE_TARGETATOR_SEMAPHORE')
        ])

    def on_prepare(self):
        self.secret = sdk2.yav.Secret(DEFAULT_ROBOT_KEY)
        self.private_ssh_key = self.secret.data()["priv-key"]

        arcadia_url = os.path.join(Arcadia.ARCADIA_TRUNK_URL, self.TV_FIRMWARE_REPO_DIR)
        Arcadia.checkout(arcadia_url, self.ARC_PATH, revision=self.Parameters.revision-1)

        with open(os.path.join(self.ARC_PATH, self.Parameters.configuration_path)) as json_file:
            self.configuration = Config.from_json(json.load(json_file), json_file.name.split("/")[-1].replace(".json", ""))

    def process_target(self, config):
        logging.info("No processing required")

    def edit_targets_list(self, config):
        self.remove_target_from_list(config)

    def remove_target_from_list(self, config):
        branch_name = "{}-{}".format(config.ticket, datetime.datetime.today().strftime('%Y-%m-%d-%H-%M-%S'))
        try:
            with sdk2.ssh.Key(private_part=self.private_ssh_key), ProcessLog(self, logger=logging.getLogger('git')) as pl:
                sp.check_call("git clone \"{}\" {}".format(self.TARGETS_REPO, self.TARGETS_REPO_DIR), shell=True, stderr=pl.stderr, stdout=pl.stdout)
                os.chdir(self.TARGETS_REPO_DIR)
                sp.check_call("git config user.email \"{}@yandex-team.ru\"".format(self.ROBOT_ALIAS), shell=True, stderr=pl.stderr, stdout=pl.stdout)
                sp.check_call("git config user.name \"{}\"".format(self.ROBOT_ALIAS), shell=True, stderr=pl.stderr, stdout=pl.stdout)
                sp.check_call("git checkout -b {}".format(branch_name), shell=True, stderr=pl.stderr, stdout=pl.stdout)
        except sp.CalledProcessError as e:
            logging.error('git error: %s', e)
            raise e

        with open(self.TARGETS_FILE_PATTERN.format(config.platform), 'r+') as file:
            data = json.loads(file.read())
            file.seek(0)

            data["targets"].pop(config.sw_name, None)

            file.write(json.dumps(data, indent=2, sort_keys=True))
            file.truncate()

        try:
            with sdk2.ssh.Key(private_part=self.private_ssh_key), ProcessLog(self, logger=logging.getLogger('git')) as pl:
                has_changes = sp.call("git diff --no-ext-diff --quiet", shell=True, stderr=pl.stderr, stdout=pl.stdout)
                auth = HTTPBasicAuth(self.ROBOT_ALIAS, str(self.secret.data()["bb-personal-token"]))
                if has_changes:
                    request = self.add_remove_target(config, branch_name, pl, auth, remove_target=True)
                else:
                    logging.info("No changes detected in target_lists repo")

                if self.STAGE_TWO_ENABLED:
                    if has_changes:
                        pr_id = request.json()["id"]
                        request = requests.post(
                            'https://bb.yandex-team.ru/rest/api/1.0/projects/YANDEX-TV/repos/targets_lists/pull-requests/{}/merge'.format(pr_id),
                            auth=auth,
                            json={"version": 0}
                        )
                        logging.info("BB merge response code: {}".format(request.status_code))
                        logging.debug("Merge response: {}".format(request.content))
        except sp.CalledProcessError as e:
            logging.error('git commit and push failed: %s', e)
            raise e
