import json
import logging
import os

import requests
import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm

import sandbox.projects.common.binary_task as binary_task

from sandbox import sdk2

from sandbox.projects.common.vcs.arc import Arc
from sandbox.projects.common.binary_task import LastRefreshableBinary

from sandbox.projects.tv.common import notify_duty, get_resource_search_arguments
from sandbox.projects.tv.common import ALLOWED_CUSTOMERS, PLATFORMS, DEFAULT_ROBOT_KEY
from sandbox.projects.tv.common import Config, ConfigParsingException

from sandbox.projects.tv.warden.warden import find_all_configs, check_configs_unique


class ConfigWarden(LastRefreshableBinary, sdk2.Task):

    ARC_PATH = "arcadia"
    TV_FIRMWARE_REPO_DIR = "smart_devices/tv/platforms"
    RESOURCES_DIR = "resources"
    RELEASE_CONFIG = "release.json"

    class Requirements(sdk2.Requirements):
        dns = ctm.DnsType.DNS64
        client_tags = ctc.Tag.LINUX_BIONIC

        # Multislot requirements
        cores = 1
        ram = 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        _lbrp = binary_task.binary_release_parameters(stable=True)

        revision = sdk2.parameters.String("Change revision", required=True)
        pr_id = sdk2.parameters.Integer("Related PR id")

    @property
    def binary_executor_query(self):
        return get_resource_search_arguments(self)

    def on_execute(self):
        secret = sdk2.yav.Secret(DEFAULT_ROBOT_KEY)
        arc_token = secret.data()["arc-token"]
        arc_client = Arc(arc_oauth_token=arc_token)

        mount_point = arc_client.mount_path(None, None, fetch_all=False)

        mp = mount_point._work_path

        arc_client.checkout(mp, str(self.Parameters.revision))

        os.chdir(mp)

        all_changed_files = self.get_changed_files(arc_client, mp, self.Parameters.revision)
        author = self.get_author(arc_client, mp, self.Parameters.revision)

        # Filter only targets files
        changed_files = [file_path for file_path in all_changed_files if self.can_get_platform(file_path)]

        logging.info("Author: {}".format(author))
        logging.info("Changed files: {}".format(changed_files))

        result_merged = ""

        for path in changed_files:
            logging.info("Checking path: {}".format(path))
            result = ""
            keep_proccessing = True

            if not os.path.isfile(path):
                logging.info("{} file removed, skip".format(path))
                continue

            if " " in path:
                result += "{}: Contains space in the name\n\n".format(path)
                keep_proccessing = False

            if keep_proccessing:
                try:
                    with open(path) as f:
                        config = Config.from_json(json.load(f), f.name.split("/")[-1].replace(".json", ""))
                except ConfigParsingException as e:
                    result += "{}: Json is invalid: {}\n\n".format(path, e.message)
                    keep_proccessing = False
                except Exception:
                    result += "{}: Json is invalid\n\n".format(path)
                    keep_proccessing = False

            if keep_proccessing:
                if config.customer != config.customer.upper():
                    result += "{}: Customer is not written in caps ({}).\n\n".format(path, config.customer)

                if config.customer not in ALLOWED_CUSTOMERS:
                    result += "{}: Customer is not recognized ({}).\n" \
                        "If you are adding a new customer, change ALLOWED_CUSTOMERS in ConfigWarden\n\n".format(path, config.customer)

                if config.sw_name.upper() != config.sw_name:
                    result += "{}: SW_NAME is not written in caps ({})\n\n".format(path, config.sw_name)

                if config.boot_logo not in os.listdir(os.path.join(mp, self.TV_FIRMWARE_REPO_DIR, self.RESOURCES_DIR)):
                    result += "{}: Bootlogo does not exist in resources ({})\n\n".format(path, config.boot_logo)

            if len(result) > 0:
                result_merged += result
                if author == 'robot-edi':
                    notify_duty(
                        "# Config Warden check failed\n{}\n\nhttps://a.yandex-team.ru/review/{}/details".format(result, self.Parameters.pr_id),
                        config.ticket,
                        secret
                    )
                else:
                    # if changes are made manually, its enough to post comment in PR
                    # no need to spam in tickets
                    pass

        all_odms = [os.path.join(ConfigWarden.TV_FIRMWARE_REPO_DIR, odm) for odm in PLATFORMS]
        all_paths = find_all_configs(all_odms)
        problems = check_configs_unique(all_paths)
        if len(problems) > 0:
            more = "**More problems:**\n```\n{}\n```".format(json.dumps(problems, indent=4))
            result_merged += more

        mount_point.unmount()

        if len(result_merged) > 0:
            if self.Parameters.pr_id is not None and self.Parameters.pr_id != 0:
                self.create_task_comment(self.Parameters.pr_id, result_merged, arc_token)
            raise AssertionError(result_merged)

    def get_changed_files(self, arc_client, mount_point, revision):
        return [x["path"] for x in arc_client.show(mount_point, revision, as_dict=True, name_status=True)[0]["names"]]

    def get_author(self, arc_client, mount_point, revision):
        return arc_client.show(mount_point, revision, as_dict=True)[0]["commits"][0]["author"]

    def can_get_platform(self, config_path):
        if not os.path.isfile(config_path):
            return False

        if self.TV_FIRMWARE_REPO_DIR not in config_path:
            return False

        if self.RESOURCES_DIR in config_path:
            return False

        if self.RELEASE_CONFIG in config_path:
            return False

        try:
            # noinspection PyStatementEffect
            config_path.split('/')[-3] + "/" + config_path.split('/')[-2]
        except Exception:
            return False
        else:
            return True

    def create_task_comment(self, pr_id, message, arc_token):
        req_headers = {
            'Authorization': 'OAuth {}'.format(arc_token),
            'Content-Type': 'application/json'
        }

        comment = "# Config Warden check failed\n{}".format(message)

        logging.info("Publishing comment: {}".format(comment))

        response = requests.post(
            "https://a.yandex-team.ru/api/v1/pull-requests/{}/comments".format(pr_id),
            headers=req_headers,
            json={
                "content": comment,
                "issue": False
            }
        )
        logging.debug("Comment creation status: {}".format(response.status_code))
        logging.debug("Response: {}".format(response.content))
