import collections
import json
import logging
import os

from sandbox.projects.tv.common import Config
from sandbox.projects.tv.common import ConfigParsingException


def check_configs_unique(paths):
    problems = {}
    configs = []
    parsing_problems = []
    for path in paths:
        try:
            with open(path) as f:
                configs.append({
                    'config': Config.from_json(json.load(f), path.split("/")[-1].replace(".json", "")),
                    'path': path
                })
        except ConfigParsingException as e:
            parsing_problems += ["Missing required fields in {}: {}".format(path, e.message)]
        except Exception as e:
            parsing_problems += ["Failed to parse {}. Check logs for details".format(path)]
            logging.info(e)
    if len(parsing_problems) > 0:
        problems['parsing'] = parsing_problems

    check_unique(lambda c: c.props.device, configs, problems, 'device is not unique')
    check_unique(lambda c: c.ticket, configs, problems, 'ticket is not unique')
    check_unique(lambda c: c.target, configs, problems, 'target is not unique')

    filter_whitelist(problems, __problems_whitelist)

    return problems


def with_transform(transform):
    def reducing_lambda(acc, config):
        val = transform(config['config'])
        if val in acc:
            acc[val] += [config['path']]
        else:
            acc[val] = [config['path']]
        return acc
    return reducing_lambda


def check_unique(transform, configs, problems, desc):
    paths_for_val = reduce(with_transform(transform), configs, {})
    non_unique_list = filter(lambda i: len(i[1]) > 1, paths_for_val.items())
    non_unique_list = map(lambda l: (l[0], sorted(l[1])), non_unique_list)  # sort paths
    non_unique_list = sorted(non_unique_list)
    non_unique = collections.OrderedDict(non_unique_list)
    if len(non_unique) > 0:
        problems[desc] = non_unique


def filter_whitelist(problems, whitelist):
    for group_name in whitelist:
        if group_name not in problems:
            continue
        problems_group = problems[group_name]
        whitelist_group = whitelist[group_name]
        for prob_name in whitelist_group:
            if problems_group.get(prob_name) == whitelist_group.get(prob_name):
                problems_group.pop(prob_name)
        if len(problems_group) == 0:
            problems.pop(group_name)


def find_all_configs(paths):
    found = []
    for path in paths:
        find_json_recursive(path, found)
    return found


def find_json_recursive(path, found):
    if os.path.isfile(path):
        if path.endswith('.json'):
            found.append(path)
    elif os.path.isdir(path):
        for f in os.listdir(path):
            find_json_recursive(os.path.join(path, f), found)


__problems_whitelist = {
    "device is not unique": {
        "24LH1211.PT236AT02-1": [
            "smart_devices/tv/platforms/cv/6681/V-HOME_312_24LH1211.PT236AT02-1_CV6681H-K24_TWBC-01-2112-0105-BCW0358_JINPIN.json",
            "smart_devices/tv/platforms/hikeen/2842/V-HOME_1017_24LH1211.PT236AT02-1_RD2842P531_JINPIN_VITYAZ.json"
        ],
        "50U540S.CC500PV5D": [
            "smart_devices/tv/platforms/cvte/9632/LEFF_CP727482_50U540S.CC500PV5D_SK706S_PC822_SUB_A3_CHAOYE_POLAR.json",
            "smart_devices/tv/platforms/cvte/9632_11/LEFF_CP752960_50U540S.CC500PV5D_ID_BD_SK706S_PC822_SUB_A3_CH_CHAOYE_POLAR.json"
        ],
        "65LU1210.JR645R3HA3L": [
            "smart_devices/tv/platforms/cv/9632/V-HOME_305_65LU1210.JR645R3HA3L_CV9632H-AH_SKD12100821_SQY.json",
            "smart_devices/tv/platforms/cvte/9632_11/V-HOME_CS747211_65LU1210.JR645R3HA3L_ID_BD_SK706S_82_SUB_A16_SQY_VITYAZ.json"
        ],
        "VHIX-40F152MSY.V400HJ9-PE1": [
            "smart_devices/tv/platforms/cv/6681/HI_VHIX-40F152MSY.V400HJ9-PE1_CV6681H-B42_JINPIN_VITYAZ.json",
            "smart_devices/tv/platforms/hikeen/2842/HI_1019_VHIX-40F152MSY.V400HJ9-PE1_RD2842P637_JINPIN_VITYAZ.json"
        ]
    }
}
