# -*- coding: utf-8 -*-

import logging
import os
import shutil

import lxml.etree as ET

from sandbox import sdk2
from sandbox.common import utils

from sandbox.projects import resource_types
import sandbox.projects.resource_types.releasers as resource_releasers
from sandbox.projects.geoadv.task import GeoadvTask, JugglerStatus, GeoadvEnvironment
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk.environments import VirtualEnvironment


class TycoonAds(sdk2.Resource):
    """ Tycoon adverts """
    releasers = resource_releasers.tycoon
    releasable = True


class TycoonStyles(sdk2.Resource):
    """ Tycoon pre-defined styles """


class XsdResolver(ET.Resolver):
    def __init__(self, catalog_dir):
        self.catalogDir = catalog_dir
        self.xsdByUrl = {
            "http://www.w3.org/2001/xml.xsd": "xml.xsd",
            "http://docs.oasis-open.org/election/external/xAL.xsd": "xAL.xsd"
        }

    def resolve(self, url, pubid, context):
        logging.debug("Resolving %s" % url)
        xsdFile = self.xsdByUrl.get(url)
        if xsdFile:
            logging.debug("Resolved %s" % url)
            return self.resolve_filename(os.path.join(self.catalogDir, xsdFile), context)


class YtCluster(utils.Enum):
    utils.Enum.lower_case()
    HAHN = None
    ARNOLD = None


class TycoonAdverts(GeoadvTask):
    """
    Provide Tycoon adverts to Maps Search
    """

    GEN_DIR = "gen"

    class Requirements(sdk2.Requirements):
        # Requirements for execution on multislot https://docs.yandex-team.ru/sandbox/dev/requirements#multislot
        cores = 1
        ram = 12 * 1024
        disk_space = 12 * 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(GeoadvTask.Parameters):
        description = "Tycoon adverts"

        ytCluster = sdk2.parameters.String(
            "YT cluster to run on",
            choices=list((v, v) for v in YtCluster),
            default_value=YtCluster.HAHN,
            required=True
        )

        createRelease = sdk2.parameters.Bool("Create release", default=False)

        validateXml = sdk2.parameters.Bool("Validate XML", default=True)

        noDefaultPage = sdk2.parameters.Bool("Not generate default page", default=False)
        businessPage = sdk2.parameters.Bool("Generate business page", default=False)

        with validateXml.value[True]:
            schemasUrl = sdk2.parameters.ArcadiaUrl(
                description="Maps doc schemas",
                required=True,
                default_value="arcadia:/arc/trunk/arcadia/maps/doc/schemas"
            )

    def validate_adverts_xml(self, xml):
        if not self.Parameters.validateXml:
            return True
        schemas_path = sdk2.svn.Arcadia.get_arcadia_src_dir(self.Parameters.schemasUrl)
        parser = ET.XMLParser()
        parser.resolvers.add(XsdResolver(os.path.join(schemas_path, "catalog")))

        schema_root_file = os.path.join(schemas_path, "biz", "advert", "1.x", "geoproduct.xsd")

        logging.info("Validating XML vs schema url: %s, path: %s, root file: %s",
                     self.Parameters.schemasUrl, schemas_path, schema_root_file)

        schema = ET.XMLSchema(ET.parse(schema_root_file, parser))
        try:
            schema.assertValid(xml)
            return True
        except Exception as e:
            logging.error("XML is invalid: %s", e)
            return False

    def ttl_by_env(self):
        if self.Parameters.environment == GeoadvEnvironment.PRODUCTION:
            return 5
        else:
            return 2

    def gen_tycoon_adverts(self):
        logging.info("Generate TYCOON_ADVERTS")

        xml = ET.parse(os.path.join(self.GEN_DIR, "ads.xml"))
        is_valid_xml = self.validate_adverts_xml(xml)

        res_data = sdk2.ResourceData(resource_types.TYCOON_ADVERTS(
            self, self.Parameters.description, "tycoon_ads", ttl=self.ttl_by_env()
        ))
        res_data.path.mkdir(0o755, parents=True, exist_ok=True)

        xml.write(
            str(os.path.join(str(res_data.path), "ads.xml")),
            encoding="UTF-8",
            xml_declaration=True,
            pretty_print=True
        )

        shutil.copy(os.path.join(self.GEN_DIR, "styles.tar.gz"), str(res_data.path))

        res_data.ready()

        return is_valid_xml

    def on_execute(self):
        with VirtualEnvironment() as venv:
            env = os.environ.copy()
            env["YT_TOKEN"] = sdk2.Vault.data(self.owner, "yt_token")
            env["YQL_TOKEN"] = sdk2.Vault.data(self.owner, "YQL_TOKEN")

            venv.pip("pip==19.1.1 setuptools==41.0.1")
            venv.pip("-i https://pypi.yandex-team.ru/simple/ " +
                     " ".join(["yandex-yt==0.10.8",
                               "yql==1.2.101",
                               "yandex-yt-yson-bindings-skynet==0.3.29-1",
                               "requests==2.24.0",
                               "urllib3==1.25.11",
                               "lxml==4.6.3"]))

            # styles_path = str(sdk2.ResourceData(self.Parameters.styles).path)
            # shutil.copy(os.path.join(styles_path, "Styles.xml"), "Styles.xml")
            logging.info("modified os.environ=%s", env)
            executable = venv.executable
            logging.info("executable=%s", executable)

            cmd = [
                executable,
                os.path.join(os.path.dirname(__file__), "generate_adverts.py"),
                self.GEN_DIR
            ]

            if self.Parameters.ytCluster:
                cmd.append("--yt-cluster")
                cmd.append(self.Parameters.ytCluster)

            if self.Parameters.environment:
                cmd.append("--src-env")
                cmd.append(self.Parameters.environment)

                cmd.append("--dst-env")
                cmd.append(self.Parameters.environment)

            if self.Parameters.noDefaultPage:
                cmd.append("--no-default-page")

            if self.Parameters.businessPage:
                cmd.append("--generate-business-page")

            process.run_process(
                cmd=cmd,
                log_prefix="generate_adverts",
                outputs_to_one_file=False,
                wait=True,
                check=True,
                environment=env
            )

            is_valid_xml = self.gen_tycoon_adverts()

            if self.Parameters.createRelease and is_valid_xml:
                self.mark_released_resources(self.release_status(), ttl=self.ttl_by_env())
                self.notify_juggler("adverts-export", JugglerStatus.OK)

            if not is_valid_xml:
                raise RuntimeError('Invalid XML')
