# -*- coding: utf-8 -*-

import pprint
import time
import logging
from urlparse import urlparse
from datetime import timedelta
from url_util import complete_protocol, add_query_variable_to_url

from dict_util import safe_get

from test_layer_data import TEST_ADVERTS, URL_SHOULD_RESOLVE, URL_SHOULD_NOT_RESOLVE

IS_BOOKING = "is_booking"

RUBRIC_TO_CTA_TITLE = [
    ([30776,  # Ресторан
      30774,  # Кафе
      30770,  # Бары
      31401,  # Кальян-бары
      31495,  # Кофейня
      30775,  # Пиццерия
      31375,  # Суши-бар
      30762,  # Ночной клуб
      31426,  # Караоке-клуб
      31551,  # Антикафе
      31286  # Спортбар
      ], "Забронировать столик"),  # категория Кафе
    ([30210,  # Салон красоты
      31657,  # Ногтевая студия
      30211,  # Парикмахерская
      31546,  # Косметология
      3501411425,  # Барбершоп
      30085,  # Массажный салон
      30207,  # Тату-салон
      30208,  # Спа-салон
      3501539206,  # Салон бровей и ресниц
      3501569754,  # Лазерная эпиляция
      30214,  # Визажисты, стилисты
      3501479439,  # Соляная пещера
      3501621390  # Флоатинг
      ], "Записаться онлайн"),  # категория Красота
    ([30068,  # Медцентр, клиника
      30056,  # Стоматологическая клиника
      30083,  # Медицинская лаборатория
      30895,  # Ветеринарная клиника
      30067,  # Наркологическая клиника
      30069,  # Диагностический центр
      30065,  # Медицинская реабилитация
      30142,  # Больница для взрослых
      30887,  # Зоосалон, зоопарикмахерская
      30077,  # Психотерапевтическая помощь
      30066,  # Оздоровительный центр
      20269,  # Медицинская комиссия
      30114,  # Поликлиника для взрослых
      30128,  # Детская поликлиника
      30891,  # Гостиница для животных
      30057,  # Стоматологическая поликлиника
      30089,  # Коррекция зрения
      31635,  # Медицинская помощь на дому
      31658,  # Частнопрактикующие врачи
      30213,  # Пластическая хирургия
      30072,  # Гинекологическая клиника
      144152360,  # Специализированная больница
      30081,  # Нетрадиционная медицина
      30141,  # Детская больница
      30144,  # Амбулатория, здравпункт, медпункт
      30064,  # Урологический центр
      30116,  # Женская консультация
      286801534,  # Зуботехническая лаборатория
      30129,  # Госпиталь
      31649,  # Ветеринарная лаборатория
      3501479437,  # Перинатальный центр
      3501656634  # Остеопатия
      ], "Записаться онлайн"),  # категория Медицина
    ([20954  # Автомойка
      ], "Записаться онлайн"),
    ([30095  # Салон оптики
      ], "Записаться на консультацию")
]

DEFAULT_RUBRIC_CTA_TITLE = "Записаться онлайн"


def oso_create(campaign_id, version_id):
    # todo add hardcoded fields management when smvp data is ready
    oso = {
        "id": campaign_id,
        "commit_id": version_id,
        "lang": "ru",
        "banner_moderation": {"status": "accepted"},
        "extra_moderation": {"status": "accepted"},
        "new_moderation": {"status": "accepted"},
        "rich_moderation": {"status": "accepted"}
    }

    return oso


def oso_add_text_advert(oso, text_advert):
    if text_advert:
        text_advert["url"] = text_advert.pop("link", None)
        oso.update(text_advert)

    return oso


def smvp_get_text_advert_data(text_advert):
    body = safe_get(text_advert, "body")
    if body:
        return {
            "url": complete_protocol(body.pop("link", None)),
            "text": body.pop("text", None),
            "title": body.pop("title", None)
        }

    return {}


def oso_add_promotion(oso, promo):
    if promo:
        promo["url"] = promo.pop("link", None)
        promo["title"] = promo.pop("announcement", None)
        promo["details"] = promo.pop("description", None)
        # old ad timestamps are milliseconds from epoch while time functions return seconds
        try:
            if promo.get("date_from"):
                promo["start_time"] = int(time.mktime(time.strptime(promo.pop("date_from"), "%Y-%m-%d"))) * 1000
            # date_to is inclusive! have to add 1 day to simplify further checking
            if promo.get("date_to"):
                promo["stop_time"] = int(time.mktime(time.strptime(promo.pop("date_to"), "%Y-%m-%d"))
                                         + timedelta(days=1).total_seconds()) * 1000
        except ValueError as e:
            logging.warn(e)

        oso["promo"] = promo

    return oso


def smvp_get_time_from_str(body, field_name):
    return int(time.mktime(time.strptime(body.pop(field_name), "%Y-%m-%d"))) * 1000


def smvp_get_promotion_data(promo):
    body = safe_get(promo, "body")
    if body:
        body["url"] = complete_protocol(body.pop("link", None))
        body["title"] = body.pop("announcement", None)
        body["details"] = body.pop("description", None)

        # old ad timestamps are milliseconds from epoch while time functions return seconds
        try:
            if body.get("date_from"):
                body["start_time"] = smvp_get_time_from_str(body, "date_from")
            # date_to is inclusive! have to add 1 day to simplify further checking
            if body.get("date_to"):
                body["stop_time"] = smvp_get_time_from_str(body, "date_to") + timedelta(days=1).total_seconds() * 1000
        except ValueError as e:
            logging.warn(e)

        return body

    return None


def oso_add_products(oso, list_of_products):
    if list_of_products:
        for p in list_of_products:
            if "image" in p:
                p["photo"] = p.pop("image", None)

    oso["products"] = {
        "items": list_of_products,
        "currency": "rub"
    }
    return oso


def smvp_get_products_data(list_of_products):
    products = safe_get(list_of_products, "body.list")

    if products:
        for p in products:
            if "image" in p:
                p["photo"] = p.pop("image", None)
            if "url" in p:
                p["url"] = complete_protocol(p.pop("url", None))

    return {
        "items": products,
        "currency": safe_get(list_of_products, "body.currency")
    }


def oso_add_click_to_action(oso, click_to_action):
    if click_to_action:
        old_format = {
            "type": click_to_action.get("type", click_to_action.get("type", "url")),
            "value": click_to_action["link"],
            "title": click_to_action["title"]
        }
        oso["click_to_action"] = old_format
    return oso


def resolve_reservation_url(permanent_id, env_type):
    url = "https://{host}/business/{app_name}/{permalink}"
    if env_type == 'production':
        url = url.format(host="yandex.ru", app_name="book-table", permalink=permanent_id)
    else:
        url = url.format(host="l7test.yandex.ru", app_name="book-table", permalink=permanent_id)
    return url


def resolve_simple_request_url(permanent_id, env_type):
    url = "https://{host}/business/widget/request/company/{permalink}"
    if env_type == 'production':
        url = url.format(host="yandex.ru", permalink=permanent_id)
    else:
        url = url.format(host="l7test.yandex.ru", permalink=permanent_id)
    return url


def get_reservation_button_title(rubrics):

    # if any rubric belongs to a category (in priority order), use title for that category
    for (category, title) in RUBRIC_TO_CTA_TITLE:
        for rubric_id in rubrics:
            if rubric_id in category:
                return title
    logging.error("Unknown rubrics for reservation: %s", ", ".join([str(rubric) for rubric in rubrics]))
    return DEFAULT_RUBRIC_CTA_TITLE


def smvp_get_click_to_action_data(click_to_action, permanent_id, rubrics, env_type):
    body = safe_get(click_to_action, "body")
    campaign_type = safe_get(click_to_action, "campaign_type")

    cta = {}
    if campaign_type in ('SMVP', 'SPECIAL', 'SMBCRM'):
        link = resolve_simple_request_url(permanent_id, env_type) \
            if safe_get(body, "link_type", default="REGULAR") == "SIMPLE_REQUEST" \
            else safe_get(body, "link")

        cta = {
            "type": safe_get(body, "type", default="url"),
            "value": complete_protocol(link),
            "title": safe_get(body, "title")
        }
    elif campaign_type == 'RESERVATION':
        title = get_reservation_button_title(rubrics)
        if title:
            cta = {
                "type": "booking",
                "meta": {"iframe": {"width": 480, "height": 680}},
                "value": resolve_reservation_url(permanent_id, env_type),
                "title": title,
                IS_BOOKING: True
            }
    elif campaign_type == 'PARTNER':
        cta = {
            "type": safe_get(body, "type", default="booking"),
            "value": safe_get(body, "link"),
            "title": safe_get(body, "title"),
            IS_BOOKING: True
        }
        # add utm if present
        if (safe_get(body, "utm")):
            cta["value"] = add_query_variable_to_url(cta["value"], body["utm"])

    if cta:
        if campaign_type == 'SMBCRM' and is_smbcrm_widget_link(cta["value"]):
            cta["meta"] = {"iframe": {"width": 1136, "height": 616}}

        if safe_get(body, "link_type", default="REGULAR") == "SIMPLE_REQUEST":
            cta["meta"] = {"iframe": {"width": 480, "height": 680}}

        if safe_get(body, "iframe_meta"):
            cta["meta"] = {"iframe": {"width": int(safe_get(body, "iframe_meta.width")),
                                      "height": int(safe_get(body, "iframe_meta.height"))}}

        extype_value = body.get("extype")
        if extype_value:
            cta["extype"] = extype_value.lower()

    return cta


def is_smbcrm_widget_link(link):
    parsed = urlparse(link)
    return parsed.netloc == "yandex.ru" and parsed.path.startswith("/business/widget/")


def oso_add_headline(oso, headline):
    logo_href = safe_get(headline, "logo.href")
    if logo_href:
        oso["logo"] = {"href": logo_href}
    photo_href = safe_get(headline, "photo.href")
    if photo_href:
        oso["main_photo"] = {"href": photo_href}
    return oso


def smvp_get_headline_data(headline):
    body = safe_get(headline, "body")
    logo_href = safe_get(body, "logo.href")
    data = {}
    if logo_href:
        data["logo"] = {"href": logo_href}
    photo_href = safe_get(body, "photo.href")
    if photo_href:
        data["main_photo"] = {"href": photo_href}

    return data if data else None


def oso_add_call_tracking(oso, call_tracking):
    original = safe_get(call_tracking, "original")
    tracking = safe_get(call_tracking, "tracking")
    description = safe_get(call_tracking, "description")
    if original and tracking:
        oso["call_tracking"] = {
            "original": original,
            "tracking": oso_create_phone(safe_get(tracking, "details.country"),
                                         safe_get(tracking, "formatted"),
                                         description,
                                         safe_get(tracking, "details.number"),
                                         safe_get(tracking, "details.region"))
        }
    return oso


def smvp_get_call_tracking_data(call_tracking):
    body = safe_get(call_tracking, "body")
    original = safe_get(body, "original")
    tracking = safe_get(body, "tracking")
    description = safe_get(body, "description")
    if original and tracking:
        return {
            "original": original,
            "tracking": oso_create_phone(safe_get(tracking, "details.country"),
                                         safe_get(tracking, "formatted"),
                                         description,
                                         safe_get(tracking, "details.number"),
                                         safe_get(tracking, "details.region"))
        }
    return None


def smvp_get_branding_data(branding):
    body = safe_get(branding, "body")
    if body:
        body["branding"] = {"active": True}
    return body


def oso_create_phone(country_code, formatted, info, number, region_code):
    return {"country_code": country_code,
            "formatted": formatted,
            "hide": False,
            "infos": [{"locale": "ru", "value": info}],
            "number": number,
            "rank": 1,
            "region_code": region_code,
            "type": "phone"}


def convert_adverts_to_oso(campaign_id, text_advert, promo, product_list, click_to_action, headline, call_tracking):
    oso = oso_create(campaign_id, None)

    oso_add_text_advert(oso, text_advert)
    oso_add_promotion(oso, promo)
    oso_add_products(oso, product_list)
    oso_add_click_to_action(oso, click_to_action)
    oso_add_headline(oso, headline)
    oso_add_call_tracking(oso, call_tracking)
    return oso


def is_ad_transformed_branding(a):
    b = a.get("body")
    return a.get("type") == "BRANDING" and b and "branding_transformed" in b


def is_ad_active_promo(a):
    p = a.get("body")
    return a.get("type") == "PROMOTION" and p and \
           is_time_in_date_range(time.time(), p.get("date_from"), p.get("date_to"))


def cmp_ad_id_order(a1, a2):
    return a1.get("id", 0) - a2.get("id", 0)


def get_single_earliest_advert(filter_fun, adverts):
    fl = filter(filter_fun, adverts)
    fl.sort(key=lambda a: a.get("id", 0))
    return next(iter(fl), None)


def make_geoprod_ad_order(campaign_id, adverts, partner_links_disabled, priority_disabled):
    """
    See make_ad_order function from YQL code generating sandbox_export_source
    to understand the structure we want to emulate

    todo currently we do not support multiple adverts of the same type per set of permalinks
    the solution is to pick-up the earliest (with smaller id)
    for brandings we additionally pick-up only transformed
    for promos only active: now in (start_time, stop_time)
    """

    text_advert = get_single_earliest_advert(lambda a: a.get("type") == "TEXT_ADVERT", adverts)
    promo = get_single_earliest_advert(is_ad_active_promo, adverts)
    list_of_product = get_single_earliest_advert(lambda a: a.get("type") == "PRODUCT_LIST", adverts)
    branding = get_single_earliest_advert(is_ad_transformed_branding, adverts)
    click_to_action = get_single_earliest_advert(lambda a: a.get("type") == "CLICK_TO_ACTION", adverts)
    headline = get_single_earliest_advert(lambda a: a.get("type") == "HEADLINE", adverts)
    call_tracking = get_single_earliest_advert(lambda a: a.get("type") == "CALL_TRACKING", adverts)

    oso = convert_adverts_to_oso(campaign_id,
                                 safe_get(text_advert, "body"),
                                 safe_get(promo, "body"),
                                 safe_get(list_of_product, "body"),
                                 safe_get(click_to_action, "body"),
                                 safe_get(headline, "body"),
                                 safe_get(call_tracking, "body"))

    branding = safe_get(branding, "body")
    if branding:
        branding["branding"] = {"active": True}

    # todo enable links management and phones when smvp is ready
    ad_order = {
        "ads": [oso],
        "branding": branding,
        "show_partner_links": not partner_links_disabled,
        "priority_disabled": priority_disabled,
        "forwarding_number_details": None,
        "geo_original_number": None,
        "geo_redirect_number_details": None,
        "original_number": None
    }

    return ad_order


def is_time_in_date_range(time_sec, from_date, to_date):
    from_sec = time.mktime(time.strptime(from_date, "%Y-%m-%d")) if from_date else time_sec - 1
    to_sec = time.mktime(time.strptime(to_date, "%Y-%m-%d")) \
             + timedelta(days=1).total_seconds() if to_date else time_sec + 1

    return from_sec < time_sec < to_sec


def check_correct_ad_order_emulation():
    ad_order = make_geoprod_ad_order(111, TEST_ADVERTS, False, True)

    assert safe_get(ad_order, "branding.branding.active")
    assert safe_get(ad_order, "show_partner_links")
    assert safe_get(ad_order, "priority_disabled")

    ad = ad_order["ads"][0]

    for m in ["banner_moderation", "extra_moderation", "new_moderation", "rich_moderation"]:
        assert ad[m]["status"] == "accepted"

    assert ad["id"] == 111
    assert ad["url"] == "text advert url"
    assert ad["text"] == "text advert text"
    assert ad["title"] == "text advert title"

    assert "url" in ad["promo"]
    assert "start_time" in ad["promo"]
    assert "stop_time" in ad["promo"]
    assert ad["promo"]["title"] == "promo announcement"
    assert ad["promo"]["details"] == "promo description"

    assert ad["products"]["currency"] == "rub"
    assert "photo" in ad["products"]["items"][0]
    assert ad["products"]["items"][1]["title"] == "product2 title"

    assert ad["click_to_action"]["title"] == "cta url"
    assert ad["click_to_action"]["type"] == "url"
    assert ad["click_to_action"]["value"] == "http://url"

    assert ad["logo"]["href"] == "https://avatars.mdst.yandex.net/get-tycoon/" \
                                 "65823/2a0000016a4a2557104373f5444ad00b2879/logo"
    assert ad["main_photo"]["href"] == "https://avatars.mdst.yandex.net/get-tycoon/" \
                                       "65823/2a0000016a4a2514cee810d303fdaa7b335f/main"

    pp = pprint.PrettyPrinter(indent=4)
    pp.pprint(ad_order)


def check_correct_url_resolve():
    assert smvp_get_promotion_data(URL_SHOULD_RESOLVE["PROMOTION"])["url"] \
        == "http://example.com?param=value"

    assert smvp_get_products_data(URL_SHOULD_RESOLVE["PRODUCT_LIST"])["items"][0]["url"] \
        == "http://example.com"

    assert smvp_get_click_to_action_data(URL_SHOULD_RESOLVE["CLICK_TO_ACTION"], 1, "testing")["value"] \
        == "http://example.com?param=value#hash=mark"

    assert smvp_get_text_advert_data(URL_SHOULD_RESOLVE["TEXT_ADVERT"])["url"] \
        == "http://example.com"

    assert smvp_get_promotion_data(URL_SHOULD_NOT_RESOLVE["PROMOTION"])["url"] \
        == "http://example.com?param=value"

    assert smvp_get_products_data(URL_SHOULD_NOT_RESOLVE["PRODUCT_LIST"])["items"][0]["url"]\
        == "https://example.com"

    assert smvp_get_click_to_action_data(URL_SHOULD_NOT_RESOLVE["CLICK_TO_ACTION"], 1, "testing")["value"] \
        == "http://www.example.com?param=value#hash=mark"

    assert smvp_get_text_advert_data(URL_SHOULD_NOT_RESOLVE["TEXT_ADVERT"])["url"] \
        == "http://example.com"


def check_correct_advert_choice_inside_campaign():
    ta = get_single_earliest_advert(lambda a: a.get("type") == "TEXT_ADVERT", TEST_ADVERTS)
    assert ta["id"] == 1

    pr = get_single_earliest_advert(is_ad_active_promo, TEST_ADVERTS)
    assert pr["id"] == 1

    pl = get_single_earliest_advert(lambda a: a.get("type") == "PRODUCT_LIST", TEST_ADVERTS)
    assert pl["id"] == 1

    b = get_single_earliest_advert(is_ad_transformed_branding, TEST_ADVERTS)
    assert b["id"] == 1

    click_to_action = get_single_earliest_advert(lambda a: a.get("type") == "CLICK_TO_ACTION", TEST_ADVERTS)
    assert click_to_action["id"] == 6

    headline = get_single_earliest_advert(lambda a: a.get("type") == "HEADLINE", TEST_ADVERTS)
    assert headline["id"] == 7


if __name__ == "__main__":
    today_date_str = time.strftime("%Y-%m-%d", time.localtime(time.time()))
    assert is_time_in_date_range(time.time(), '2019-03-21', today_date_str)
    assert is_time_in_date_range(time.time(), today_date_str, today_date_str)
    assert is_time_in_date_range(time.time(), None, today_date_str)
    assert is_time_in_date_range(time.time(), today_date_str, None)

    check_correct_ad_order_emulation()
    check_correct_advert_choice_inside_campaign()
    check_correct_url_resolve()
    print("Приоритетное размещение на Яндекс.Картах".decode('utf-8'))
