import logging
from datetime import datetime
import os

from yql.api.v1.client import YqlClient
from yql.client.parameter_value_builder import YqlParameterValueBuilder as ValueBuilder


class DataSource(object):
    def __init__(self, yt_cluster, yql_dir=""):
        self.yql_client = YqlClient(token=os.getenv("YQL_TOKEN"), db=yt_cluster)
        self.yql_client.config.syntax_version = 1
        self.yql_dir = yql_dir

    def execute_as_single_script(self, yql_filenames, parameters):
        """
        :param yql_filenames: should be a valid yql script after their concatenation in specified order
        :param parameters:
        :return:
        """
        logging.info("Started processing of %s as single file", yql_filenames)
        query = ""
        for yql in yql_filenames:
            query += self._read_script(yql)[1]
        result_tables = self._execute_query_string(query, parameters)
        logging.info("Finished processing of %s as single file", yql_filenames)
        return result_tables

    def execute_script(self, yql_filename, parameters):
        yql_filename, query = self._read_script(yql_filename)
        logging.info("Started processing of %s", yql_filename)
        result_tables = self._execute_query_string(query, parameters)
        logging.info("Finished processing of %s", yql_filename)
        return result_tables

    def _read_script(self, yql_filename):
        header = "\n-- Below is content from %s\n" % yql_filename

        if os.path.exists(os.path.join(self.yql_dir, yql_filename)):
            yql_filename = os.path.join(self.yql_dir, yql_filename)
        try:
            with open(yql_filename, 'r') as f:
                return yql_filename, header + f.read()
        except IOError:
            import library.python.resource as resource
            return yql_filename, header + resource.find("sandbox/projects/tycoon/TycoonAdverts/yql/" + yql_filename)

    def _execute_query_string(self, yql, params):
        logging.info('YQL query is being executed...')
        logging.debug('YQL body:\n"%s"\nits params: %s', yql, params)
        start = datetime.now()
        request = self.yql_client.query(yql, syntax_version=1)
        wrapped = self._wrap_params(params)
        request.run(parameters=ValueBuilder.build_json_map(wrapped))
        logging.info(request.share_url)
        result_tables = request.get_results()
        delta = datetime.now() - start
        logging.info("Query %s", "succeeded" if bool(result_tables) else "failed")
        logging.info("It took %s sec", delta.total_seconds())
        return result_tables

    def _wrap_params(self, params):
        wp = {}
        for k, v in params.items():
            if isinstance(v, str):
                wp[k] = ValueBuilder.make_string(v)
            elif isinstance(v, list):
                if v:
                    if isinstance(v[0], int) or isinstance(v[0], long):
                        v = [ValueBuilder.make_int64(int(e)) for e in v]
                    elif isinstance(v[0], str):
                        v = [ValueBuilder.make_string(e) for e in v]
                    else:
                        assert False
                wp[k] = ValueBuilder.make_list(v)
            else:
                logging.error("Enable handling of %s params!", type(v))
                assert False
        return wp


if __name__ == "__main__":
    script_dir = os.path.join(os.path.dirname(__file__), "yql")
    script_file = os.path.join(script_dir, "campaign_duplicates.yql")

    # yql_client = YqlClient(db="hahn", token=os.getenv("YQL_TOKEN"))
    # req = yql_client.query("declare $var as String; select $var as variable;")
    # req.run(parameters=ValueBuilder.build_json_map({"$var": ValueBuilder.make_string("qqq")}))
    # results = req.get_results()
    # logging.info(req)
    # for table in results:
    #     print('\n--- Data ---')
    #     for row in table.rows:
    #         for cell, name, cell_type in zip(row, table.column_names, table.column_print_types):
    #             s = '%s(%s): %s' % (name, cell_type, str(cell))
    #             print(s.encode('utf-8'))
