#!/usr/bin/env python
# -*- coding: utf-8 -*-
# How to develop this file: https://wiki.yandex-team.ru/geoadv/dev/jeksport-na-karty/

import argparse
import json
import logging
import os
import re
import tarfile
import time
import unicodedata

import lxml.etree as ET
import requests
import yt.wrapper as yt
from requests.adapters import HTTPAdapter
from urllib3.util.retry import Retry

from ad_order_emulation import smvp_get_call_tracking_data, \
    smvp_get_branding_data, \
    smvp_get_click_to_action_data, \
    smvp_get_promotion_data, \
    smvp_get_headline_data, \
    smvp_get_text_advert_data, \
    smvp_get_products_data, IS_BOOKING
from data_source import DataSource
from dict_util import safe_get
from price_util import formatted_price_to_valid_number, formatted_price_with_currency
from url_util import ensure_url_format
from settings_util import *
from rubric_disclaimers import get_disclaimers
from test_layer_data import REAL_COMPANIES, \
    make_test_text_advert, \
    make_test_branding

'''
map below allows to mix production and testing YT tables in order to use
production as source and testing as result w/o affecting production
'''
YT_ENV = {
    "smvp": {
        "$src": "testing",
        "$dst": "testing",
    }
}

DOWNLOAD_ICONS = True

NS_MAP = {
    None: "http://maps.yandex.ru/advert/1.x",
    "xml": "http://www.w3.org/XML/1998/namespace",
    "xsi": "http://www.w3.org/2001/XMLSchema-instance"
}

with_empty = False
EMPTY_ADVERT_TEXT = "Приоритетное размещение на Яндекс.Картах".decode('utf-8') if with_empty else ""

NAVI_PLACEMENT = ["navi", "navi/datatesting"]
MOBILE_PLACEMENT = ["mobile_maps"]
DEFAULT_PLACEMENT = ["default", "experiment", "maps", "transport"]
SERP_PLACEMENT = ["serp"]

TESTING_NAVI_PLACEMENT = ["navi/datatesting"]
TESTING_MOBILE_PLACEMENT = ["mobile_maps/datatesting"]
TESTING_DEFAULT_PLACEMENT = ["maps/datatesting"]
ALLOWED_PHONE_TYPES = ['phone', 'fax', 'phone_fax']

default_icons = [
    "http://storage.mds.yandex.net/get-geoadv/225293/__gp_style_logo_v2.svg",
    "http://storage.mds.yandex.net/get-geoadv/1244551/__gp_style_pin.svg",
    "http://storage.mds.yandex.net/get-geoadv/1336908/__gp_style_logo.svg"
]

mobile_maps_icons = [{
    "name": "Dust",
    "path": ["dust_search", "standard"],
    "anchor": "0.5 0.5",
    "size": "18 18"
}, {
    "name": "DustVisited",
    "path": ["dust_search", "visited"],
    "anchor": "0.5 0.5",
    "size": "18 18"
}, {
    "name": "DustHover",
    "path": ["dust_search", "hover"],
    "anchor": "0.5 0.5",
    "size": "18 18"
}, {
    "name": "Icon",
    "path": ["drop_search", "standard"],
    "anchor": "0.5 0.89",
    "size": "32 38",
    "navi": False
}, {
    "name": "IconVisited",
    "path": ["drop_search", "visited"],
    "anchor": "0.5 0.89",
    "size": "32 38",
    "navi": False
}, {
    "name": "IconHover",
    "path": ["drop_search", "hover"],
    "anchor": "0.5 0.89",
    "size": "32 38",
    "navi": False
}, {
    "name": "Selected",
    "path": ["pin_search", "standard"],
    "anchor": "0.5 0.97",
    "size": "60 68"
}, {
    "name": "SelectedWithoutDot",
    "path": ["pin_bitmap_search", "standard"],
    "anchor": "0.5 0.94",
    "size": "60 60"
}, {
    "name": "Icon",
    "path": ["navi_drop_search", "standard"],
    "anchor": "0.5 0.5",
    "size": "32 32",
    "navi": True
}, {
    "name": "IconVisited",
    "path": ["navi_drop_search", "visited"],
    "anchor": "0.5 0.5",
    "size": "32 32",
    "navi": True
}, {
    "name": "IconHover",
    "path": ["navi_drop_search", "hover"],
    "anchor": "0.5 0.5",
    "size": "32 32",
    "navi": True
}
]

SMVP_DEFAULT_LANG = "ru"


def make_tarfile(path, source_dir):
    with tarfile.open(path, "w:gz") as tar:
        tar.add(source_dir, arcname="/")


icons_data_map = {}
icons_url_map = {}
session = requests.Session()

retries = Retry(total=5,
                backoff_factor=0.1,
                status_forcelist=[500, 502, 503, 504])
session.mount('http://', HTTPAdapter(max_retries=retries))


def download_image(path, url):
    file_name = url.split("/")[-1]
    full_name = path + "/" + file_name
    if os.path.isfile(full_name):
        return file_name

    existed_file_name = icons_url_map.get(url)

    if existed_file_name:
        return existed_file_name

    img_data = session.get(url).content
    existed_file_name = icons_data_map.get(img_data)

    if existed_file_name:
        logging.debug("Skipping local saving for '" + url + "', content is the same with '" + existed_file_name)
        icons_url_map[url] = existed_file_name
        return existed_file_name

    with open(full_name, 'wb') as handler:
        handler.write(img_data)
    icons_data_map[img_data] = file_name
    icons_url_map[url] = file_name
    return file_name


def get_icon_url(data, icons_meta):
    url = data
    for key in icons_meta["path"]:
        url = url[key]
    return url


def upper_first_letter(str_value):
    """
    This function is to upper case the 1-st letter only
    Introduced instead of capitalize() because it low cases the rest of sting
    """
    if not (isinstance(str_value, str) and len(str_value) > 0):
        return str_value
    return str_value[0].upper() + str_value[1:]


def remove_control_characters(s):
    return "".join(ch for ch in s if unicodedata.category(ch)[0] != "C")


def add_element(data, parent_elem, elem_name):
    if elem_name in data and data[elem_name]:
        elem = ET.SubElement(parent_elem, elem_name)
        elem.text = data[elem_name].decode('utf-8')
        return elem


def add_url_element(data, parent_elem, elem_name, encode_url=False):
    if elem_name in data and data[elem_name]:
        elem = ET.SubElement(parent_elem, elem_name)
        elem.text = ensure_url_format(data[elem_name].decode('utf-8'), encode_url)
        return elem


def add_mandatory_element(data, parent_elem, elem_name, alt_text):
    elem = ET.SubElement(parent_elem, elem_name)
    text = data[elem_name].decode('utf-8') if data.get(elem_name) else alt_text
    if text:
        elem.text = text
    return elem


def add_stub_text_element(parent_elem, elem_name, text):
    elem = ET.SubElement(parent_elem, elem_name)
    if text:
        elem.text = remove_control_characters(text)
    return elem


def is_moderated(advert):
    return safe_get(advert, "new_moderation.status") == "accepted"


def is_branding_active_and_transformed(branding):
    return bool(safe_get(branding, "branding.active")) and branding.get("branding_transformed")


image_url_to_branding = {}
"""
Having for certain url both True and False in resulting set means icons were already added
for both navi and maps so processing should be skipped
This hugely optimizes icons download and whole export time as per ticket GEOPROD-1661
"""


def map_by_url_to_cached_branding(branding):
    image_url = safe_get(branding, "branding.image_url")

    if not is_branding_active_and_transformed(branding) or not image_url:
        return branding

    return image_url_to_branding.setdefault(image_url, branding)


def add_icons(parent, branding, icons_dir, icons_meta, navi):
    if not is_branding_active_and_transformed(branding):
        return

    data = branding["branding_transformed"]

    for icon_data in icons_meta:
        if (safe_get(icon_data, "navi") is True and not navi) or (
                navi and safe_get(icon_data, "navi") is False):
            continue

        icon_url = get_icon_url(data, icon_data)
        file_name = download_image(icons_dir, icon_url) if DOWNLOAD_ICONS else ""
        logging.debug("Adding icon %s", file_name)

        field_name = icon_data["name"]

        style_field = ET.SubElement(parent, "field")
        style_field.text = file_name.replace(".svg", "")
        style_field.attrib["name"] = "style" + field_name

        anchor_field = ET.SubElement(parent, "field")
        anchor_field.text = icon_data["anchor"]
        anchor_field.attrib["name"] = "anchor" + field_name

        size_field = ET.SubElement(parent, "field")
        size_field.text = icon_data["size"]
        size_field.attrib["name"] = "size" + field_name


def cta_value_extract_default(value):
    return value.decode("utf-8") if isinstance(value, str) else value


def cta_value_extract_meta(value):
    return json.dumps(value, separators=(',', ':')) if value else None


DEFAULT_CTA_ATTR_DEF = {
    "type": cta_value_extract_default,
    "title": cta_value_extract_default,
    "value": cta_value_extract_default,
    "meta": cta_value_extract_meta
}

BUSINESS_CTA_ATTR_DEF = {
    "type": cta_value_extract_default,
    "title": cta_value_extract_default,
    "value": cta_value_extract_default,
    "meta": cta_value_extract_meta,
    "extype": cta_value_extract_default
}


def prepare_action_attr_set_smvp(cta, attr_def):
    if not cta:
        return None

    res = {k: v(cta[k]) for k, v in attr_def.items() if k in cta}

    return res


def create_old_xml_action(parent, attr_values):
    # type field
    xml_field = ET.SubElement(parent, "field")
    xml_field.text = attr_values["type"]
    xml_field.attrib["name"] = "actionType"

    for (name, value) in attr_values.items():
        if name == "type":
            continue

        xml_field = ET.SubElement(parent, "field")
        xml_field.text = value
        xml_field.attrib["name"] = "action" + upper_first_letter(name)


def create_old_xml_action_mobile(parent, attr_values):
    # type field
    xml_field = ET.SubElement(parent, "field")
    xml_field.text = "url"
    xml_field.attrib["name"] = "actionType"

    # check cast tel required
    cast_tel = attr_values.get("type") == "tel"

    for (name, value) in attr_values.items():
        if name == "type":
            continue

        xml_field = ET.SubElement(parent, "field")

        if name == "value":
            # for phones format url like `tel:+79991000000`
            xml_field.text = "tel:+" + value if cast_tel else ensure_url_format(value, encode=True)
        else:
            xml_field.text = value
        xml_field.attrib["name"] = "action" + upper_first_letter(name)


def create_new_xml_action(parent, attr_values):
    parent = ET.SubElement(parent, "Actions")
    parent = ET.SubElement(parent, "Action")

    cta_type_to_schema = {"url": "OpenSite", "tel": "Call", "booking": "OpenSite"}

    # type field first
    xml_field = ET.SubElement(parent, "type")
    xml_field.text = cta_type_to_schema[attr_values["type"]]

    for (name, value) in attr_values.items():
        if name == "type":
            continue

        xml_field = ET.SubElement(parent, "field")
        xml_field.text = value
        xml_field.attrib["name"] = name


def create_new_xml_action_mobile(parent, attr_values):
    parent = ET.SubElement(parent, "Actions")
    parent = ET.SubElement(parent, "Action")

    # type field first
    xml_field = ET.SubElement(parent, "type")
    xml_field.text = "OpenSite"

    # check cast tel required
    cast_tel = attr_values.get("type") == "tel"

    for (name, value) in attr_values.items():
        if name == "type":
            continue

        xml_field = ET.SubElement(parent, "field")

        if name == "value":
            # for phones format url like `tel:+79991000000`
            xml_field.text = "tel:+" + value if cast_tel else ensure_url_format(value, encode=True)
        else:
            xml_field.text = value
        xml_field.attrib["name"] = name


def add_actions_smvp(parent, cta, creator, attr_def=DEFAULT_CTA_ATTR_DEF):
    attr_values = prepare_action_attr_set_smvp(cta, attr_def)
    if not attr_values:
        return

    # this prints container with utf-8 correctly
    logging.debug("Adding action: {%s}", ", ".join(['%s="%s"' % (name, value) for name, value in attr_values.items()]))
    creator(parent, attr_values)


def strip_phone_formatting(formatted):
    """
    :param formatted: +7 (495) 555-55-55
    :return: 74955555555
    """
    return re.sub("[^0-9]", "", formatted)


def override_metadata(advert_data, phones, show_partner_links):
    if not phones and show_partner_links:
        return

    metadata = ET.SubElement(advert_data, "OverrideMetadata")
    if phones:
        phones_element = ET.SubElement(metadata, "Phones")
        for phone in filter(lambda p: p["type"] in ALLOWED_PHONE_TYPES, phones):
            phone_element = ET.SubElement(phones_element, "Phone")
            phone_element.set("type", phone["type"])

            formatted = phone.get("formatted", "")
            if not formatted:
                logging.warn("Missing non-trivial value for mandatory Phone field 'formatted'")
            ET.SubElement(phone_element, "formatted").text = formatted

            if phone.get("country_code"):
                ET.SubElement(phone_element, "country").text = phone["country_code"]

            if phone.get("region_code"):
                ET.SubElement(phone_element, "prefix").text = phone["region_code"]

            # just get with default does not cover the case when key exists but with empty string value
            number = phone.get("number") or strip_phone_formatting(formatted)
            if not number:
                logging.warn("Missing non-trivial value for mandatory Phone field 'number'")
            ET.SubElement(phone_element, "number").text = number

            for info in phone.get("infos", []):
                if info.get("value"):
                    ET.SubElement(phone_element, "info").text = info["value"].decode('utf-8')
                    break  # schema supports only one info

    if not show_partner_links:
        ET.SubElement(metadata, "hideBookingLinks").text = 'true'


def add_mobile_log(parent):
    ET.SubElement(parent, "logId").text = "__gp_log"


def add_mobile_tag(parent):
    tags_elem = ET.SubElement(parent, "Tags")

    return tags_elem


def add_age_disclaimer(advert, ad):
    if "age_mark" in ad and ad["age_mark"]:
        age = int(ad["age_mark"]) / 12
        ET.SubElement(advert, "disclaimer").text = "%d+" % age


def add_rubric_disclaimers(parent, disclaimers):
    disclaimers = [d.decode('utf-8') for d in disclaimers]
    for d in disclaimers:
        ET.SubElement(parent, "disclaimer").text = d


def insert_rubric_disclaimers(parent, after_name, disclaimers):
    after = parent.find(after_name)
    after_index = parent.index(after) if after is not None else len(parent)
    disclaimers = [d.decode('utf-8') for d in disclaimers]
    for d in disclaimers:
        after_index = after_index + 1
        e = ET.Element("disclaimer")
        e.text = d
        parent.insert(after_index, e)


def add_rich_smvp(parent, rubrics, headline, promo, product_list, encode_urls=False):
    # todo GEOPROD-1272 so far we have same rubric-defined disclaimer for advert, promo and products,
    # later we will have moderation-defined
    disclaimers = get_disclaimers(rubrics, SMVP_DEFAULT_LANG)
    if disclaimers:
        logging.debug(
            "Adding disclaimers found by rubrics to advert, promo, products %s: %s",
            rubrics, "".join(disclaimers))
        insert_rubric_disclaimers(parent, "text", disclaimers)

    add_element({}, parent, "about")

    url_template = ensure_url_format(safe_get(headline, "logo.href"), encode_urls)

    if url_template:
        logo = ET.SubElement(parent, "Logo")
        ET.SubElement(logo, "url_template").text = url_template

    url_template = ensure_url_format(safe_get(headline, "main_photo.href"), encode_urls)
    if url_template:
        photo = ET.SubElement(parent, "Photo")
        ET.SubElement(photo, "url_template").text = url_template

    add_advert_promo(parent, promo, disclaimers, encode_urls)
    add_advert_products_smvp(parent, product_list, disclaimers, encode_urls)


def add_advert_promo(parent, promo, disclaimers, encode_url=False):
    if promo:
        now = int(time.time() * 1000)
        promo_active = promo.get("start_time", 0) < now < promo.get("stop_time", now + 1)
        promo_title = promo.get("title")
        if promo_active and promo_title:
            promo_xml = ET.SubElement(parent, "Promo")
            add_element(promo, promo_xml, "title")
            add_element(promo, promo_xml, "details")

            # todo GEOPROD-1272 until MAP client(s) are ready we need to keep the customer given info in the old field
            # later we will have сustomer given info in new field only and rubric disclaimers will come unconditionally
            if promo.get("disclaimer"):
                add_element(promo, promo_xml, "disclaimer")
            else:
                add_rubric_disclaimers(promo_xml, disclaimers)

            add_url_element(promo, promo_xml, "url", encode_url)

            url_template = ensure_url_format(safe_get(promo, "image.href"), encode_url)
            if url_template:
                banner = ET.SubElement(promo_xml, "Banner")
                ET.SubElement(banner, "url_template").text = url_template

            if promo.get("disclaimer"):
                promo["full_disclaimer"] = promo["disclaimer"]
                add_element(promo, promo_xml, "full_disclaimer")


def add_advert_product(parent, product_data, currency, encode_url=False):
    add_element(product_data, parent, "title")
    add_url_element(product_data, parent, "url", encode_url)
    photo = ET.SubElement(parent, "Photo")
    url_template = ensure_url_format(safe_get(product_data, "photo.href"), encode_url)
    if url_template:
        ET.SubElement(photo, "url_template").text = url_template

    formatted = product_data.get("price")
    numeric = formatted_price_to_valid_number(formatted)
    # GEOPROD-1590 / GEOPROD-1592
    # it was decided we want to have price as arbitrary string to write there credit percents,
    # price ranges, etc, however, per current xml definition Price element cannot handle this
    # so a choice could be to add it to the title element...
    # However, it is always better to do this explicitly on customer side
    # to have reasonable and expected formatting, punctuation, etc
    if formatted and numeric is not None:
        price = ET.SubElement(parent, "Price")
        ET.SubElement(price, "value").text = str(numeric)
        ET.SubElement(price, "text").text = \
            formatted_price_with_currency(numeric, formatted, currency).decode("utf-8")
        ET.SubElement(price, "currency").text = currency.upper().decode("utf-8")


def add_advert_products_smvp(parent, product_list, disclaimers, encode_urls=False):
    product_items = safe_get(product_list, "items")
    currency = safe_get(product_list, "currency")

    if product_items:
        products_currency = currency if currency else 'rub'

        for item in product_items:
            product = ET.SubElement(parent, "Product")
            add_advert_product(product, item, products_currency, encode_urls)
            add_rubric_disclaimers(product, disclaimers)


def add_adverts_smvp(parent, text_advert, encode_url=False):
    advert = ET.SubElement(parent, "Advert")

    add_mandatory_element(text_advert, advert, "title", None)
    add_mandatory_element(text_advert, advert, "text", EMPTY_ADVERT_TEXT)
    add_url_element(text_advert, advert, "url", encode_url)

    return advert


def add_permanent_ids(parent, permanent_ids):
    companies = ET.SubElement(parent, "Companies")
    for permanent_id in permanent_ids:
        ET.SubElement(companies, "id").text = str(permanent_id)


# Используется геопоиском, чтобы перестать форсить организацию в выдаче.
# По умолчанию гепоиск ставит в true.
def add_prioritized_off(advert_data):
    prioritized = ET.SubElement(advert_data, "prioritized")
    prioritized.text = "false"


# Пробрасывается на клиенты, чтобы клиенты понимали, что не нужно рисовать зеленую метку.
# По умолчанию ставим на клиентах в true.
#
# Добавляет официальное поле highlighted поддерживаемое MapKit-ом.
# https://st.yandex-team.ru/MAPSMOBCORE-9385
def add_highlighted_off(advert_data, tags_elem):
    highlightedTag = ET.SubElement(tags_elem, "field")
    highlightedTag.attrib["name"] = "highlighted"
    highlightedTag.text = "false"
    highlighted = ET.SubElement(advert_data, "highlighted")
    highlighted.text = "false"


# Для мобильных карт существует способ снять метку геопродукта, используемый для пинов.
# https://st.yandex-team.ru/GEOADVDEV-1342
#
# Используем его пока не появилось решение на схеме.
def add_highlighted_off_legacy_mobile(tags_elem):
    advertTypeTag = ET.SubElement(tags_elem, "field")
    advertTypeTag.attrib["name"] = "advert_type"
    advertTypeTag.text = "menu_icon"


def add_priority_strenth(advert_data, priority_strength):
    highlighted = ET.SubElement(advert_data, "priority_strength")
    highlighted.text = str(priority_strength)


def add_hide_serp_offer(tags_elem):
    advertTypeTag = ET.SubElement(tags_elem, "field")
    advertTypeTag.attrib["name"] = "hideSerpOffer"
    advertTypeTag.text = "true"


def add_priority_settings(advert_data, settings):
    if is_priority_disabled(settings):
        add_prioritized_off(advert_data)
    else:
        priority_rate = get_priority_rate(settings)
        if priority_rate is not None:
            add_priority_strenth(advert_data, priority_rate)


def add_advert_data_smvp(parent,
                         campaign_ids,
                         permanent_ids,
                         icons_dir,
                         icons_meta,
                         page_ids,
                         branding,
                         settings,
                         cta,
                         text_advert):
    advert_data = ET.SubElement(parent, "AdvertData")
    for place in page_ids:
        ET.SubElement(advert_data, "pageId").text = place

    ET.SubElement(advert_data, "logId").text = ",".join(str(x) for x in campaign_ids)

    tags_elem = ET.SubElement(advert_data, "Tags")

    add_icons(tags_elem, branding, icons_dir, icons_meta, False)

    # добавляем extype на страницу serp, пока bprofile нормально не доезжает до поиска - GEOPROD-2975
    add_actions_smvp(tags_elem, cta, create_old_xml_action, BUSINESS_CTA_ATTR_DEF)

    advert = add_adverts_smvp(advert_data, text_advert)

    add_priority_settings(advert_data, settings)

    if is_highlight_disabled(settings):
        add_highlighted_off(advert_data, tags_elem)

    if is_hide_serp_offer(settings):
        add_hide_serp_offer(tags_elem)

    if 1683515647 in permanent_ids and is_testing_enabled(settings):
        add_priority_strenth(advert_data, '0.5')

    add_permanent_ids(advert_data, permanent_ids)

    return advert


def add_advert_mobile_data_smvp(parent,
                                permanent_ids,
                                icons_dir,
                                icons_meta,
                                page_ids,
                                branding,
                                settings,
                                cta,
                                text_advert):
    advert_data = ET.SubElement(parent, "AdvertData")
    for place in page_ids:
        ET.SubElement(advert_data, "pageId").text = place

    add_mobile_log(advert_data)
    tags_elem = add_mobile_tag(advert_data)

    add_icons(tags_elem, branding, icons_dir, icons_meta, False)

    add_actions_smvp(tags_elem, cta, create_old_xml_action_mobile)

    advert = add_adverts_smvp(advert_data, text_advert, encode_url=True)

    add_priority_settings(advert_data, settings)

    if is_highlight_disabled(settings):
        add_highlighted_off(advert_data, tags_elem)
        add_highlighted_off_legacy_mobile(tags_elem)

    if is_hide_serp_offer(settings):
        add_hide_serp_offer(tags_elem)

    add_permanent_ids(advert_data, permanent_ids)

    return advert


def add_advert_navi_data_smvp(parent,
                              permanent_ids,
                              company_name,
                              icons_dir,
                              icons_meta,
                              page_ids,
                              branding,
                              settings,
                              cta,
                              promotion,
                              text_advert):
    advert_data = ET.SubElement(parent, "AdvertData")
    for place in page_ids:
        ET.SubElement(advert_data, "pageId").text = place

    add_mobile_log(advert_data)
    tags_elem = add_mobile_tag(advert_data)

    add_icons(tags_elem, branding, icons_dir, icons_meta, True)

    add_actions_smvp(tags_elem, cta, create_old_xml_action_mobile)

    advert = add_navi_adverts_smvp(advert_data, company_name, promotion, text_advert)

    add_priority_settings(advert_data, settings)

    if is_highlight_disabled(settings):
        add_highlighted_off(advert_data, tags_elem)
        add_highlighted_off_legacy_mobile(tags_elem)

    if is_hide_serp_offer(settings):
        add_hide_serp_offer(tags_elem)

    add_permanent_ids(advert_data, permanent_ids)

    return advert


def add_navi_adverts_smvp(parent, company_name, promo, text_advert):
    advert = ET.SubElement(parent, "Advert")

    title_elem = ET.SubElement(advert, "title")
    if company_name:
        title_elem.text = company_name

    text = ""
    if promo:
        now = int(time.time() * 1000)
        promo_active = promo.get("start_time", 0) < now < promo.get("stop_time", now + 1)
        promo_title = promo.get("title")
        if promo_active and promo_title:
            text = promo_title.decode('utf-8')

    if not text:
        text = text_advert["title"].decode('utf-8') if text_advert.get("title") else EMPTY_ADVERT_TEXT

    add_stub_text_element(advert, "text", text)
    add_url_element(text_advert, advert, "url", encode_url=True)

    return advert


def add_advert_data_business(parent,
                             permanent_ids,
                             cta,
                             settings):
    if not cta:
        return

    advert_data = ET.SubElement(parent, "AdvertData")

    ET.SubElement(advert_data, "pageId").text = "bprofile"

    tags_elem = ET.SubElement(advert_data, "Tags")

    advert = add_adverts_smvp(advert_data, {})

    add_actions_smvp(advert, cta, create_new_xml_action, BUSINESS_CTA_ATTR_DEF)

    add_priority_settings(advert_data, settings)

    if is_highlight_disabled(settings):
        add_highlighted_off(advert_data, tags_elem)

    add_permanent_ids(advert_data, permanent_ids)

    return advert


def smvp_apply_call_tracking(call_tracking, company_phones):
    if not call_tracking:
        return []

    if not company_phones:
        logging.warning("No company phones; call tracking active")
        company_phones = []

    num_to_replace = call_tracking["original"]

    if company_phones:
        for index, phone in enumerate(company_phones):
            if strip_phone_formatting(phone["formatted"]) == num_to_replace:
                del (company_phones[index])  # remove the original phone, will be replaced with tracking
                break

    # always put the tracking number at 1st position
    company_phones.insert(0, call_tracking["tracking"])

    return company_phones


total_campaigns = 0
smvp_total_campaigns = 0
"""
After export fail with human factor we decided to add checks for campaigns number and YQL target tables
For smvp we count campaigns and geoprod orders are grouped, currently there are 15,5k totally
"""


def validate_total_campaigns(env_type, name, campaign_num, campaign_num_threshold):
    if env_type == "production":
        if campaign_num < campaign_num_threshold:
            raise Exception("total_campaigns=%d is below current threshold %d, check what happens with export!"
                            % (campaign_num, campaign_num_threshold))
        else:
            logging.info("%s generated campaigns number %d is above current threshold %d, all seems OK!"
                         % (name, campaign_num, campaign_num_threshold))


def validate_args(args):
    if args.dst_env == "production":
        if args.src_env != args.dst_env and not args.no_yql:
            raise Exception("You are not allowed to use '%s' as YQL source data when target data is '%s'!"
                            % (args.src_env, args.dst_env))


def resolve_page_ids(testing):
    if testing:
        return TESTING_DEFAULT_PLACEMENT, TESTING_MOBILE_PLACEMENT, TESTING_NAVI_PLACEMENT, []
    else:
        return DEFAULT_PLACEMENT, MOBILE_PLACEMENT, NAVI_PLACEMENT, SERP_PLACEMENT


def generate_advert_smvp(ad_data_list, campaign_ids, permanent_ids, icons_dir, icons_meta,
                         company_name, company_phones, rubrics,
                         text_advert, promo, product_list, branding, click_to_action, headline, call_tracking,
                         partner_links_disabled, settings):
    logging.info("Generating adverts for campaign_ids=%s...", campaign_ids)
    try:
        show_partner_links = not partner_links_disabled
        testing_enabled = is_testing_enabled(settings)

        maps_page_ids, mobile_page_ids, navi_page_ids, serp_page_ids = resolve_page_ids(testing_enabled)

        # we do not expect here any changes https://st.yandex-team.ru/GEOPROD-10927
        realty_rubrics = [31094, 31118, 31616, 1191769628, 3501488242]

        has_realty_rubrics = bool(set(rubrics) & set(realty_rubrics))

        overridden_phones = []

        if call_tracking:
            overridden_phones = [
                phone
                for phone in smvp_apply_call_tracking(call_tracking, company_phones)
                if not bool(phone.get("hide", False))
            ]

        branding = map_by_url_to_cached_branding(branding)

        # single advert for majority of items to avoid significant output growth
        if not has_realty_rubrics:
            maps_page_ids = maps_page_ids + serp_page_ids

        maps_advert = add_advert_data_smvp(
            parent=ad_data_list,
            campaign_ids=campaign_ids,
            permanent_ids=permanent_ids,
            icons_dir=icons_dir,
            icons_meta=icons_meta,
            page_ids=maps_page_ids,
            branding=branding,
            settings=settings,
            cta=click_to_action,
            text_advert=text_advert
        )

        serp_advert = None
        if has_realty_rubrics and serp_page_ids:
            serp_advert = add_advert_data_smvp(
                parent=ad_data_list,
                campaign_ids=campaign_ids,
                permanent_ids=permanent_ids,
                icons_dir=icons_dir,
                icons_meta=icons_meta,
                page_ids=serp_page_ids,
                branding=branding,
                settings=settings,
                cta=None,
                text_advert=text_advert
            )

        mobile_maps_advert = add_advert_mobile_data_smvp(
            parent=ad_data_list,
            permanent_ids=permanent_ids,
            icons_dir=icons_dir,
            icons_meta=icons_meta,
            page_ids=mobile_page_ids,
            branding=branding,
            settings=settings,
            cta=click_to_action,
            text_advert=text_advert
        )

        navi_advert = add_advert_navi_data_smvp(
            parent=ad_data_list,
            permanent_ids=permanent_ids,
            company_name=company_name,
            icons_dir=icons_dir,
            icons_meta=icons_meta,
            page_ids=navi_page_ids,
            branding=branding,
            settings=settings,
            cta=click_to_action,
            promotion=promo,
            text_advert=text_advert
        )

        if GENERATE_BUSINESS_PAGE and not testing_enabled:
            add_advert_data_business(
                parent=ad_data_list,
                permanent_ids=permanent_ids,
                cta=click_to_action,
                settings=settings
            )

        if maps_advert:
            add_rich_smvp(maps_advert, rubrics, headline, promo, product_list, encode_urls=False)
            override_metadata(maps_advert, overridden_phones, show_partner_links)

        if serp_advert:
            add_rich_smvp(serp_advert, rubrics, headline, promo, product_list, encode_urls=False)
            override_metadata(serp_advert, overridden_phones, show_partner_links)

        if navi_advert:
            add_rich_smvp(navi_advert, rubrics, headline, promo, product_list, encode_urls=True)
            override_metadata(navi_advert, overridden_phones, show_partner_links)

        if mobile_maps_advert:
            add_rich_smvp(mobile_maps_advert, rubrics, headline, promo, product_list, encode_urls=True)
            # GEOPROD-3294 - disable partner buttons for mobile if we have our own booking button
            show_partner_links_mobile = show_partner_links and (
                click_to_action is not None and not click_to_action.get(IS_BOOKING)
            )
            override_metadata(mobile_maps_advert, overridden_phones, show_partner_links_mobile)

        # БЯК получает стандартный формат CTA кнопок, потому что поддерживает схему type=Call.
        # добавляем extype на страницу serp, пока bprofile нормально не доезжает до поиска - GEOPROD-2975
        add_actions_smvp(maps_advert, click_to_action, create_new_xml_action, BUSINESS_CTA_ATTR_DEF)

        # Мобильные клиенты умеют только схему type=Url, поэтому пока поставляем им отдельным способом телефоны.
        # Для "Кафе Мостовой" включаем кнопку на новом протоколе для эксперимента:
        # https://yandex.ru/maps/?ll=58.899147%2C52.718038&mode=search&oid=1089729139&ol=biz
        for advert in [x for x in [navi_advert, mobile_maps_advert] if x is not None]:
            if 1089729139 in permanent_ids:
                add_actions_smvp(advert, click_to_action, create_new_xml_action)
            else:
                add_actions_smvp(advert, click_to_action, create_new_xml_action_mobile)

        logging.info("Successfully generated adverts for order_id=%s", campaign_ids)
        global total_campaigns
        total_campaigns = total_campaigns + 1
    except ValueError as e:
        logging.warn(e)


geoprod_root = "//home/geoadv/geoprod_backend"
smvp_orgs_bad_status = []


def generate_smvp_ads(ad_data_list, icons_dir, export_source="smvp_export_data"):
    logging.info("Started smvp ads generation")

    input_table = "{ROOT}/{DST_ENV}/tmp/{TABLE}".format(
        TABLE=export_source,
        ROOT=geoprod_root,
        DST_ENV=YT_ENV["smvp"]["$dst"])

    for row in yt.read_table(input_table, format="yson"):
        settings = row["settings"] if row["settings"] else []
        campaign_ids = row["campaign_ids"]
        permanent_ids = row["permanent_ids"]
        rubrics = row["rubrics"]
        company_phones = row["company_phones"]
        company_name = row["company_name"].decode('utf-8') if row["company_name"] else None

        if not company_name:
            logging.warn("Undefined company_name for permanent_ids=%s", str(permanent_ids))
            smvp_orgs_bad_status.extend(permanent_ids)

        partner_links_disabled = row["partner_links_disabled"]

        text_advert = smvp_get_text_advert_data(row["text_advert"])
        promo = smvp_get_promotion_data(row["promotion"])
        product_list = smvp_get_products_data(row["products"])
        branding = smvp_get_branding_data(row["branding"])
        # for cases where permanent id is used in CTA only one permanent id is present in list (YQL responsibility)
        permanent_id = permanent_ids[0]

        click_to_action = smvp_get_click_to_action_data(row["cta"], permanent_id, rubrics, YT_ENV["smvp"]["$dst"])

        headline = smvp_get_headline_data(row["headline"])
        call_tracking = smvp_get_call_tracking_data(row["call_tracking"])

        generate_advert_smvp(ad_data_list, campaign_ids, permanent_ids, icons_dir, mobile_maps_icons,
                             company_name, company_phones, rubrics,
                             text_advert, promo, product_list, branding, click_to_action, headline,
                             call_tracking, partner_links_disabled, settings)

        global smvp_total_campaigns
        smvp_total_campaigns = smvp_total_campaigns + 1

    logging.info("Finished smvp ads generation")


def generate_testsuite_ads(ad_data_list, icons_dir):
    logging.info("Started testsuite ads generation")

    order_id = 0
    for c in REAL_COMPANIES:
        try:
            order_id = order_id + 1
            logging.info("Generating adverts for order_id=%d...", order_id)
            permanent_ids = c["permalinks"].values()

            add_advert_mobile_data_smvp(
                parent=ad_data_list,
                permanent_ids=permanent_ids,
                icons_dir=icons_dir,
                icons_meta=mobile_maps_icons,
                page_ids=["testsuite"],
                branding=smvp_get_branding_data(make_test_branding()),
                settings=[{"priority_disabled": False}],
                cta={},
                text_advert=smvp_get_text_advert_data(make_test_text_advert(c))
            )
            logging.info("Successfully generated adverts for order_id=%d", order_id)
        except ValueError as e:
            logging.warn(e)

    logging.info("Finished testsuite ads generation")


def main(args):
    global DOWNLOAD_ICONS
    DOWNLOAD_ICONS = not args.no_icons

    global DEFAULT_PLACEMENT
    if args.no_default_page:
        DEFAULT_PLACEMENT.remove("default")

    global GENERATE_BUSINESS_PAGE
    GENERATE_BUSINESS_PAGE = args.generate_business_page

    global YT_ENV
    YT_ENV["smvp"]["$dst"] = args.dst_env
    YT_ENV["smvp"]["$src"] = args.src_env

    logging.info("Command line args: %s", args)
    logging.info("YT_ENV=%s", YT_ENV)
    logging.info("DOWNLOAD_ICONS=%s", DOWNLOAD_ICONS)
    logging.info("YT_CLUSTER=%s", args.yt_cluster)

    validate_args(args)

    if not os.path.exists(args.outputDir):
        os.mkdir(args.outputDir, 0o755)

    icons_dir = os.path.join(args.outputDir, "img")
    if not os.path.exists(icons_dir):
        os.mkdir(icons_dir, 0o755)

    yt.config['proxy']['url'] = args.yt_cluster

    dsc = DataSource(args.yt_cluster, yql_dir=os.path.join(os.path.dirname(__file__), "yql"))

    # latest optimization for x1,5 enhancement merges all in single script,
    # for separate src/dst for tycoon/smvp execute one by one
    if not args.no_yql:
        res = dsc.execute_as_single_script(["smvp_export_data.yql"], YT_ENV["smvp"])

        if len(res.errors) > 0:
            for err in res.errors:
                logging.error("Operation error: %s", str(err))
            raise Exception("YQL operation failed")

    ads_file = os.path.join(args.outputDir, "ads.xml")
    ad_data_list = ET.Element("AdvertDataList", nsmap={None: NS_MAP[None], "xml": NS_MAP["xml"]})
    generate_smvp_ads(ad_data_list, icons_dir)
    generate_testsuite_ads(ad_data_list, icons_dir)
    ET.ElementTree(ad_data_list).write(ads_file, encoding='utf-8', xml_declaration=True, pretty_print=True)

    validate_total_campaigns(YT_ENV["smvp"]["$dst"], "Totally", total_campaigns, 30000)
    validate_total_campaigns(YT_ENV["smvp"]["$dst"], "SMVP", smvp_total_campaigns, 30000)

    tar_file = os.path.join(args.outputDir, "styles.tar.gz")

    for icon_url in default_icons:
        download_image(icons_dir, icon_url)

    make_tarfile(tar_file, icons_dir)


def parse_arguments(params=None):
    parser = argparse.ArgumentParser(description='Generates adverts XML files')
    parser.add_argument('outputDir', help='Output directory')
    parser.add_argument(
        '-ny', '--no-yql', default=False, action='store_const', const=True,
        help='Disable YT data source re-creation with YQL scripts')
    parser.add_argument(
        '-ni', '--no-icons', default=False, action='store_const', const=True,
        help='Disable icons downloading (their names will not be in XML too)')
    parser.add_argument(
        '-de', '--dst-env', default='testing', action='store',
        help='During local development point production and --no-yql to test only XML generation')
    parser.add_argument(
        '-se', '--src-env', default='testing', action='store',
        help='During local development point production and --dst-env=testing to test YQL processing for product data')
    parser.add_argument(
        '-nd', '--no-default-page', default=False, action='store_const', const=True,
        help='Disable producing AdvertData with default page_id')
    parser.add_argument(
        '-bp', '--generate-business-page', default=False, action='store_const', const=True,
        help='Produce AdvertData with bprofile page_id with CTAs')
    parser.add_argument(
        '-yc', '--yt-cluster', default='hahn', action='store', choices=['hahn', 'arnold'],
        help='Cluster for YQL running and input/output YT tables')
    return parser.parse_args(params)


if __name__ == "__main__":
    # noinspection PyUnresolvedReferences
    yt.config['pickling']['module_filter'] = lambda module: 'hashlib' not in getattr(module, '__name__', '') \
                                                            and 'lxml' not in getattr(module, '__name__', '')
    # noinspection PyCallingNonCallable
    yt.update_config({'pickling': {'python_binary': '/skynet/python/bin/python'}})
    logging.basicConfig(level=logging.INFO,
                        format='%(asctime)s %(levelname)-8s %(message)s')
    main(parse_arguments())
