# -*- coding: utf-8 -*-
import re

# translates for currencies
CURRENCIES = {
    "RUB": "р.",
    "BYN": "BYN",
    "KZT": "KZT"
}


REMOVE_WS_PATTERN = re.compile(r'\s+')


def formatted_price_to_valid_number(price):
    if not price or isinstance(price, float) or isinstance(price, int):
        return price

    no_ws_str = re.sub(REMOVE_WS_PATTERN, "", str(price))

    # convert decimal point, need a permanent solution with storing raw, not formatted value in DB
    no_ws_str = '.'.join(no_ws_str.rsplit(',', 1))

    try:
        return int(no_ws_str)
    except ValueError:
        try:
            return float(no_ws_str)
        except ValueError:
            return None


def formatted_price_with_currency(numeric, formatted, currency):
    """
    :return: formatted price with currency in case it has numeric value or just formatted in case it is "for free" or so
    """
    localized_currency = CURRENCIES.get(currency.upper(), currency)
    formatted = str(formatted)

    if not (numeric is None
            or formatted.upper().endswith(currency.upper())
            or formatted.upper().endswith(localized_currency.upper())):
        formatted = formatted + " " + localized_currency

    return formatted


if __name__ == "__main__":
    assert formatted_price_to_valid_number("0,00") == int(0)
    assert formatted_price_to_valid_number("1 000,") == int(1000)
    assert formatted_price_to_valid_number("1 000,00") == int(1000)
    assert formatted_price_to_valid_number("1 000,98") == float(1000.98)
    assert formatted_price_to_valid_number("1 000.98") == float(1000.98)
    assert formatted_price_to_valid_number("1 000,9") == float(1000.9)
    assert formatted_price_to_valid_number("1 000.9") == float(1000.9)

    assert formatted_price_with_currency(1000.98, "1 000,98", "rub") == "1 000,98 р."
    assert formatted_price_with_currency(1000.98, "1 000,98", "Byn") == "1 000,98 BYN"
    assert formatted_price_with_currency(1000.98, "1 000,98 KZT", "kzt") == "1 000,98 KZT"
    assert formatted_price_with_currency(1000.98, "1 000,98 р.", "rub") == "1 000,98 р."
    assert formatted_price_with_currency(10.8, "10,8", "%") == "10,8 %"
    assert formatted_price_with_currency(None, "от 10,8 %", "RUB") == "от 10,8 %"

    pattern = re.compile(
        r'\[Sandbox\] TYCOON_ADVERTS #(\d+) is (FAILURE|DUPLICATES(.*))')
    ss = ["[Sandbox] TYCOON_ADVERTS #777 is FAILURE",
          "[Sandbox] TYCOON_ADVERTS #777 is DUPLICATES on testing",
          "[Sandbox] TYCOON_ADVERTS #777 is DUPLICATES on production"]
    for s in ss:
        m = pattern.match(s)
        if m:
            print("%s was matched" % s)
            print(m.group(1))
        else:
            print("%s was NOT matched" % s)
