from dict_util import safe_get
import functools

"""
The module contains methods for working with 'settings' entity in smvp_export_data
This entity is a list of structures containing configurations of all campaigns associated with permalink
"""


def all_settings_match(settings, path):
    return all([safe_get(s, path) for s in settings])


def all_settings_match_func(settings, f):
    return all([f(s) for s in settings])


def any_settings_match(settings, path):
    return any([safe_get(s, path) for s in settings])


def any_settings_match_func(settings, f):
    return any([f(s) for s in settings])


# Priority is disabled if every campaign has 'priority_disabled' or 'priority_disabled_in_flight' flag
def is_priority_disabled(settings):
    return all_settings_match_func(settings,
                                   lambda s: safe_get(s, "priority_disabled") or safe_get(s,
                                                                                          "priority_disabled_in_flight"))


# Highlight is disabled if every campaign has 'priority_disabled' flag
def is_highlight_disabled(settings):
    return all_settings_match(settings, "priority_disabled")


# Offer in serp is hidden if there are active campaigns and each campaign has the required flag
def is_hide_serp_offer(settings):
    return len(settings) > 0 and all_settings_match(settings, "settings.is_promotion_in_serp_disabled")


# Testing scope is activated if there is at least 1 campaign having the required flag
def is_testing_enabled(settings):
    return any_settings_match(settings, "settings.is_testing")


def is_priority_disabled_single(settings_for_campaign):
    return safe_get(settings_for_campaign, 'priority_disabled', False) or safe_get(settings_for_campaign, 'priority_disabled_in_flight', False)


def merge_priority_rates(left, right):
    return None if any(map(lambda v: v is None, (left, right))) else max(left, right)


def get_priority_rate(settings):
    """
    Get priority rate of settings list. Returns Union[None,float]
    """
    with_priority = filter(lambda s: not is_priority_disabled_single(s), settings)
    priority_rates = list(map(lambda s: safe_get(s, 'priority_rate'), with_priority))
    if priority_rates:
        return functools.reduce(merge_priority_rates, priority_rates)
    else:
        return None


if __name__ == "__main__":
    assert is_priority_disabled([])
    assert not is_priority_disabled([{"priority_disabled": False, "priority_disabled_in_flight": False}])
    assert is_priority_disabled([{"priority_disabled": True, "priority_disabled_in_flight": False}])
    assert is_priority_disabled([{"priority_disabled": False, "priority_disabled_in_flight": True}])
    assert is_priority_disabled([{"priority_disabled": True, "priority_disabled_in_flight": False},
                                 {"priority_disabled": False, "priority_disabled_in_flight": True}])
    assert not is_priority_disabled([{"priority_disabled": False, "priority_disabled_in_flight": False},
                                     {"priority_disabled": True, "priority_disabled_in_flight": True}])

    assert is_highlight_disabled([])
    assert is_highlight_disabled([{"priority_disabled": True, "priority_disabled_in_flight": False}])
    assert not is_highlight_disabled([{"priority_disabled": False, "priority_disabled_in_flight": True}])
    assert not is_highlight_disabled([{"priority_disabled": True, "priority_disabled_in_flight": False},
                                      {"priority_disabled": False, "priority_disabled_in_flight": True}])

    assert not is_hide_serp_offer([])
    assert not is_hide_serp_offer([{"settings": {"is_promotion_in_serp_disabled": False}}])
    assert is_hide_serp_offer([{"settings": {"is_promotion_in_serp_disabled": True}}])
    assert not is_hide_serp_offer([{"settings": {"is_promotion_in_serp_disabled": True}},
                                   {"settings": {"is_promotion_in_serp_disabled": False}}])

    assert not is_testing_enabled([])
    assert not is_testing_enabled([{"settings": {"is_testing": False}}])
    assert is_testing_enabled([{"settings": {"is_testing": True}}])
    assert is_testing_enabled([{"settings": {"is_testing": True}},
                               {"settings": {"is_testing": False}}])

    def pr(value):
        return {'priority_rate': value}

    assert get_priority_rate([]) is None
    assert get_priority_rate([{}]) is None
    assert abs(get_priority_rate([pr(0.5)]) - 0.5) <= 0.001
    assert get_priority_rate([pr(0.5), {}]) is None
    assert abs(get_priority_rate([pr(0.5), pr(0.6)]) - 0.6) <= 0.001
    assert get_priority_rate([pr(0.5), pr(0.6), None]) is None
