# coding: utf-8

import re
from urlparse import urlparse
from urllib3.util import parse_url


def ensure_protocol(url):
    return "https:" + url if url and url.startswith("//") else url


def complete_protocol(url):
    if not url or re.search('^https?://', url):
        return url

    return 'http://' + url


def encode_url(url):
    return parse_url(url).url


def ensure_url_format(url, encode=False):
    if not url:
        return url

    url = url.strip()
    url = ensure_protocol(url)
    url = complete_protocol(url)

    if encode:
        url = encode_url(url)

    return url


def add_query_variable_to_url(url, variable):
    query = urlparse(url).query
    if query:
        url += "&" + variable
    else:
        url += "?" + variable

    return url


if __name__ == "__main__":
    assert ensure_protocol(None) is None
    assert ensure_protocol('http://example.com') == 'http://example.com'
    assert ensure_protocol('example.com') == 'example.com'
    assert ensure_protocol('//example.com') == 'https://example.com'

    assert complete_protocol(None) is None
    assert complete_protocol('http://example.com') == 'http://example.com'
    assert complete_protocol('https://example.com') == 'https://example.com'
    assert complete_protocol('example.com') == 'http://example.com'

    assert encode_url("http://example.com?foo=bar") == "http://example.com?foo=bar"
    assert encode_url("москва.рф") == "xn--80adxhks.xn--p1ai"
    assert encode_url("http://москва.рф?foo=bar") == "http://xn--80adxhks.xn--p1ai?foo=bar"
    assert encode_url("москва.рф?foo=1&бар=2") == "xn--80adxhks.xn--p1ai?foo=1&%D0%B1%D0%B0%D1%80=2"
    assert encode_url("москва.рф?text=%D0%BA%D0%B0%D1%84%D0%B5") == "xn--80adxhks.xn--p1ai?text=%D0%BA%D0%B0%D1%84%D0%B5"
    assert encode_url("http://xn--80adxhks.xn--p1ai?foo=bar") == "http://xn--80adxhks.xn--p1ai?foo=bar"
    assert encode_url("xn--80adxhks.xn--p1ai?text=%D0%BA%D0%B0%D1%84%D0%B5") == "xn--80adxhks.xn--p1ai?text=%D0%BA%D0%B0%D1%84%D0%B5"

    assert ensure_url_format(None) is None
    assert ensure_url_format('http://example.com') == 'http://example.com'
    assert ensure_url_format('  http://example.com ') == 'http://example.com'
    assert ensure_url_format('//example.com') == 'https://example.com'
    assert ensure_url_format('example.com') == 'http://example.com'
    assert ensure_url_format('москва.рф') == 'http://москва.рф'
    assert ensure_url_format('москва.рф', encode=True) == 'http://xn--80adxhks.xn--p1ai'

    assert add_query_variable_to_url("http://example.com", "qqq=eee") == "http://example.com?qqq=eee"
    assert add_query_variable_to_url("http://example.com?rrr=ttt", "qqq=eee") == "http://example.com?rrr=ttt&qqq=eee"
