# -*- coding: utf-8 -*-

import logging

from sandbox import sdk2
from sandbox.common.errors import TaskError
from sandbox.projects.tycoon.task import TycoonTask, JugglerStatus
from sandbox.sandboxsdk.environments import PipEnvironment


class TycoonYqlMetric(TycoonTask):
    """
        Execute YQL query and send metric to Graphite and Juggler
    """
    class Parameters(TycoonTask.Parameters):
        yql_script = sdk2.parameters.ArcadiaUrl(
            description="Metric YQL script",
            required=True,
            default_value="arcadia:arc/trunk/arcadia/sprav/tycoon/scripts"
        )
        metric_name = sdk2.parameters.String(
            "Metric name",
            required=True
        )
        with sdk2.parameters.Group("Juggler parameters") as group_juggler:
            juggler_service = sdk2.parameters.String(
                "Juggler service",
                required=False
            )
            level_warn = sdk2.parameters.Float(
                "Warning level",
                required=False
            )
            level_crit = sdk2.parameters.Float(
                "Critical level",
                required=False
            )

    class Requirements(sdk2.Task.Requirements):
        environments = [PipEnvironment("yql")]

    def calculate_metric(self, query):
        from yql.api.v1.client import YqlClient
        yql_client = YqlClient(db="hahn", token=sdk2.Vault.data(self.owner, "YQL_TOKEN"))
        request = yql_client.query(query)
        request.run()
        for result in request.get_results():
            assert len(result.columns) == 1, "More than one column in result set"
            expected_types = ["Int32", "Int64", "Uint32", "Uint64", "Float", "Double"]
            assert result.columns[0][1] in expected_types, "Expect numeric result type"
            for row in result.rows:
                for item in row:
                    return float(item)
        if not request.is_success:
            raise TaskError("YQL query failed")

    def juggler_status(self, value):
        if self.Parameters.level_crit is not None and value > self.Parameters.level_crit:
            return JugglerStatus.CRIT
        elif self.Parameters.level_warn is not None and value > self.Parameters.level_warn:
            return JugglerStatus.WARN
        else:
            return JugglerStatus.OK

    def on_execute(self):
        query = self.yql_query(sdk2.svn.Arcadia.cat(self.Parameters.yql_script))
        metric_value = self.calculate_metric(query)
        logging.info("Calculated metric value %f", metric_value)

        if self.Parameters.juggler_service:
            self.notify_juggler(self.Parameters.juggler_service, self.juggler_status(metric_value), metric_value)

        self.send_metric(self.Parameters.metric_name, metric_value)
