# coding: utf-8

import logging
import re

from sandbox import sdk2
from sandbox.common import utils
from sandbox.common.types.task import ReleaseStatus
from sandbox.projects.common.juggler import jclient


class JugglerStatus(utils.Enum):
    OK = None
    WARN = None
    CRIT = None


class TycoonEnvironment(utils.Enum):
    utils.Enum.lower_case()
    PRODUCTION = None
    TESTING = None


RELEASE_STATUS_BY_ENV = {
    TycoonEnvironment.TESTING: ReleaseStatus.TESTING,
    TycoonEnvironment.PRODUCTION: ReleaseStatus.STABLE,
}


JUGGLER_HOST_BY_ENV = {
    TycoonEnvironment.TESTING: "tycoon-back-test",
    TycoonEnvironment.PRODUCTION: "tycoon-back-prod",
}

ALTAY_ENV_BY_ENV = {
    TycoonEnvironment.TESTING: "testing",
    TycoonEnvironment.PRODUCTION: "prod",
}


class TycoonTask(sdk2.Task):
    """
        Basic tycoon task
    """

    class Parameters(sdk2.Task.Parameters):
        environment = sdk2.parameters.String(
            "Environment",
            choices=list((v, v) for v in TycoonEnvironment),
            default_value=TycoonEnvironment.TESTING,
            required=True
        )

    def release_status(self):
        return RELEASE_STATUS_BY_ENV.get(self.Parameters.environment)

    def notify_juggler(self, service, status, value=None):
        host = JUGGLER_HOST_BY_ENV.get(self.Parameters.environment)
        jclient.send_events_to_juggler(host, service, status, str(value) if value else str(self.id))
        logging.info("Notification to juggler %s %s %s", host, service, status)

    def send_metric(self, name, value):
        # TODO send metric to solomon
        pass

    def yql_query(self, template, **substitutions):
        subst = {
            "ENV": self.Parameters.environment,
            "ALTAY_ENV": ALTAY_ENV_BY_ENV.get(self.Parameters.environment),
            "EARTH_REGION_ID": str(10000),
        }
        subst.update(substitutions)
        return re.sub(
            r"\{\{(\w+)\}\}",
            lambda match: subst[match.group(1)] if match.group(1) in subst else "",
            template
        )
