# coding: utf-8

import json
import logging
import os
import shutil
import subprocess
import sys
import tarfile

import sandbox.common.types.client as ctc
import sandbox.projects.common.build.parameters
import sandbox.sdk2 as sdk2

from sandbox.projects.common.arcadia import sdk
from sandbox.projects.common.build.YaMake import YaMakeTask
from sandbox.projects.common.nanny import nanny

from sandbox.projects.ugc.BuildUgcServer.python_dist import PutVaultDataToHome
import sandbox.projects.ugc.resources as resources


def _get_unneeded_params_groups():
    groups = set([
        sandbox.projects.common.build.parameters.TEST_SYSTEM_GROUP_NAME,
        sandbox.projects.common.build.parameters.JAVA_GROUP_NAME,
    ])
    for p in YaMakeTask.input_parameters:
        if p.name == 'targets':
            groups.add(p.group)
            break
    return groups


class BuildUgcServer(YaMakeTask, nanny.ReleaseToNannyTask):
    type = 'BUILD_UGC_SERVER'
    client_tags = ctc.Tag.Group.LINUX

    input_parameters = [p for p in YaMakeTask.input_parameters if p.group not in _get_unneeded_params_groups()]

    execution_space = 100 * 1024  # 100 GB

    cores = 24

    package = 'entity/ugc/db/python/package/pkg-pypi.json'

    def get_targets(self):
        return [
            'entity/ugc/server/bin',
            'entity/ugc/server/config',
        ]

    def get_arts(self):
        return [
            {'path': 'entity/ugc/server/bin/ugcserver'},
            {'path': 'entity/ugc/server/config/entity/ugc/server/config'},
        ]

    def get_resources(self):
        return {
            'ugcserver': {
                'resource_type': resources.UgcServerExecutable,
                'description': 'Ugc Server executable',
                'resource_path': 'ugcserver',
            },
            'config': {
                'resource_type': resources.UgcComponentServerConfigDir,
                'description': 'Ugc Server config directory',
                'resource_path': 'config',
            }
        }

    def post_build(self, source_dir, output_dir, pack_dir):
        try:
            self._build_pypi_package(source_dir, output_dir, pack_dir)
        except Exception as e:
            logging.error('Failed to build pypi package: %s' % str(e))

    def on_release(self, additional_parameters):
        additional_parameters['release_comments'] += 'UGC server release\n'
        nanny.ReleaseToNannyTask.on_release(self, additional_parameters)
        YaMakeTask.on_release(self, additional_parameters)

        try:
            self._upload_pypi_package()
        except Exception as e:
            logging.error('Failed to upload pypi package: %s' % str(e))

    def _build_pypi_package(self, source_dir, output_dir, pack_dir):
        # do_package adds 'arcadia' to source root, but do_build don't. C - consistency.
        ya_package_source_root = os.path.dirname(source_dir)
        sdk.do_package(ya_package_source_root, [self.package], checkout=True)

        with open('packages.json') as fp:
            packages = json.load(fp)
            pkg_path = packages[0]['path']
            pkg_file = os.path.basename(pkg_path)
            resources.UgcDbPypiPackage(sdk2.Task[self.id], pkg_file, pkg_path)

        p = subprocess.Popen(
            [sys.executable, 'setup.py', 'preprocess'],
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE)
        out, err = p.communicate()
        if p.returncode != 0:
            raise Exception('Upload command failed: %s' % err)

    def _upload_pypi_package(self):
        task = sdk2.Task[self.id]
        resource = resources.UgcDbPypiPackage.find(task=task).first()
        if resource is None:
            raise Exception('Resource for pypi package not found')
        resource_data = sdk2.ResourceData(resource)
        resource_path = str(resource_data.path)

        pkg_dir = os.path.join(os.path.dirname(resource_path), 'pkg_dir')
        shutil.rmtree(pkg_dir, ignore_errors=True)
        os.mkdir(pkg_dir)

        with tarfile.open(resource_path, 'r:gz') as tar:
            tar.extractall(pkg_dir)
        os.chdir(pkg_dir)

        access_key = sdk2.Vault.data(self.owner, 'robot-ugcdb-pypi-access-key')
        secret_key = sdk2.Vault.data(self.owner, 'robot-ugcdb-pypi-secret-key')

        with PutVaultDataToHome(access_key=access_key, secret_key=secret_key, debug=True):
            p = subprocess.Popen(
                [sys.executable, 'setup.py', 'sdist', 'upload', '--show-response', '-r', 'yandex'],
                stdout=subprocess.PIPE,
                stderr=subprocess.PIPE)
            out, err = p.communicate()
            if p.returncode != 0:
                raise Exception('Upload command failed: %s' % err)

            logging.info('Uploaded pypi package: %s' % out)


__Task__ = BuildUgcServer
