import logging
import copy

import sandbox.common.types.task as ctt

from sandbox import sdk2
from sandbox import common

from sandbox.projects.trendbox_ci.beta.tasks.trendbox_ci_job import TrendboxCiJob
from sandbox.sandboxsdk.channel import channel


class UgcFrontTest(sdk2.Task):
    """Runs front tests for UGC"""

    name = 'SANDBOX_CI_UGC_PAGES_E2E'

    class Parameters(sdk2.Task.Parameters):
        srcrwr = sdk2.parameters.String("backend for rewrite")

    class Context(sdk2.Task.Context):
        subtask_id = None

    def _get_last_task(self):
        trendbox_job_name = 'Hermione e2e: release'
        ugc_pages_reg_exp = 'search-interfaces/frontend@release/ugc-pages.*'

        task = sdk2.Task.find(
            type=TrendboxCiJob,
            description='{}.*\n.*{}'.format(trendbox_job_name, ugc_pages_reg_exp),
            status=ctt.Status.Group.SUCCEED
        ).first()

        if task:
            logging.debug('Find last release task: {task}'.format(task=task))

        if task is None:
            message = 'Could not find success {task} with name {name} and branch regexp: {regexp}'.format(
                task=TrendboxCiJob,
                name=trendbox_job_name,
                regexp=ugc_pages_reg_exp
            )
            channel.sandbox.send_email('ugc-dev@yandex-team.ru', '', 'Error in release e2e tests', message)
            raise common.errors.TaskFailure(message)

        return task

    def _get_subconfigs(self, config, path):
        step = path[0]
        sub_path = path[1:]
        config_part = config.get(step)

        if not sub_path:
            return config_part

        if isinstance(config_part, list):
            return [self._get_subconfigs(el, sub_path) for el in config_part]

        return self._get_subconfigs(config_part, sub_path)

    def _modify_trendbox_config(self, original_trendbox_config):
        trendbox_config = copy.deepcopy(original_trendbox_config)
        pathToEnv = ['lifecycle', 'steps', 'options', 'env']

        tag = trendbox_config['workcopies']['after']['branch']
        if tag.startswith('release/'):
            trendbox_config['workcopies']['after']['tag'] = tag[len('release/'):]
            del trendbox_config['workcopies']['after']['branch']

        if self.Parameters.srcrwr:
            for env in self._get_subconfigs(trendbox_config, pathToEnv):
                env.update({
                    'SRCRWR': self.Parameters.srcrwr
                })

        return trendbox_config

    def _get_subtask_parameters(self, task_to_clone, trendbox_config):
        return dict(
            # Common params
            description='Hermione e2e: other components',
            owner=task_to_clone.owner,
            _container=task_to_clone.Parameters._container,
            build_id=task_to_clone.Parameters.build_id,
            build_request_id=task_to_clone.Parameters.build_request_id,
            webhook_url=task_to_clone.Parameters.webhook_url,
            webhook_payload=task_to_clone.Parameters.webhook_payload,
            use_last_binary_archive=task_to_clone.Parameters.use_last_binary_archive,
            trendbox_cli_resource=task_to_clone.Parameters.trendbox_cli_resource,
            job_id=task_to_clone.Parameters.job_id,
            is_privileged_container=task_to_clone.Parameters.is_privileged_container,
            workcopies=task_to_clone.Parameters.workcopies,
            trendbox_web_logs_url_template=task_to_clone.Parameters.trendbox_web_logs_url_template,
            # Modified config
            trendbox_config=trendbox_config
        )

    def on_execute(self):
        release_task = self._get_last_task()
        modified_trendbox_config = self._modify_trendbox_config(
            release_task.Parameters.trendbox_config,
        )

        with self.memoize_stage.create_children:
            subtask = TrendboxCiJob(
                self,
                **self._get_subtask_parameters(release_task, modified_trendbox_config)
            ).enqueue()
            self.Context.subtask_id = subtask.id

            raise sdk2.WaitTask(subtask, list(ctt.Status.Group.FINISH + ctt.Status.Group.BREAK), True)

        subtask = self.find(id=self.Context.subtask_id).first()

        if subtask.status not in ctt.Status.Group.SUCCEED:
            message = 'Subtask {} ({}) was finished with the status of {}'.format(subtask.type, subtask.id, subtask.status)
            channel.sandbox.send_email('ugc-dev@yandex-team.ru', '', 'Error in release e2e tests', message)
            raise common.errors.TaskFailure(message)
