import yt.wrapper as yt

import argparse
import logging as log
import os
import re
import subprocess
import sys


def run(yt_proxy, yt_token, logs_path, log_tables_count, ammo_path):
    proc_env = dict(os.environ, YT_PROXY=yt_proxy.encode("utf-8"), YT_TOKEN=yt_token.encode("utf-8"))
    p = subprocess.Popen(
        [
            sys.executable, __file__,
            "--logs-path", logs_path,
            "--log-tables-count", str(log_tables_count),
            "--ammo-path", ammo_path,
            "--use-skynet-python"
        ],
        stdin=subprocess.PIPE,
        stdout=subprocess.PIPE,
        stderr=subprocess.PIPE,
        env=proc_env
    )

    (out, err) = p.communicate()
    if out:
        log.info(out)
    if err:
        log.error(err)
    if p.returncode:
        raise Exception("Ammo map failed with code {:d}".format(p.returncode))


class Request:
    def __init__(self, path_regex, methods, tag):
        self.path_regex = re.compile(path_regex)
        self.methods = methods
        self.tag = tag


REQUEST_LIST = [
    Request("^/api/1\.x/biz",            {"GET"}, "/api/1.x/biz"),
    Request("^/ugc/sideblock",           {"GET"}, "/ugc/sideblock"),
    Request("^/ext/market/poll/visitor", {"GET"}, "/ext/market/poll/visitor"),
    Request("^/ext/market/poll/user",    {"GET"}, "/ext/market/poll/user"),
    Request("^/fotki/latest",            {"GET"}, "/fotki/latest"),
    Request("^/ugcview",                 {"GET"}, "/ugcview"),
    #  Request("^/ugcpub/cabinet",          {"GET"}, "/ugcpub/cabinet"),
    Request("^/ugcpub/push-org",         {"GET"}, "/ugcpub/push-org"),
]

REQUEST_TEMPLATE = (
    "%s %s HTTP/1.1\r\n"
    "%s\r\n"
)

AMMO_TEMPLATE = (
    "%d %s\n"
    "%s\n"
)


def make_ammo(row):
    if not row["protocol"] == "http" or not row["status"] == "200":
        return

    req = row["request"]
    method = row["method"]
    tag = "unknown"
    for r in REQUEST_LIST:
        if r.path_regex.match(req) and method in r.methods:
            tag = r.tag
            break
    else:  # current request not in whitelist
        return

    headers = ""
    for header in row["headers"].split('\n'):
        header.strip()
        if header:
            headers += "{}\r\n".format(header)

    req = REQUEST_TEMPLATE % (method, req, headers)
    yield {
        "ammo": AMMO_TEMPLATE % (len(req), tag, req)
    }


def create_parser():
    parser = argparse.ArgumentParser()
    parser.add_argument("--logs-path", type=str, required=True)
    parser.add_argument("--log-tables-count", type=int, required=True)
    parser.add_argument("--ammo-path", type=str, required=True)
    parser.add_argument("--use-skynet-python", action="store_true")
    return parser


def main(args):
    if args.use_skynet_python:
        yt.config["pickling"]["python_binary"] = "/skynet/python/bin/python"

    input_tables = [
        "{}/{}".format(args.logs_path, t)
        for t in yt.list(args.logs_path, sort=True)[-args.log_tables_count:]
    ]

    yt.run_map(
        make_ammo,
        input_tables,
        args.ammo_path
    )


if __name__ == "__main__":
    args = create_parser().parse_args()
    main(args)
