import logging as log

from sandbox import sdk2
from sandbox.projects.common.gencfg.api_client import GencfgApiClient
from sandbox.common import errors
from sandbox.common.types import task as ctt
from sandbox.common.types import resource as ctr
from sandbox.projects.app_host import resources as app_host_resources
from sandbox.projects.release_machine.input_params2 import ComponentName2
from sandbox.projects.release_machine.helpers.startrek_helper import STHelper
from sandbox.projects.tank.ShootViaTankapi import ShootViaTankapi
from sandbox.projects.tank.LoadTestResults import LoadTestResults
from sandbox.projects.ugc.release_quality.PrepareUgcDbAmmo import PrepareUgcDbAmmo

import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.core.task_env as task_env
import sandbox.projects.release_machine.components.all as rmc


GENCFG_API = "http://api.gencfg.yandex-team.ru"
YANDEX_TANK_GENCFG_GROUP = "ALL_RCLOUD_TANKS"
PHANTOM_CONFIG_TEMPLATE = """
[phantom]
rps_schedule={rps_schedule}
writelog = all
address = {target}

[meta]
task = {st_ticket}
job_name = UGC stress test
operator = {operator}
component = {component}
"""

UGC_DB_REGRESS_COMPONENT_FOR_HTTP = "3723"
UGC_DB_REGRESS_COMPONENT_FOR_APPHOST = "3766"
UGC_SERVER_REGRESS_COMPONENT_FOR_HTTP = "3722"
UGC_SERVER_REGRESS_COMPONENT_FOR_APPHOST = ""  # XXX: after first shooting add new component

# XXX: stress shooting must be performed only for SAS intances because blackbox has stress-instances only in SAS
COMPONENTS_INFO = {
    "ugc db":     ["PREPARE_UGC_DB_AMMO",     "UGC_DB_HTTP_AMMO",     "UGC_DB_APPHOST_AMMO",     "SAS_UGC_DB_BACKEND_TEST", UGC_DB_REGRESS_COMPONENT_FOR_HTTP, UGC_DB_REGRESS_COMPONENT_FOR_APPHOST],
    "ugc server": ["PREPARE_UGC_SERVER_AMMO", "UGC_SERVER_HTTP_AMMO", "UGC_SERVER_APPHOST_AMMO", "SAS_UGC_SERVER_TEST", UGC_SERVER_REGRESS_COMPONENT_FOR_HTTP, UGC_SERVER_REGRESS_COMPONENT_FOR_APPHOST],
}


class UgcStressTest(sdk2.Task):
    """Runs load test for UGC"""

    class Requirements(sdk2.Task.Requirements):
        environments = [task_env.TaskRequirements.startrek_client]
        client_tags = task_env.TaskTags.startrek_client

    class Parameters(ComponentName2):
        st_ticket = sdk2.parameters.String(
            "ST ticket for shooting. You need to specify it if you do not use Release Machine")

        release_number = sdk2.parameters.Integer(
            "Release number for release stress test done by Release Machine",
            default=None)

        with sdk2.parameters.Group("Ammo parameters"):
            generate_ammo = sdk2.parameters.Bool(
                "Generate ammo? Uses latest ready resource if False",
                required=True,
                default=True)

            with generate_ammo.value[True]:
                access_log_path = sdk2.parameters.String(
                    "YT path to access logs",
                    default_value=PrepareUgcDbAmmo.Parameters.access_log_path.default)
                ammo_count = sdk2.parameters.Integer(
                    "Number of ammos",
                    default_value=PrepareUgcDbAmmo.Parameters.ammo_count.default)
                apphost_ammo_gen_tool = sdk2.parameters.Resource(
                    "Tool for making apphost ammon from raw contexts [make_tank_ammo]",
                    resource_type=app_host_resources.APP_HOST_TOOL_MAKE_TANK_AMMO_EXECUTABLE,
                    default_value=PrepareUgcDbAmmo.Parameters.make_ammo.default)

        rps_schedule = sdk2.parameters.String(
            "Shooting schedule in phantom format",
            required=True,
            default="line(10, 100, 5m) const(100, 20m)")

        with sdk2.parameters.String("Component for stress testing") as component:
            component.values["ugc db"] = component.Value(value="ugc_db", default=True)
            component.values["ugc server"] = component.Value(value="ugc_server")

        notification_list = sdk2.parameters.String(
            "Email for notification about problem",
            required=True,
            default="ugc-dev@yandex-team.ru")

    class Context(sdk2.Context):
        shoot_data = {
            "http": {
                "tanks": [],
                "config": "",
                "ammo_resource_id": None,
                "lunapark_link": "",
                "shoot_id": None,
            },
            "apphost": {
                "tanks": [],
                "config": "",
                "ammo_resource_id": None,
                "lunapark_link": "",
                "shoot_id": None,
            },
        }
        report_tasks = []
        component = {
            "gen_ammo_task": "",
            "ammo_http_resource": "",
            "ammo_apphost_resource": "",
            "gencfg_group": "",
            "http": "",  # id of component for shooting
            "apphost": "",  # of component for shooting
        }

    def _get_targets(self, gencfg, dc, shoot_type):
        group = gencfg.get_group_instances(self.Context.component["gencfg_group"])
        instances = [
            "{host}:{port}".format(host=inst["hbf"]["interfaces"]["backbone"]["hostname"], port=(inst["port"] + 1 if shoot_type == "apphost" else inst["port"]))
            for inst in group["instances"] if inst["dc"] == dc]
        assert instances
        return instances

    def _get_tanks(self, gencfg, dc):
        group = gencfg.get_group_instances(YANDEX_TANK_GENCFG_GROUP)
        instances = [
            "{host}:{port}".format(host=inst["hostname"], port=inst["port"])
            for inst in group["instances"] if inst["dc"] == dc]
        assert instances
        return instances

    def _get_ammo_resource(self, shoot_type):
        ammo_resource = sdk2.Resource[self.Context.component["ammo_{}_resource".format(shoot_type)]]
        resource = ammo_resource.find(
            state=ctr.State.READY,
        ).order(-sdk2.Resource.id).first()
        if not resource:
            raise errors.TaskError("Ugc ammo not found")
        else:
            sdk2.ResourceData(resource)
            return resource

    def _generate_ammo(self):
        ammo_generation = sdk2.Task[self.Context.component["gen_ammo_task"]](
            self,
            description="Generate ammo for task {}".format(self.id),
            owner=self.owner,
            access_log_path=self.Parameters.access_log_path,
            ammo_count=self.Parameters.ammo_count,
            make_ammo=self.Parameters.apphost_ammo_gen_tool,
        ).enqueue()

        raise sdk2.WaitTask(
            [ammo_generation.id],
            ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
            wait_all=True)

    def _get_st_ticket(self, release_number, component_name):
        if release_number and component_name:
            st_helper = STHelper(sdk2.Vault.data(rm_const.COMMON_TOKEN_OWNER, rm_const.COMMON_TOKEN_NAME))
            c_info = rmc.COMPONENTS[component_name]()
            return st_helper.find_ticket_by_release_number(release_number, c_info).key
        return self.Parameters.st_ticket

    def _prepare_shooting(self, shoot_type="http"):
        gencfg = GencfgApiClient(GENCFG_API)
        targets = self._get_targets(gencfg, "sas", shoot_type)

        self.Context.shoot_data[shoot_type]["tanks"] = self._get_tanks(gencfg, "sas")
        self.Context.shoot_data[shoot_type]["ammo_resource_id"] = self._get_ammo_resource(shoot_type).id
        self.Context.shoot_data[shoot_type]["config"] = PHANTOM_CONFIG_TEMPLATE.format(
            rps_schedule=self.Parameters.rps_schedule,
            target=targets[0],
            st_ticket=self._get_st_ticket(self.Parameters.release_number, self.Parameters.component_name),
            operator=self.author,
            component=self.Context.component[shoot_type])

    def _start_shooting(self, shoot_type="http"):
        shoot_data = self.Context.shoot_data[shoot_type]
        shoot = ShootViaTankapi(
            self,
            description="UGC stress test shooting",
            ammo_source="resource",
            ammo_resource=shoot_data["ammo_resource_id"],
            use_monitoring=True,
            monitoring_source="file",
            config_source="file",
            config_content=shoot_data["config"],
            tanks=shoot_data["tanks"]
        ).enqueue()

        self.Context.shoot_data[shoot_type]["shoot_id"] = shoot.id
        log.info("Start stress shooting")
        raise sdk2.WaitTask(
            [shoot.id],
            ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
            wait_all=True,
            timeout=14400)

    def _start_reporting(self, shoot_type="http"):
        self.Context.shoot_data[shoot_type]["lunapark_link"] = sdk2.Task[self.Context.shoot_data[shoot_type]["shoot_id"]].Parameters.lunapark_link
        lunapark_shoot_id = sdk2.Task[self.Context.shoot_data[shoot_type]["shoot_id"]].Parameters.lunapark_job_id
        if lunapark_shoot_id:
            report = LoadTestResults(
                self,
                only_regression=True,
                shoot_id=lunapark_shoot_id,
                report_type="const",
                additional_message="UGC stress test shooting result",
                send_letter=True,
                mail_recipients=self.Parameters.notification_list
            ).enqueue()
            self.Context.report_tasks.append(report.id)

    def on_enqueue(self):
        if not ((self.Parameters.release_number and self.Parameters.component_name) or self.Parameters.st_ticket):
            raise errors.TaskError("Neither st_tisket, nor release_number with component_name not found")

    def on_execute(self):
        self.Context.component["gen_ammo_task"] = COMPONENTS_INFO[self.Parameters.component][0]
        self.Context.component["ammo_http_resource"] = COMPONENTS_INFO[self.Parameters.component][1]
        self.Context.component["ammo_apphost_resource"] = COMPONENTS_INFO[self.Parameters.component][2]
        self.Context.component["gencfg_group"] = COMPONENTS_INFO[self.Parameters.component][3]
        self.Context.component["http"] = COMPONENTS_INFO[self.Parameters.component][4]
        self.Context.component["apphost"] = COMPONENTS_INFO[self.Parameters.component][5]

        with self.memoize_stage.generate_ammo:
            if self.Parameters.generate_ammo:
                self._generate_ammo()

        log.info("Start http stress shooting")
        self._prepare_shooting()

        with self.memoize_stage.http_shooting:
            self._start_shooting()

        with self.memoize_stage.http_report:
            self._start_reporting()

        log.info("Start apphost stress shooting")
        self._prepare_shooting("apphost")

        with self.memoize_stage.apphost_shooting:
            self._start_shooting("apphost")

        with self.memoize_stage.apphost_report:
            self._start_reporting("apphost")

        with self.memoize_stage.reporting:
            if self.Context.report_tasks:
                log.info("Wait reporting")
                raise sdk2.WaitTask(
                    self.Context.report_tasks,
                    ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
                    wait_all=True
                )
