# -*- coding: utf-8 -*-

import json
import logging
import os
import re
import shutil

import sandbox.common.errors as errors
import sandbox.common.types.client as ctc
import sandbox.common.types.resource as ctr
import sandbox.common.types.task as ctt

from sandbox import sdk2
from sandbox.common import enum
from sandbox.projects.common import constants
from sandbox.projects.common import link_builder
from sandbox.projects.common.arcadia import sdk as arcadiasdk
from sandbox.projects.common.build import parameters as build_parameters
from sandbox.projects.common.build.KosherYaPackage import KosherYaPackage
from sandbox.projects.common.build.ya_package_config import consts as ya_package_consts

from ..resources import (
    US_RT_BINARIES, US_RT_CONFIGS, US_RT_DATA_FILES,
    US_RT_RESOURCES_SPEC, US_RT_DEPLOY_TOOL, US_RT_STAGE_CONFIG, US_RT_MISC,
    DEFAULT_RESOURCE_TTL
)


class ReleaseStatus(enum.Enum):
    """ Release statuses. """

    enum.Enum.preserve_order()
    enum.Enum.lower_case()

    STABLE = None
    HAMSTER = None
    DEFAULT_PRESTABLE = None
    PRESTABLE = None
    PRESTABLE_AB = None
    PRESTABLE_ZERO = None
    PRESTABLE_HEAVY = None


RELEASE_STATUS_MATCH = {
    ReleaseStatus.STABLE: ctt.ReleaseStatus.STABLE,
    ReleaseStatus.HAMSTER: ctt.ReleaseStatus.STABLE,
    ReleaseStatus.DEFAULT_PRESTABLE: ctt.ReleaseStatus.PRESTABLE,
    ReleaseStatus.PRESTABLE: ctt.ReleaseStatus.PRESTABLE,
    ReleaseStatus.PRESTABLE_ZERO: ctt.ReleaseStatus.PRESTABLE,
    ReleaseStatus.PRESTABLE_AB: ctt.ReleaseStatus.PRESTABLE,
    ReleaseStatus.PRESTABLE_HEAVY: ctt.ReleaseStatus.PRESTABLE,
}


class Stages(object):
    MAIN = 'user-sessions-rt'
    HAMSTER = 'user-sessions-rt-hamster'
    PRESTABLE = 'user-sessions-rt-prestable'
    PRESTABLE_ZERO = 'user-sessions-rt-prestable-zero'
    PRESTABLE_AB = 'user-sessions-rt-prestable-ab'
    PRESTABLE_HEAVY = 'user-sessions-rt-prestable-heavy'

    @classmethod
    def get_all_stages(cls):
        return [cls.PRESTABLE, cls.HAMSTER, cls.MAIN, cls.PRESTABLE_AB, cls.PRESTABLE_ZERO, cls.PRESTABLE_HEAVY]

    @classmethod
    def get_stages_by_release_status(cls, release_status):
        if release_status == ReleaseStatus.STABLE:
            return [cls.HAMSTER, cls.MAIN, cls.PRESTABLE, cls.PRESTABLE_AB]

        if release_status == ReleaseStatus.HAMSTER:
            return [cls.HAMSTER]

        if release_status == ReleaseStatus.DEFAULT_PRESTABLE:
            return [cls.PRESTABLE_ZERO, cls.PRESTABLE_HEAVY]

        if release_status == ReleaseStatus.PRESTABLE:
            return [cls.PRESTABLE]

        if release_status == ReleaseStatus.PRESTABLE_AB:
            return [cls.PRESTABLE_AB]

        if release_status == ReleaseStatus.PRESTABLE_ZERO:
            return [cls.PRESTABLE_ZERO]

        if release_status == ReleaseStatus.PRESTABLE_HEAVY:
            return [cls.PRESTABLE_HEAVY]

        raise ValueError('Unknown release status: {!r}'.format(release_status))


class ArtifactTypes(object):
    BINARIES = 'binaries'
    CONFIGS = 'configs'
    DATA = 'data'
    DEPLOY_TOOL = 'deploy_tool'


class FindModes(object):
    BY_ATTRS = 'by_attrs'
    BY_SUBTASK = 'by_subtask'


class ConfigTypes(object):
    NEW = 'new'
    OLD = 'old'
    DIFF = 'diff'


class StageSettings(object):
    def __init__(self, need_update, update_non_globs, cleanup_globs, globs):
        self.need_update = need_update
        self.update_non_globs = update_non_globs
        self.cleanup_globs = cleanup_globs
        self.globs = globs


class YaMakeArtifact(object):
    def __init__(self, target, artifact_rel_path):
        self.target = target
        self.artifact = os.path.join(target, artifact_rel_path)


LOGGER = logging.getLogger(__name__)

RESOURCE_TYPES_MAPPING = {
    ArtifactTypes.BINARIES: US_RT_BINARIES,
    ArtifactTypes.CONFIGS: US_RT_CONFIGS,
    ArtifactTypes.DATA: US_RT_DATA_FILES,
    ArtifactTypes.DEPLOY_TOOL: US_RT_DEPLOY_TOOL,
}

PACKAGES_MAPPING = {
    ArtifactTypes.BINARIES: 'quality/user_sessions/rt/packages/us-rt-binaries/pkg.json',
    ArtifactTypes.CONFIGS: 'quality/user_sessions/rt/packages/us-rt-config-generators/pkg.json',
    ArtifactTypes.DATA: 'quality/user_sessions/rt/packages/us-rt-resources/pkg.json',
}

YA_MAKE_ARTIFACTS_MAPPING = {
    ArtifactTypes.DEPLOY_TOOL: YaMakeArtifact('quality/user_sessions/rt/deploy', 'manage-stage'),
}

ARCADIA_TRUNK_URL = 'arcadia:/arc/trunk/arcadia'
ARCADIA_SPECS_DIR = 'quality/user_sessions/rt/deploy/resource_specs'
ARCADIA_COMMIT_AUTHOR = 'zomb-sandbox-rw'

KEY_DEFAULT_RESOURCES = 'default_resources'
KEY_CUSTOM_RESOURCES = 'custom_resources'

YA_PACKAGE_TASK_NAME = 'KOSHER_YA_PACKAGE'
YA_MAKE_TASK_NAME = 'KOSHER_YA_MAKE'


class PackageParameters(KosherYaPackage.Parameters):
    strip_binaries = build_parameters.StripBinaries2(default_value=True)
    full_strip_binaries = build_parameters.FullStripBinaries(default_value=True)
    save_build_output = build_parameters.SaveBuildOutput(default_value=False)
    use_new_format = build_parameters.UseNewFormat(default_value=True)
    package_type = build_parameters.PackageType(default_value=ya_package_consts.PackageType.TARBALL.value)


class UserSessionsRtReleaser(sdk2.Task):
    """Build & release resources for User Sessions RT processes"""

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.Group.LINUX
        disk_space = 1024  # 1Gb

    class Parameters(sdk2.Task.Parameters):
        # common parameters
        priority = ctt.Priority(ctt.Priority.Class.USER, ctt.Priority.Subclass.NORMAL)
        kill_timeout = 90 * 60  # 90 minutes
        max_restarts = 2

        # custom parameters
        with sdk2.parameters.Group('Common release settings') as common_settings:
            with sdk2.parameters.CheckGroup('Resources to build and update') as deployable_artifacts:
                deployable_artifacts.values[ArtifactTypes.BINARIES] = deployable_artifacts.Value('Binaries', checked=True)
                deployable_artifacts.values[ArtifactTypes.CONFIGS] = deployable_artifacts.Value('Configs', checked=True)
                deployable_artifacts.values[ArtifactTypes.DATA] = deployable_artifacts.Value('Data files', checked=False)

            enable_auto_deploy = sdk2.parameters.Bool('Enable auto deploy', default=True)
            with enable_auto_deploy.value[True]:
                dctl_token = sdk2.parameters.YavSecret(
                    'Secret with token for dctl authorization',
                    default='sec-01f3p4fxy8h0szsp64qfb4edsd#token',
                )
                vault_token = sdk2.parameters.YavSecret(
                    'Secret with token for vault authorization',
                    default='sec-01f3pgac89tae1wwd8g4wbb2pe#token',
                )

        with sdk2.parameters.Group('Stages and deploy units customization') as stages_settings:
            update_main_stage = sdk2.parameters.Bool('Update main stage resources', default=True)
            with update_main_stage.value[True]:
                update_main_non_globs = sdk2.parameters.Bool('Update main stage resources for non-globbed units', default=True)
                cleanup_main_globs = sdk2.parameters.Bool('Cleanup existing main stage globs', default=True)
                main_globs = sdk2.parameters.List('Main stage deploy units (globs)', default=[])

            update_hamster_stage = sdk2.parameters.Bool('Update hamster stage resources', default=True)
            with update_hamster_stage.value[True]:
                update_hamster_non_globs = sdk2.parameters.Bool('Update hamster stage resources for non-globbed units', default=True)
                cleanup_hamster_globs = sdk2.parameters.Bool('Cleanup existing hamster stage globs', default=True)
                hamster_globs = sdk2.parameters.List('Hamster stage deploy units (globs)', default=[])

            update_prestable_stage = sdk2.parameters.Bool('Update prestable resources', default=True)
            with update_prestable_stage.value[True]:
                update_prestable_non_globs = sdk2.parameters.Bool('Update prestable stage resources for non-globbed units', default=True)
                cleanup_prestable_globs = sdk2.parameters.Bool('Cleanup existing prestable stage globs', default=True)
                prestable_globs = sdk2.parameters.List('Prestable stage deploy units (globs)', default=[])

            update_prestable_ab_stage = sdk2.parameters.Bool('Update prestable-ab resources', default=True)
            with update_prestable_ab_stage.value[True]:
                update_prestable_ab_non_globs = sdk2.parameters.Bool('Update prestable ab stage resources for non-globbed units', default=True)
                cleanup_prestable_ab_globs = sdk2.parameters.Bool('Cleanup existing prestable ab stage globs', default=True)
                prestable_ab_globs = sdk2.parameters.List('Prestable ab stage deploy units (globs)', default=[])

            update_prestable_zero_stage = sdk2.parameters.Bool('Update prestable-zero resources', default=True)
            with update_prestable_zero_stage.value[True]:
                update_prestable_zero_non_globs = sdk2.parameters.Bool('Update prestable zero stage resources for non-globbed units', default=True)
                cleanup_prestable_zero_globs = sdk2.parameters.Bool('Cleanup existing prestable zero stage globs', default=True)
                prestable_zero_globs = sdk2.parameters.List('Prestable zero stage deploy units (globs)', default=[])

            update_prestable_heavy_stage = sdk2.parameters.Bool('Update prestable-heavy resources', default=True)
            with update_prestable_heavy_stage.value[True]:
                update_prestable_heavy_non_globs = sdk2.parameters.Bool('Update prestable heavy stage resources for non-globbed units', default=True)
                cleanup_prestable_heavy_globs = sdk2.parameters.Bool('Cleanup existing prestable heavy stage globs', default=True)
                prestable_heavy_globs = sdk2.parameters.List('Prestable heavy stage deploy units (globs)', default=[])

        # package parameters
        with sdk2.parameters.Group('Package build settings') as package_settings:
            pack_params = PackageParameters()

    class Context(sdk2.Task.Context):
        # artifact => resource_id
        resource_ids = None
        # artifact => subtask_id
        subtask_ids = None
        # stage => resource_id
        spec_resource_ids = None
        # config_type => stage => resource_id
        config_resource_ids = None
        release_statuses = None
        ready_for_deploy = False
        prepared = False

    def on_prepare(self):
        if self.Context.prepared:
            LOGGER.info('Skip preparing: already prepared')
            return

        self.Context.resource_ids = {}
        self.Context.subtask_ids = {}
        self.Context.spec_resource_ids = {}
        self.Context.release_statuses = []

        if self.Parameters.enable_auto_deploy:
            self.Context.config_resource_ids = {}

        # allow resources reusage if there is no patch and url contains specific revision
        if (
            not self.Parameters.arcadia_patch and
            re.search(r'@\d+$', self.Parameters.checkout_arcadia_from_url) is not None
        ):
            for artifact in self._get_extended_build_artifacts():
                self._try_fill_resource_id(artifact, find_mode=FindModes.BY_ATTRS)

        self.Context.prepared = True

        self.Context.save()

    def on_execute(self):
        with self.memoize_stage.first_step:
            self._run_subtasks()

            if self.Context.subtask_ids:
                raise sdk2.WaitTask(
                    self.Context.subtask_ids.values(),
                    ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
                    wait_all=True,
                )

        with self.memoize_stage.second_step:
            self._check_subtask_statuses()

            for artifact in self._get_extended_build_artifacts():
                self._try_fill_resource_id(artifact, find_mode=FindModes.BY_SUBTASK)

            self._make_spec_resources()

            if self.Parameters.enable_auto_deploy:
                self._make_stage_config_resources()

                self.Context.ready_for_deploy = True
                self.Context.save()

    def get_release_status(self, parameters):
        if 'release_tag' in parameters:
            release_tag = parameters['release_tag'].lower().strip()
        else:
            release_tag = parameters.get('release_subject', '').lower().strip()

        release_status = parameters['release_status']
        if release_status == ctt.ReleaseStatus.STABLE:
            if release_tag == 'hamster':
                return ReleaseStatus.HAMSTER
            if release_tag == '':
                return ReleaseStatus.STABLE

        if release_status == ctt.ReleaseStatus.PRESTABLE:
            if release_tag == '':
                return ReleaseStatus.DEFAULT_PRESTABLE
            if release_tag == 'ab':
                return ReleaseStatus.PRESTABLE_AB
            if release_tag == 'zero':
                return ReleaseStatus.PRESTABLE_ZERO
            if release_tag == 'preprod':
                return ReleaseStatus.PRESTABLE
            if release_tag == 'heavy':
                return ReleaseStatus.PRESTABLE_HEAVY

        raise errors.ReleaseError('Release actions. Unknown release_tag {}'.format(release_tag))

    def on_release(self, parameters):
        LOGGER.debug('Release parameters: %s', parameters)

        info_lines = [
            'Initiated release with parameters:',
            '  - release_status: {!r}'.format(parameters['release_status']),
            '  - subject: {!r}'.format(parameters.get('release_subject')),
            '  - release_tag: {!r}'.format(parameters.get('release_tag')),
        ]

        self.set_info('\n'.join(info_lines), do_escape=False)

        release_status = self.get_release_status(parameters)
        LOGGER.debug('Release status: %s', release_status)

        current_statuses = self._load_release_statuses()

        if release_status in current_statuses:
            LOGGER.info('Skip release actions, already released')
            return

        parameters['release_comments'] = 'Released to: {}'.format(release_status)

        if self.Context.ready_for_deploy:
            for stage in Stages.get_stages_by_release_status(release_status):
                self._deploy(stage, release_status)

        self._mark_released_artifacts(release_status)

        self._add_release_status(release_status)

    @property
    def release_template(self):
        return sdk2.ReleaseTemplate(
            subject='zero/ab/hamster',
            message='Choose "stable" to release to preprod, hamster, prestable-ab and production.\n'
                    'Choose "stable", print "hamster" in Subject to release to hamster only.\n'
                    'Choose "prestable" and print in subject:\n'
                    '  - nothing for default prestable release\n'
                    '  - "ab" for release to prestable-ab\n'
                    '  - "zero" for release to prestable-zero\n'
                    '  - "preprod" for release to preprod\n'
                    '  - "heavy" for release to prestable-heavy\n',
            types=[
                ctt.ReleaseStatus.PRESTABLE,
                ctt.ReleaseStatus.STABLE,
            ]
        )

    def _mark_released_artifacts(self, release_status):
        LOGGER.info('Mark releaseable artifact resources')

        for resource_id in self.Context.resource_ids.values():
            resource = sdk2.Resource[resource_id]
            if resource.type.releasable and resource.state == ctr.State.READY:
                resource.ttl = 'inf'
                resource.released = RELEASE_STATUS_MATCH[release_status]

    def _load_release_statuses(self):
        return set(self.Context.release_statuses)

    def _clear_release_statuses(self):
        self.Context.release_statuses = []
        self.Context.save()

    def _add_release_status(self, release_status):
        data = set(self.Context.release_statuses)
        data.add(release_status)
        if release_status == ReleaseStatus.STABLE:
            data.add(ReleaseStatus.HAMSTER)
        self.Context.release_statuses = list(data)
        self.Context.save()

    def _deploy(self, stage, release_status):
        new_config_resource_ids = self.Context.config_resource_ids.setdefault(
            ConfigTypes.NEW, {}
        )

        config_resource_id = new_config_resource_ids.get(stage)
        if config_resource_id is None:
            LOGGER.info('Skip release actions for %s, no changes in config', stage)
            return

        self.set_info('Deploy config for {} stage'.format(stage), do_escape=False)

        tool_resource = sdk2.Resource[self.Context.resource_ids[ArtifactTypes.DEPLOY_TOOL]]
        tool_data = sdk2.ResourceData(tool_resource)

        config_resource = sdk2.Resource[config_resource_id]
        config_data = sdk2.ResourceData(config_resource)

        dctl_token = self.Parameters.dctl_token.data()[self.Parameters.dctl_token.default_key]

        with arcadiasdk.mount_arc_path(self.Parameters.checkout_arcadia_from_url, fallback=True) as arcadia_dir:
            deploy_env = os.environ.copy()
            deploy_env.update(
                DCTL_YP_TOKEN=dctl_token,
                US_RT_YA_TOOL=os.path.join(str(arcadia_dir), 'ya'),
            )

            with sdk2.helpers.ProcessLog(self, logger='deploy-{}'.format(stage)) as pl:
                sdk2.helpers.subprocess.check_call(
                    [str(tool_data.path), stage, 'deploy', '-cf', str(config_data.path)],
                    env=deploy_env,
                    stdout=pl.stdout,
                    stderr=pl.stderr,
                )

        if config_resource.type.releasable:
            config_resource.ttl = 'inf'
            config_resource.released = RELEASE_STATUS_MATCH[release_status]

        self._commit_spec(stage)

    def _commit_spec(self, stage):
        LOGGER.info('Prepare spec commit to arcadia')

        spec_resource = sdk2.Resource[self.Context.spec_resource_ids[stage]]
        spec_data = sdk2.ResourceData(spec_resource)
        LOGGER.debug('Resource spec file: %s', str(spec_data.path))

        filename = '{}.json'.format(stage)
        LOGGER.debug('Spec filename: %s', filename)
        svn_url = sdk2.svn.Arcadia.trunk_url(ARCADIA_SPECS_DIR)
        LOGGER.debug('Svn url: %s', svn_url)
        svn_dir = sdk2.svn.Arcadia.get_arcadia_src_dir(svn_url)
        LOGGER.debug('Svn checkout directory: %s', svn_dir)
        svn_file = os.path.join(svn_dir, filename)
        LOGGER.debug('Svn spec file: %s', svn_file)

        if not os.path.isfile(svn_file):
            raise errors.ReleaseError('Spec file not found in checkouted arcadia')

        shutil.copy(str(spec_data.path), svn_file)

        message = 'Update {} resources spec SKIP_CHECK'.format(stage)

        LOGGER.info('Commit resources spec to arcadia')
        LOGGER.debug('Commit message: %s', message)
        LOGGER.debug('Commit author: %s', ARCADIA_COMMIT_AUTHOR)

        sdk2.svn.Arcadia.commit(svn_dir, message, user=ARCADIA_COMMIT_AUTHOR)

    def _get_extended_build_artifacts(self):
        if not self.Parameters.enable_auto_deploy:
            return self.Parameters.deployable_artifacts

        return self.Parameters.deployable_artifacts + [ArtifactTypes.DEPLOY_TOOL]

    def _get_common_vcs_attrs(self):
        return {
            'checkout_arcadia_from_url': self.Parameters.checkout_arcadia_from_url,
            'arcadia_patch': self.Parameters.arcadia_patch,
            'build_type': self.Parameters.build_type
        }

    def _set_artifact_resource_attrs(self, resource):
        if not resource:
            return

        attrs = self._get_common_vcs_attrs()
        for attr_name, attr_value in attrs.items():
            setattr(resource, attr_name, attr_value)

    @staticmethod
    def _try_get_unique_resource(find_query):
        resources = list(find_query.limit(2))
        count = len(resources)

        if count == 1:
            return resources[0]

        if count > 1:
            LOGGER.info('Found several resources, discard them')
        else:
            LOGGER.info('No resources found')

        return None

    def _try_fill_resource_id(self, artifact, find_mode):
        if artifact in self.Context.resource_ids:
            return

        resource_type = RESOURCE_TYPES_MAPPING[artifact]

        LOGGER.info('Lookup unique %s resource for %s artifact', resource_type.name, artifact)

        resource = None

        if find_mode == FindModes.BY_SUBTASK:
            subtask_id = self.Context.subtask_ids.get(artifact)
            if subtask_id is None:
                raise errors.TaskFailure('Subtask for artifact {} not found'.format(artifact))
            LOGGER.debug('Lookup resource by subtask id: %s', subtask_id)
            find_query = resource_type.find(state=ctr.State.READY, task_id=subtask_id)
            resource = self._try_get_unique_resource(find_query)
            self._set_artifact_resource_attrs(resource)
        elif find_mode == FindModes.BY_ATTRS:
            resource_attrs = self._get_common_vcs_attrs()
            LOGGER.debug('Lookup resource by attributes: %s', resource_attrs)
            find_query = resource_type.find(state=ctr.State.READY, attrs=resource_attrs)
            resource = self._try_get_unique_resource(find_query)
        else:
            raise errors.TaskFailure('Unknown find mode: {!r}'.format(find_mode))

        if resource and resource.state == ctr.State.READY:
            self.Context.resource_ids[artifact] = resource.id
            self.Context.save()
            resource_link = link_builder.resource_link(resource.id)
            self.set_info(
                'Found ready {} resource: {}'.format(resource_type.name, resource_link),
                do_escape=False,
            )
        elif find_mode == FindModes.BY_SUBTASK:
            raise errors.TaskFailure('Unique resource {} not found in subtask'.format(resource_type.name))

    def _check_subtask_statuses(self):
        for subtask_id in self.Context.subtask_ids.values():
            if sdk2.Task[subtask_id].status == ctt.Status.SUCCESS:
                continue

            subtask_link = link_builder.task_link(subtask_id)

            self.set_info('Subtask {} failed'.format(subtask_link), do_escape=False)

            raise errors.TaskFailure('Subtask failed')

    def _run_subtasks(self):
        for artifact in self._get_extended_build_artifacts():
            if artifact in self.Context.resource_ids:
                LOGGER.info('Found resource for artifact %s, skip subtask creation', artifact)
                continue

            self._run_artifact_subtask(artifact)

    def _run_artifact_subtask(self, artifact):
        if artifact in self.Context.subtask_ids:
            LOGGER.info('Found subtask for artifact %s, skip subtask creation', artifact)
            return

        LOGGER.info('Prepare subtask for %s artifact', artifact)

        if artifact in PACKAGES_MAPPING:
            subtask = self._run_ya_package(artifact)
        elif artifact in YA_MAKE_ARTIFACTS_MAPPING:
            subtask = self._run_ya_make(artifact)
        else:
            raise errors.TaskFailure('Cannot run subtask for {} artifact'.format(artifact))

        self.Context.subtask_ids[artifact] = subtask.id
        self.Context.save()

        subtask_link = link_builder.task_link(subtask.id)

        self.set_info(
            'Run subtask {} to build {} resource'.format(subtask_link, RESOURCE_TYPES_MAPPING[artifact]),
            do_escape=False,
        )

    def _run_ya_package(self, artifact):
        resource_type = RESOURCE_TYPES_MAPPING[artifact]

        subtask_requirements=dict(
            client_tags=ctc.Tag.LINUX_XENIAL,
        )

        subtask = sdk2.Task[YA_PACKAGE_TASK_NAME](
            self,
            __requirements__=subtask_requirements,
            # common parameters
            description='Subtask of {} (to build {})'.format(self.id, resource_type.name),
            **dict(self.Parameters.pack_params)
        )
        subtask.Parameters.packages = PACKAGES_MAPPING[artifact]
        subtask.Parameters.resource_type = resource_type.name
        subtask.Parameters.package_ttl = DEFAULT_RESOURCE_TTL

        subtask.save().enqueue()

        return subtask

    def _run_ya_make(self, artifact):
        resource_type = RESOURCE_TYPES_MAPPING[artifact]

        ya_make_artifact = YA_MAKE_ARTIFACTS_MAPPING[artifact]

        subtask = sdk2.Task[YA_MAKE_TASK_NAME](
            self,
            # common parameters
            description='Subtask of {} (to build {})'.format(self.id, resource_type.name),
            kill_timeout=self.Parameters.kill_timeout,
            # task specific parameters
            checkout_arcadia_from_url=self.Parameters.checkout_arcadia_from_url,
            arcadia_patch=self.Parameters.arcadia_patch,
            arts=ya_make_artifact.artifact,
            result_rt=resource_type.name,
            result_single_file=True,
            result_ttl=str(DEFAULT_RESOURCE_TTL),
            targets=ya_make_artifact.target,
            use_aapi_fuse=True,
            aapi_fallback=True,
            build_system=constants.SEMI_DISTBUILD_BUILD_SYSTEM,
            strip_binaries=True,
            ya_timeout=self.Parameters.kill_timeout,
            sandbox_tags=ctc.Tag.LINUX_XENIAL,
        )

        subtask.enqueue()

        return subtask

    @staticmethod
    def _fetch_trunk_specs():
        LOGGER.info('Fetch specs from trunk')

        specs = {}
        with arcadiasdk.mount_arc_path(ARCADIA_TRUNK_URL, fallback=True) as arcadia_dir:
            for stage in Stages.get_all_stages():
                filename = '{}.json'.format(stage)
                filepath = os.path.join(arcadia_dir, ARCADIA_SPECS_DIR, filename)
                with open(filepath) as source:
                    spec = json.load(source)

                LOGGER.debug('%s spec: %s', stage, spec)
                specs[stage] = spec

        return specs

    def _get_stage_settings(self):
        params = self.Parameters

        return {
            Stages.MAIN: StageSettings(
                need_update=params.update_main_stage,
                update_non_globs=params.update_main_non_globs,
                cleanup_globs=params.cleanup_main_globs,
                globs=params.main_globs,
            ),
            Stages.HAMSTER: StageSettings(
                need_update=params.update_hamster_stage,
                update_non_globs=params.update_hamster_non_globs,
                cleanup_globs=params.cleanup_hamster_globs,
                globs=params.hamster_globs,
            ),
            Stages.PRESTABLE: StageSettings(
                need_update=params.update_prestable_stage,
                update_non_globs=params.update_prestable_non_globs,
                cleanup_globs=params.cleanup_prestable_globs,
                globs=params.prestable_globs,
            ),
            Stages.PRESTABLE_AB: StageSettings(
                need_update=params.update_prestable_ab_stage,
                update_non_globs=params.update_prestable_ab_non_globs,
                cleanup_globs=params.cleanup_prestable_ab_globs,
                globs=params.prestable_ab_globs,
            ),
            Stages.PRESTABLE_ZERO: StageSettings(
                need_update=params.update_prestable_zero_stage,
                update_non_globs=params.update_prestable_zero_non_globs,
                cleanup_globs=params.cleanup_prestable_zero_globs,
                globs=params.prestable_zero_globs,
            ),
            Stages.PRESTABLE_HEAVY: StageSettings(
                need_update=params.update_prestable_heavy_stage,
                update_non_globs=params.update_prestable_heavy_non_globs,
                cleanup_globs=params.cleanup_prestable_heavy_globs,
                globs=params.prestable_heavy_globs,
            ),
        }

    def _make_resource_url(self, artifact):
        return 'sbr:{}'.format(self.Context.resource_ids[artifact])

    def _update_resources_spec(self, spec, settings):
        params = self.Parameters

        # fill missing spec fields
        spec.setdefault(KEY_CUSTOM_RESOURCES, {})
        spec.setdefault(KEY_DEFAULT_RESOURCES, {})
        for artifact in params.deployable_artifacts:
            spec[KEY_CUSTOM_RESOURCES].setdefault(artifact, {})

        if not settings.need_update:
            return

        for artifact in params.deployable_artifacts:
            resource_url = self._make_resource_url(artifact)

            if settings.update_non_globs:
                spec[KEY_DEFAULT_RESOURCES][artifact] = resource_url

            globs_mapping = spec[KEY_CUSTOM_RESOURCES][artifact]

            if settings.cleanup_globs:
                globs_mapping.clear()

            for glob in settings.globs:
                globs_mapping[glob] = resource_url

    @staticmethod
    def _fill_resource_data(resource, content, need_eol=False):
        LOGGER.debug('Fill data for resource %s', resource.id)

        resource_data = sdk2.ResourceData(resource)

        with open(str(resource_data.path), 'w') as sink:
            sink.write(content)
            if need_eol:
                sink.write('\n')

        resource_data.ready()

    def _make_spec_resources(self):
        LOGGER.info('Prepare specs for stages')

        stage_specs = self._fetch_trunk_specs()
        stage_settings = self._get_stage_settings()

        for stage, spec in stage_specs.items():
            LOGGER.info('Prepare spec resource for %s stage', stage)

            self._update_resources_spec(spec, stage_settings[stage])

            data = json.dumps(spec, indent=4, sort_keys=True, separators=(',', ': '))

            resource = US_RT_RESOURCES_SPEC(
                self,
                'Stage resources spec (release task: {})'.format(self.id),
                '{}.json'.format(stage),
                # attributes
                stage=stage,
            )

            self._fill_resource_data(resource, data, need_eol=True)

            LOGGER.info('Created spec resource: %s', resource.id)

            self.Context.spec_resource_ids[stage] = resource.id
            self.Context.save()

        info_lines = ['Specs:']
        for stage in sorted(stage_specs):
            link = link_builder.resource_link(self.Context.spec_resource_ids[stage])
            info_lines.append('  - {}: {}'.format(stage, link))

        self.set_info('\n'.join(info_lines), do_escape=False)

    def _make_stage_config_resources(self):
        LOGGER.info('Prepare YDeploy configs for stages')

        tool_resource = sdk2.Resource[self.Context.resource_ids[ArtifactTypes.DEPLOY_TOOL]]
        tool_data = sdk2.ResourceData(tool_resource)

        dctl_token = self.Parameters.dctl_token.data()[self.Parameters.dctl_token.default_key]
        vault_token = self.Parameters.vault_token.data()[self.Parameters.vault_token.default_key]

        old_conf_path = os.path.abspath('_old_conf.yaml')
        new_conf_path = os.path.abspath('_new_conf.yaml')
        diff_conf_path = os.path.abspath('_conf.diff')

        with arcadiasdk.mount_arc_path(self.Parameters.checkout_arcadia_from_url, fallback=True) as arcadia_dir:
            configure_env = os.environ.copy()
            configure_env.update(
                DCTL_YP_TOKEN=dctl_token,
                YP_TOKEN=dctl_token,
                VAULT_TOKEN=vault_token,
                US_RT_YA_TOOL=os.path.join(str(arcadia_dir), 'ya'),
            )

            for stage in Stages.get_all_stages():
                spec_resource = sdk2.Resource[self.Context.spec_resource_ids[stage]]
                spec_data = sdk2.ResourceData(spec_resource)

                with sdk2.helpers.ProcessLog(self, logger='configure-{}'.format(stage)) as pl:
                    sdk2.helpers.subprocess.check_call(
                        [
                            str(tool_data.path), stage, 'configure',
                            '-rs', str(spec_data.path),
                            '-oc', old_conf_path,
                            '-nc', new_conf_path,
                            '--enable-vault',
                        ],
                        env=configure_env,
                        stdout=pl.stdout,
                        stderr=pl.stderr,
                    )

                with sdk2.helpers.ProcessLog(self, logger='diff-{}'.format(stage)) as pl:
                    with open(diff_conf_path, 'w') as diff_file:
                        sdk2.helpers.subprocess.call(
                            [
                                'diff', '-u',
                                old_conf_path,
                                new_conf_path,
                            ],
                            env=configure_env,
                            stdout=diff_file,
                            stderr=pl.stderr,
                        )

                with open(old_conf_path) as source:
                    old_data = source.read()

                with open(new_conf_path) as source:
                    new_data = source.read()

                with open(diff_conf_path) as source:
                    diff_data = source.read()

                if old_data != new_data:
                    self._make_stage_config_resource(stage, old_data, ConfigTypes.OLD)
                    self._make_stage_config_resource(stage, new_data, ConfigTypes.NEW)
                    self._make_stage_config_resource(stage, diff_data, ConfigTypes.DIFF)
                else:
                    LOGGER.info('Skip %s stage config resource creation, no diff', stage)

                shutil.rmtree(old_conf_path, ignore_errors=True)
                shutil.rmtree(new_conf_path, ignore_errors=True)

        info_lines = ['Changed stage configs:']

        old_config_resource_ids = self.Context.config_resource_ids.setdefault(
            ConfigTypes.OLD, {}
        )

        new_config_resource_ids = self.Context.config_resource_ids.setdefault(
            ConfigTypes.NEW, {}
        )

        diff_config_resource_ids = self.Context.config_resource_ids.setdefault(
            ConfigTypes.DIFF, {}
        )

        for stage, resource_id in new_config_resource_ids.items():
            old_link = link_builder.resource_link(old_config_resource_ids[stage])
            new_link = link_builder.resource_link(resource_id)
            diff_link = link_builder.resource_link(diff_config_resource_ids[stage])

            info_lines.append(
                '  - {}: {} => {} (diff: {})'.format(stage, old_link, new_link, diff_link)
            )

        self.set_info('\n'.join(info_lines), do_escape=False)

    def _make_stage_config_resource(self, stage, data, config_type):
        resource_type = US_RT_STAGE_CONFIG if config_type == ConfigTypes.NEW else US_RT_MISC

        LOGGER.info('Prepare resource for config type %s of %s stage', config_type, stage)

        storage = self.Context.config_resource_ids.setdefault(config_type, {})

        if stage in storage:
            raise errors.TaskFailure(
                'Attempt to rewrite {} config for stage {}'.format(config_type, stage)
            )

        if config_type == ConfigTypes.DIFF:
            filename = '{}-conf.diff'.format(stage)
            description = 'YDeploy diff stage config (release task: {})'
        else:
            filename = '{}-{}-conf.yaml'.format(config_type, stage)
            description = 'YDeploy stage config (release task: {})'

        resource = resource_type(
            self,
            description.format(self.id),
            filename,
            # attributes
            stage=stage,
        )

        self._fill_resource_data(resource, data)

        LOGGER.info('Created stage config resource: %s', resource.id)

        storage[stage] = resource.id

        self.Context.save()
